/*
 * Decompiled with CFR 0.152.
 */
package elki.index.tree.betula;

import elki.data.NumberVector;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.relation.Relation;
import elki.index.tree.betula.CFNode;
import elki.index.tree.betula.distance.BIRCHRadiusDistance;
import elki.index.tree.betula.distance.BIRCHVarianceIncreaseDistance;
import elki.index.tree.betula.distance.CFDistance;
import elki.index.tree.betula.distance.RadiusDistance;
import elki.index.tree.betula.distance.VarianceIncreaseDistance;
import elki.index.tree.betula.features.AsClusterFeature;
import elki.index.tree.betula.features.BIRCHCF;
import elki.index.tree.betula.features.ClusterFeature;
import elki.index.tree.betula.features.VIIFeature;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.statistics.DoubleStatistic;
import elki.logging.statistics.Duration;
import elki.logging.statistics.LongStatistic;
import elki.logging.statistics.Statistic;
import elki.math.MathUtil;
import elki.utilities.datastructures.arrays.DoubleIntegerArrayQuickSort;
import elki.utilities.datastructures.iterator.Iter;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.References;
import elki.utilities.io.FormatUtil;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.GreaterEqualConstraint;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.EnumParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import it.unimi.dsi.fastutil.objects.Reference2ObjectOpenHashMap;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Map;

@References(value={@Reference(authors="T. Zhang, R. Ramakrishnan, M. Livny", title="BIRCH: An Efficient Data Clustering Method for Very Large Databases", booktitle="Proc. 1996 ACM SIGMOD International Conference on Management of Data", url="https://doi.org/10.1145/233269.233324", bibkey="DBLP:conf/sigmod/ZhangRL96"), @Reference(authors="T. Zhang, R. Ramakrishnan, M. Livny", title="BIRCH: A New Data Clustering Algorithm and Its Applications", booktitle="Data Min. Knowl. Discovery", url="https://doi.org/10.1023/A:1009783824328", bibkey="DBLP:journals/datamine/ZhangRL97"), @Reference(authors="Andreas Lang and Erich Schubert", title="BETULA: Numerically Stable CF-Trees for BIRCH Clustering", booktitle="Int. Conf on Similarity Search and Applications", url="https://doi.org/10.1007/978-3-030-60936-8_22", bibkey="DBLP:conf/sisap/LangS20"), @Reference(authors="Andreas Lang and Erich Schubert", title="BETULA: Fast Clustering of Large Data with Improved BIRCH CF-Trees", booktitle="Information Systems", url="https://doi.org/10.1016/j.is.2021.101918", bibkey="DBLP:journals/is/LangS22")})
public class CFTree<L extends ClusterFeature> {
    public static final Logging LOG = Logging.getLogger(CFTree.class);
    ClusterFeature.Factory<L> factory;
    CFDistance dist;
    CFDistance abs;
    Threshold tCriterium;
    double thresholdsq;
    int capacity;
    CFNode<L> root = null;
    int leaves;
    int maxleaves;
    int rebuildstat;
    long diststat = 0L;
    long absstat = 0L;
    Map<ClusterFeature, ArrayModifiableDBIDs> idmap;

    public CFTree(ClusterFeature.Factory<L> factory, CFDistance dist, CFDistance abs, double threshold, int capacity, Threshold tCriterium, int maxleaves, boolean storeIds) {
        this.factory = factory;
        this.dist = dist;
        this.abs = abs;
        this.thresholdsq = threshold * threshold;
        this.capacity = capacity;
        this.tCriterium = tCriterium;
        this.maxleaves = maxleaves;
        this.idmap = storeIds ? new Reference2ObjectOpenHashMap(maxleaves) : null;
    }

    public void insert(NumberVector nv, DBIDRef dbid) {
        if (this.root == null) {
            int dim = nv.getDimensionality();
            L leaf = this.factory.make(dim);
            if (this.idmap != null) {
                ArrayModifiableDBIDs list = DBIDUtil.newArray();
                list.add(dbid);
                this.idmap.put((ClusterFeature)leaf, list);
            }
            leaf.addToStatistics(nv);
            this.root = new CFNode<L>(this.factory.make(dim), this.capacity);
            this.root.add(0, (AsClusterFeature)leaf);
            ++this.leaves;
            return;
        }
        CFNode<L> other = this.insert(this.root, nv, dbid);
        if (other != null) {
            int dim = other.getCF().getDimensionality();
            CFNode<L> newnode = new CFNode<L>(this.factory.make(dim), this.capacity);
            newnode.add(0, this.root);
            newnode.add(1, other);
            this.root = newnode;
        }
        if (this.leaves > this.maxleaves) {
            if (LOG.isVerbose()) {
                LOG.verbose((CharSequence)"Compacting CF-tree.");
            }
            ++this.rebuildstat;
            this.rebuildTree();
        }
    }

    protected void rebuildTree() {
        int dim = this.root.getCF().getDimensionality();
        int oldLeaves = this.leaves;
        ArrayList cfs = new ArrayList(this.leaves);
        double[] thresholds = new double[this.leaves];
        this.estimateThreshold(this.root, cfs, thresholds);
        int[] order = MathUtil.sequence((int)0, (int)this.leaves);
        double t = 0.0;
        if (this.tCriterium == Threshold.MEAN) {
            int n = 0;
            for (int i = 0; i < this.leaves; ++i) {
                if (!(thresholds[i] < Double.POSITIVE_INFINITY)) continue;
                t += Math.sqrt(thresholds[i]);
                ++n;
            }
            t /= (double)n;
            t *= t;
        } else if (this.tCriterium == Threshold.MEDIAN) {
            DoubleIntegerArrayQuickSort.sort((double[])thresholds, (int[])order, (int)this.leaves);
            int median = cfs.size() >>> 1;
            t = thresholds[median];
            while (t == Double.POSITIVE_INFINITY && median > 0) {
                t = thresholds[--median];
            }
        } else {
            throw new IllegalStateException("Unknown threshold heuristic.");
        }
        this.thresholdsq = t > this.thresholdsq ? t : this.thresholdsq;
        LOG.debug((CharSequence)("New squared threshold: " + this.thresholdsq));
        this.leaves = 0;
        this.root = new CFNode<L>(this.factory.make(dim), this.capacity);
        this.root.add(0, (AsClusterFeature)cfs.get(order[cfs.size() - 1]));
        ++this.leaves;
        for (int i = cfs.size() - 2; i >= 0; --i) {
            CFNode<L> other = this.insert(this.root, (AsClusterFeature)cfs.get(order[i]));
            if (other == null) continue;
            CFNode<L> newnode = new CFNode<L>(this.factory.make(dim), this.capacity);
            newnode.add(0, this.root);
            newnode.add(1, other);
            this.root = newnode;
        }
        if (this.leaves > oldLeaves) {
            throw new IllegalStateException("Could not reduce the number of leaves when compacting tree");
        }
    }

    private void estimateThreshold(CFNode<L> current, ArrayList<L> cfs, double[] thresholds) {
        AsClusterFeature ci;
        int offset = cfs.size();
        if (current.getChild(0) instanceof CFNode) {
            for (int i = 0; i < this.capacity && current.getChild(i) != null; ++i) {
                this.estimateThreshold((CFNode)current.getChild(i), cfs, thresholds);
            }
            return;
        }
        assert (current.getChild(0) instanceof ClusterFeature) : "Node is neither child nor inner?";
        if (current.getChild(1) == null) {
            thresholds[offset++] = Double.POSITIVE_INFINITY;
            cfs.add((ClusterFeature)current.getChild(0));
            return;
        }
        double[] best = new double[this.capacity];
        Arrays.fill(best, Double.POSITIVE_INFINITY);
        int[] besti = new int[this.capacity];
        for (int i = 0; i < this.capacity && (ci = current.getChild(i)) != null; ++i) {
            double bi = best[i];
            int bestii = besti[i];
            for (int j = i + 1; j < this.capacity && current.getChild(j) != null; ++j) {
                double d = this.sqdistance(ci.getCF(), current.getChild(j).getCF());
                if (d < bi) {
                    bi = d;
                    bestii = j;
                }
                if (!(d < best[j])) continue;
                best[j] = d;
                besti[j] = i;
            }
            thresholds[offset++] = this.sqabsorption(ci.getCF(), current.getChild(bestii).getCF());
            cfs.add((ClusterFeature)ci);
        }
    }

    private CFNode<L> insert(CFNode<L> node, NumberVector nv, DBIDRef dbid) {
        AsClusterFeature cf;
        assert (node.getChild(0) != null) : "Unexpected empty node!";
        AsClusterFeature best = node.getChild(0);
        double bestd = this.sqdistance(nv, best.getCF());
        for (int i = 1; i < this.capacity && (cf = node.getChild(i)) != null; ++i) {
            double d2 = this.sqdistance(nv, cf.getCF());
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        if (!(best instanceof CFNode)) {
            if (this.sqabsorption(nv, best.getCF()) <= this.thresholdsq) {
                best.getCF().addToStatistics(nv);
                if (this.idmap != null) {
                    this.idmap.get((ClusterFeature)best).add(dbid);
                }
                node.getCF().addToStatistics(nv);
                return null;
            }
            L bestl = this.factory.make(nv.getDimensionality());
            if (this.idmap != null) {
                ArrayModifiableDBIDs list = DBIDUtil.newArray();
                list.add(dbid);
                this.idmap.put((ClusterFeature)bestl, list);
            }
            bestl.addToStatistics(nv);
            ++this.leaves;
            return node.add((AsClusterFeature)bestl) ? null : this.split(node, (AsClusterFeature)bestl);
        }
        assert (best instanceof CFNode) : "Node is neither child nor inner?";
        CFNode<L> newchild = this.insert((CFNode)best, nv, dbid);
        if (newchild == null) {
            node.getCF().addToStatistics(nv);
            return null;
        }
        if (node.setChild(newchild)) {
            node.getCF().addToStatistics(nv);
            return null;
        }
        return this.split(node, newchild);
    }

    public L findLeaf(NumberVector nv) {
        if (this.root == null) {
            throw new IllegalStateException("CFTree not yet built.");
        }
        return this.findLeaf(this.root, nv);
    }

    private L findLeaf(CFNode<L> node, NumberVector nv) {
        AsClusterFeature cf;
        assert (node.getChild(0) != null) : "Unexpected empty node!";
        AsClusterFeature best = node.getChild(0);
        double bestd = this.sqdistance(nv, best.getCF());
        for (int i = 1; i < this.capacity && (cf = node.getChild(i)) != null; ++i) {
            double d2 = this.sqdistance(nv, cf.getCF());
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        return (L)(best instanceof CFNode ? this.findLeaf((CFNode)best, nv) : (ClusterFeature)best);
    }

    private CFNode<L> split(CFNode<L> node, AsClusterFeature newchild) {
        assert (node.getChild(this.capacity - 1) != null) : "Node to split is not empty!";
        CFNode<L> newn = new CFNode<L>(this.factory.make(node.getCF().getDimensionality()), this.capacity);
        int size = this.capacity + 1;
        int m1 = -1;
        int m2 = -1;
        double maxd = Double.NEGATIVE_INFINITY;
        double[][] dists = new double[size][size];
        for (int i = 0; i < this.capacity; ++i) {
            ClusterFeature ci = node.getChild(i).getCF();
            for (int j = i + 1; j < this.capacity; ++j) {
                double d = this.sqdistance(ci, node.getChild(j).getCF());
                dists[j][i] = d;
                dists[i][j] = d;
                double d2 = d;
                if (!(d2 > maxd)) continue;
                maxd = d2;
                m1 = i;
                m2 = j;
            }
            double d = this.sqdistance(ci, newchild.getCF());
            dists[this.capacity][i] = d;
            dists[i][this.capacity] = d;
            double d3 = d;
            if (!(d3 > maxd)) continue;
            maxd = d3;
            m1 = i;
            m2 = this.capacity;
        }
        node.getCF().resetStatistics();
        newn.getCF().resetStatistics();
        int si = 0;
        int sj = 0;
        double[] d1s = dists[m1];
        double[] d2s = dists[m2];
        for (int i = 0; i < this.capacity; ++i) {
            double d1 = d1s[i];
            double d2 = d2s[i];
            if (i == m1 || i != m2 && (d1 < d2 || d1 == d2 && si <= sj)) {
                node.add(si++, node.getChild(i));
                continue;
            }
            newn.add(sj++, node.getChild(i));
        }
        double d1 = d1s[this.capacity];
        double d2 = d2s[this.capacity];
        if (this.capacity != m2 && (d1 < d2 || d1 == d2 && si <= sj)) {
            node.add(si++, newchild);
        } else {
            newn.add(sj++, newchild);
        }
        for (int j = si; j < this.capacity; ++j) {
            node.setChild(j, null);
        }
        return newn;
    }

    private CFNode<L> insert(CFNode<L> node, AsClusterFeature nleaf) {
        AsClusterFeature cf;
        assert (node.getChild(0) != null) : "Unexpected empty node!";
        AsClusterFeature best = node.getChild(0);
        double bestd = this.sqdistance(best.getCF(), nleaf.getCF());
        for (int i = 1; i < this.capacity && (cf = node.getChild(i)) != null; ++i) {
            double d2 = this.sqdistance(cf.getCF(), nleaf.getCF());
            if (!(d2 < bestd)) continue;
            best = cf;
            bestd = d2;
        }
        assert (best != nleaf);
        if (!(best instanceof CFNode)) {
            if (this.sqabsorption(best.getCF(), nleaf.getCF()) <= this.thresholdsq) {
                best.getCF().addToStatistics(nleaf.getCF());
                if (this.idmap != null) {
                    this.idmap.get(best).addDBIDs((DBIDs)this.idmap.remove(nleaf));
                }
                node.getCF().addToStatistics(nleaf.getCF());
                return null;
            }
            ++this.leaves;
            return node.add(nleaf) ? null : this.split(node, nleaf);
        }
        assert (best instanceof CFNode) : "Node is neither child nor inner?";
        CFNode<L> newchild = this.insert((CFNode)best, nleaf);
        if (newchild == null) {
            node.getCF().addToStatistics(nleaf.getCF());
            return null;
        }
        if (node.setChild(newchild)) {
            node.getCF().addToStatistics(nleaf.getCF());
            return null;
        }
        return this.split(node, newchild);
    }

    public LeafIterator<L> leafIterator() {
        return new LeafIterator(this.root);
    }

    public ArrayList<L> getLeaves() {
        ArrayList<L> cfs = new ArrayList<L>(this.leaves);
        LeafIterator<L> iter = this.leafIterator();
        while (iter.valid()) {
            cfs.add(iter.get());
            iter.advance();
        }
        return cfs;
    }

    protected static StringBuilder printDebug(StringBuilder buf, ClusterFeature n, int d) {
        FormatUtil.appendSpace((StringBuilder)buf, (int)d).append(n.getWeight());
        for (int i = 0; i < n.getDimensionality(); ++i) {
            buf.append(' ').append(n.centroid(i));
        }
        buf.append(" - ").append(n.getWeight()).append('\n');
        if (n instanceof CFNode) {
            CFNode node = (CFNode)((Object)n);
            for (int i = 0; i < node.capacity(); ++i) {
                AsClusterFeature c = node.getChild(i);
                if (c == null) continue;
                CFTree.printDebug(buf, c.getCF(), d + 1);
            }
        }
        return buf;
    }

    private double sqdistance(ClusterFeature cf1, ClusterFeature cf2) {
        ++this.diststat;
        return this.dist.squaredDistance(cf1, cf2);
    }

    private double sqdistance(NumberVector nv, ClusterFeature cf) {
        ++this.diststat;
        return this.dist.squaredDistance(nv, cf);
    }

    private double sqabsorption(ClusterFeature cf1, ClusterFeature cf2) {
        ++this.absstat;
        return this.abs.squaredDistance(cf1.getCF(), cf2.getCF());
    }

    private double sqabsorption(NumberVector nv, ClusterFeature cf) {
        ++this.absstat;
        return this.abs.squaredDistance(nv, cf);
    }

    public int numLeaves() {
        return this.leaves;
    }

    public CFNode<L> getRoot() {
        return this.root;
    }

    public int getCapacity() {
        return this.capacity;
    }

    public DBIDs getDBIDs(ClusterFeature cf) {
        return (DBIDs)this.idmap.get(cf);
    }

    public static class Factory<L extends ClusterFeature> {
        ClusterFeature.Factory<L> factory;
        CFDistance dist;
        CFDistance abs;
        double threshold;
        int branchingFactor;
        double maxleaves;
        Threshold tCriterium;

        public Factory(ClusterFeature.Factory<L> factory, CFDistance dist, CFDistance abs, double threshold, int branchingFactor, double maxleaves, Threshold tCriterium) {
            this.factory = factory;
            this.dist = dist;
            this.abs = abs;
            this.threshold = threshold;
            this.branchingFactor = branchingFactor;
            this.maxleaves = maxleaves;
            this.tCriterium = tCriterium;
        }

        public CFTree<L> newTree(DBIDs ids, Relation<? extends NumberVector> relation, boolean storeIds) {
            String prefix = CFTree.class.getName();
            Duration buildtime = LOG.newDuration(prefix + ".buildtime").begin();
            int max = (int)(this.maxleaves <= 1.0 ? this.maxleaves * (double)ids.size() : this.maxleaves);
            CFTree<L> tree = new CFTree<L>(this.factory, this.dist, this.abs, this.threshold, this.branchingFactor, this.tCriterium, max, storeIds);
            FiniteProgress prog = LOG.isVerbose() ? new FiniteProgress("Building tree", relation.size(), LOG) : null;
            DBIDIter it = relation.iterDBIDs();
            while (it.valid()) {
                tree.insert((NumberVector)relation.get((DBIDRef)it), (DBIDRef)it);
                LOG.incrementProcessed((AbstractProgress)prog);
                it.advance();
            }
            LOG.statistics((Statistic)buildtime.end());
            LOG.statistics((Statistic)new LongStatistic(prefix + ".rebuilds", (long)tree.rebuildstat));
            LOG.statistics((Statistic)new LongStatistic(prefix + ".leaves", (long)tree.leaves));
            LOG.statistics((Statistic)new LongStatistic(prefix + ".distance-calculations", tree.diststat));
            LOG.statistics((Statistic)new LongStatistic(prefix + ".absorption-calculations", tree.absstat));
            LOG.statistics((Statistic)new DoubleStatistic(prefix + ".threshold", Math.sqrt(tree.thresholdsq)));
            LOG.ensureCompleted(prog);
            return tree;
        }

        public static class Par<L extends ClusterFeature>
        implements Parameterizer {
            public static final OptionID FEATURES_ID = new OptionID("cftree.features", "Cluster features to use.");
            public static final OptionID DISTANCE_ID = new OptionID("cftree.distance", "Distance function to use for node assignment.");
            public static final OptionID ABSORPTION_ID = new OptionID("cftree.absorption", "Absorption criterion to use.");
            public static final OptionID THRESHOLD_ID = new OptionID("cftree.threshold", "Threshold for adding points to existing nodes in the CF-Tree.");
            public static final OptionID SPLIT_ID = new OptionID("cftree.threshold.heuristic", "Threshold heuristic to use (mean or median).");
            public static final OptionID BRANCHING_ID = new OptionID("cftree.branching", "Maximum branching factor of the CF-Tree");
            public static final OptionID MAXLEAVES_ID = new OptionID("cftree.maxleaves", "Maximum number of leaves (if less than 1, the values is assumed to be relative)");
            ClusterFeature.Factory<L> factory;
            CFDistance dist;
            CFDistance abs;
            double threshold = 0.0;
            int branchingFactor;
            double maxleaves;
            Threshold tCriterium;

            public void configure(Parameterization config) {
                new ObjectParameter(FEATURES_ID, ClusterFeature.Factory.class, VIIFeature.Factory.class).grab(config, x -> {
                    this.factory = x;
                });
                boolean isbirch = this.factory != null && this.factory.getClass() == BIRCHCF.Factory.class;
                new ObjectParameter(DISTANCE_ID, CFDistance.class, isbirch ? BIRCHVarianceIncreaseDistance.class : VarianceIncreaseDistance.class).grab(config, x -> {
                    this.dist = x;
                });
                new ObjectParameter(ABSORPTION_ID, CFDistance.class, isbirch ? BIRCHRadiusDistance.class : RadiusDistance.class).grab(config, x -> {
                    this.abs = x;
                });
                new EnumParameter(SPLIT_ID, Threshold.class, (Enum)Threshold.MEAN).grab(config, x -> {
                    this.tCriterium = x;
                });
                ((DoubleParameter)((DoubleParameter)new DoubleParameter(THRESHOLD_ID).setOptional(true)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_DOUBLE)).grab(config, x -> {
                    this.threshold = x;
                });
                ((IntParameter)((IntParameter)new IntParameter(BRANCHING_ID).addConstraint((ParameterConstraint)new GreaterEqualConstraint(2))).setDefaultValue((Object)64)).grab(config, x -> {
                    this.branchingFactor = x;
                });
                ((DoubleParameter)((DoubleParameter)new DoubleParameter(MAXLEAVES_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).setDefaultValue((Object)0.05)).grab(config, x -> {
                    this.maxleaves = x;
                });
            }

            public Factory<L> make() {
                return new Factory<L>(this.factory, this.dist, this.abs, this.threshold, this.branchingFactor, this.maxleaves, this.tCriterium);
            }
        }
    }

    public static class LeafIterator<L extends ClusterFeature>
    implements Iter {
        private ArrayList<Object> queue = new ArrayList();
        private L current;

        private LeafIterator(CFNode<L> root) {
            this.queue.add(root);
            this.advance();
        }

        public boolean valid() {
            return this.current != null;
        }

        public L get() {
            return this.current;
        }

        public Iter advance() {
            this.current = null;
            while (!this.queue.isEmpty()) {
                AsClusterFeature c;
                Object f = this.queue.remove(this.queue.size() - 1);
                if (!(f instanceof CFNode)) {
                    this.current = (ClusterFeature)f;
                    break;
                }
                CFNode node = (CFNode)f;
                for (int i = 0; i < node.capacity() && (c = node.getChild(i)) != null; ++i) {
                    this.queue.add(c);
                }
            }
            return this;
        }
    }

    public static enum Threshold {
        MEAN,
        MEDIAN;

    }
}

