/*
 * Decompiled with CFR 0.152.
 */
package elki.datasource.filter.transform;

import elki.data.NumberVector;
import elki.data.type.SimpleTypeInformation;
import elki.data.type.TypeUtil;
import elki.datasource.filter.AbstractVectorConversionFilter;
import elki.logging.Logging;
import elki.math.MeanVarianceMinMax;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.ParameterException;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.WrongParameterValueException;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleListParameter;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.EnumParameter;
import elki.utilities.optionhandling.parameters.LongParameter;
import elki.utilities.optionhandling.parameters.Parameter;
import java.util.Random;

@Title(value="Data Perturbation for Outlier Detection Ensembles")
@Description(value="A filter to perturb a datasset on read by an additive noise component, implemented for use in an outlier ensemble (this reference).")
@Reference(authors="A. Zimek, R. J. G. B. Campello, J. Sander", title="Data Perturbation for Outlier Detection Ensembles", booktitle="Proc. 26th International Conference on Scientific and Statistical Database Management (SSDBM), Aalborg, Denmark, 2014", url="https://doi.org/10.1145/2618243.2618257", bibkey="DBLP:conf/ssdbm/ZimekCS14")
public class PerturbationFilter<V extends NumberVector>
extends AbstractVectorConversionFilter<V, V> {
    private static final Logging LOG = Logging.getLogger(PerturbationFilter.class);
    private ScalingReference scalingreference;
    private NoiseDistribution noisedistribution;
    private final Random RANDOM;
    private double percentage;
    private MeanVarianceMinMax[] mvs = null;
    private double[] scalingreferencevalues = new double[0];
    private Random[] randomPerAttribute = null;
    private double[] maxima;
    private double[] minima;
    private int dimensionality = 0;

    public PerturbationFilter(Long seed, double percentage, ScalingReference scalingreference, double[] minima, double[] maxima, NoiseDistribution noisedistribution) {
        this.percentage = percentage;
        this.scalingreference = scalingreference;
        this.minima = minima;
        this.maxima = maxima;
        this.noisedistribution = noisedistribution;
        this.RANDOM = seed == null ? new Random() : new Random(seed);
    }

    @Override
    protected boolean prepareStart(SimpleTypeInformation<V> in) {
        if (this.scalingreference == ScalingReference.MINMAX && this.minima.length != 0 && this.maxima.length != 0) {
            this.dimensionality = this.minima.length;
            this.scalingreferencevalues = new double[this.dimensionality];
            this.randomPerAttribute = new Random[this.dimensionality];
            for (int d = 0; d < this.dimensionality; ++d) {
                this.scalingreferencevalues[d] = (this.maxima[d] - this.minima[d]) * this.percentage;
                if (this.scalingreferencevalues[d] == 0.0 || Double.isNaN(this.scalingreferencevalues[d])) {
                    this.scalingreferencevalues[d] = this.percentage;
                }
                this.randomPerAttribute[d] = new Random(this.RANDOM.nextLong());
            }
            return false;
        }
        if (this.scalingreference == ScalingReference.UNITCUBE) {
            return false;
        }
        return this.scalingreferencevalues.length == 0;
    }

    @Override
    protected void prepareProcessInstance(V featureVector) {
        if (this.mvs == null) {
            this.dimensionality = featureVector.getDimensionality();
            this.mvs = MeanVarianceMinMax.newArray((int)this.dimensionality);
        }
        for (int d = 0; d < featureVector.getDimensionality(); ++d) {
            this.mvs[d].put(featureVector.doubleValue(d));
        }
    }

    @Override
    protected void prepareComplete() {
        StringBuilder buf = LOG.isDebuggingFine() ? new StringBuilder(1000) : null;
        this.scalingreferencevalues = new double[this.dimensionality];
        this.randomPerAttribute = new Random[this.dimensionality];
        if (this.scalingreference == ScalingReference.STDDEV) {
            if (buf != null) {
                buf.append("Standard deviation per attribute: ");
            }
            for (int d = 0; d < this.dimensionality; ++d) {
                this.scalingreferencevalues[d] = this.mvs[d].getSampleStddev() * this.percentage;
                if (this.scalingreferencevalues[d] == 0.0 || Double.isNaN(this.scalingreferencevalues[d])) {
                    this.scalingreferencevalues[d] = this.percentage;
                }
                this.randomPerAttribute[d] = new Random(this.RANDOM.nextLong());
                if (buf == null) continue;
                buf.append(' ').append(d).append(": ").append(this.scalingreferencevalues[d] / this.percentage);
            }
        } else if (this.scalingreference == ScalingReference.MINMAX && this.minima.length == 0 && this.maxima.length == 0) {
            if (buf != null) {
                buf.append("extension per attribute: ");
            }
            for (int d = 0; d < this.dimensionality; ++d) {
                this.scalingreferencevalues[d] = (this.mvs[d].getMax() - this.mvs[d].getMin()) * this.percentage;
                if (this.scalingreferencevalues[d] == 0.0 || Double.isNaN(this.scalingreferencevalues[d])) {
                    this.scalingreferencevalues[d] = this.percentage;
                }
                this.randomPerAttribute[d] = new Random(this.RANDOM.nextLong());
                if (buf == null) continue;
                buf.append(' ').append(d).append(": ").append(this.scalingreferencevalues[d] / this.percentage);
            }
        }
        this.mvs = null;
        if (buf != null) {
            LOG.debugFine((CharSequence)buf.toString());
        }
    }

    @Override
    protected SimpleTypeInformation<? super V> getInputTypeRestriction() {
        return TypeUtil.NUMBER_VECTOR_FIELD;
    }

    @Override
    protected V filterSingleObject(V featureVector) {
        if (this.scalingreference == ScalingReference.UNITCUBE && this.dimensionality == 0) {
            this.dimensionality = featureVector.getDimensionality();
            this.scalingreferencevalues = new double[this.dimensionality];
            this.randomPerAttribute = new Random[this.dimensionality];
            for (int d = 0; d < this.dimensionality; ++d) {
                this.scalingreferencevalues[d] = this.percentage;
                this.randomPerAttribute[d] = new Random(this.RANDOM.nextLong());
            }
        }
        if (this.scalingreferencevalues.length != featureVector.getDimensionality()) {
            throw new IllegalArgumentException("FeatureVectors and given Minima/Maxima differ in length.");
        }
        double[] values = new double[featureVector.getDimensionality()];
        for (int d = 0; d < featureVector.getDimensionality(); ++d) {
            if (this.noisedistribution.equals((Object)NoiseDistribution.GAUSSIAN)) {
                values[d] = featureVector.doubleValue(d) + this.randomPerAttribute[d].nextGaussian() * this.scalingreferencevalues[d];
                continue;
            }
            if (!this.noisedistribution.equals((Object)NoiseDistribution.UNIFORM)) continue;
            values[d] = featureVector.doubleValue(d) + this.randomPerAttribute[d].nextDouble() * this.scalingreferencevalues[d];
        }
        return (V)this.factory.newNumberVector(values);
    }

    @Override
    protected SimpleTypeInformation<? super V> convertedType(SimpleTypeInformation<V> in) {
        this.initializeOutputType(in);
        return in;
    }

    @Override
    protected Logging getLogger() {
        return LOG;
    }

    public static class Par<V extends NumberVector>
    implements Parameterizer {
        public static final OptionID MINIMA_ID = new OptionID("perturbationfilter.min", "Only used, if " + (Object)((Object)ScalingReference.MINMAX) + " is set as scaling reference: a comma separated concatenation of the minimum values in each dimension assumed as a reference. If no value is specified, the minimum value of the attribute range in this dimension will be taken.");
        public static final OptionID MAXIMA_ID = new OptionID("perturbationfilter.max", "Only used, if " + (Object)((Object)ScalingReference.MINMAX) + " is set as scaling reference: a comma separated concatenation of the maximum values in each dimension assumed as a reference. If no value is specified, the maximum value of the attribute range in this dimension will be taken.");
        private double[] maxima = new double[0];
        private double[] minima = new double[0];
        public static final OptionID SEED_ID = new OptionID("perturbationfilter.seed", "Seed for random noise generation.");
        protected Long seed = null;
        public static final OptionID PERCENTAGE_ID = new OptionID("perturbationfilter.percentage", "Percentage of the standard deviation of the random Gaussian noise generation per attribute, given the standard deviation of the corresponding attribute in the original data distribution (assuming a Gaussian distribution there).");
        public static final OptionID SCALINGREFERENCE_ID = new OptionID("perturbationfilter.scalingreference", "The reference for scaling the Gaussian noise. Default is " + (Object)((Object)ScalingReference.UNITCUBE) + ", parameter " + PERCENTAGE_ID.getName() + " will then directly define the standard deviation of all noise Gaussians. For options " + (Object)((Object)ScalingReference.STDDEV) + " and  " + (Object)((Object)ScalingReference.MINMAX) + ", the percentage of the attributewise standard deviation or extension, repectively, will define the attributewise standard deviation of the noise Gaussians.");
        public static final OptionID NOISEDISTRIBUTION_ID = new OptionID("perturbationfilter.noisedistribution", "The nature of the noise distribution, default is " + (Object)((Object)NoiseDistribution.UNIFORM));
        protected double percentage;
        protected ScalingReference scalingreference;
        protected NoiseDistribution noisedistribution;

        public void configure(Parameterization config) {
            new EnumParameter(SCALINGREFERENCE_ID, ScalingReference.class, (Enum)ScalingReference.UNITCUBE).grab(config, x -> {
                this.scalingreference = x;
            });
            new EnumParameter(NOISEDISTRIBUTION_ID, NoiseDistribution.class, (Enum)NoiseDistribution.UNIFORM).grab(config, x -> {
                this.noisedistribution = x;
            });
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(PERCENTAGE_ID, 0.01).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).addConstraint((ParameterConstraint)CommonConstraints.LESS_EQUAL_ONE_DOUBLE)).grab(config, x -> {
                this.percentage = x;
            });
            ((LongParameter)new LongParameter(SEED_ID).setOptional(true)).grab(config, x -> {
                this.seed = x;
            });
            DoubleListParameter minimaP = (DoubleListParameter)new DoubleListParameter(MINIMA_ID).setOptional(true);
            minimaP.grab(config, x -> {
                this.minima = (double[])x.clone();
            });
            DoubleListParameter maximaP = (DoubleListParameter)new DoubleListParameter(MAXIMA_ID).setOptional(!minimaP.isDefined());
            maximaP.grab(config, x -> {
                this.maxima = (double[])x.clone();
            });
            if (this.minima != null && this.maxima != null && this.minima.length != this.maxima.length) {
                config.reportError((ParameterException)new WrongParameterValueException((Parameter)minimaP, "and", (Parameter)maximaP, "must have the same number of values."));
            }
        }

        public PerturbationFilter<V> make() {
            return new PerturbationFilter(this.seed, this.percentage, this.scalingreference, this.minima, this.maxima, this.noisedistribution);
        }
    }

    public static enum NoiseDistribution {
        GAUSSIAN,
        UNIFORM;

    }

    public static enum ScalingReference {
        UNITCUBE,
        STDDEV,
        MINMAX;

    }
}

