/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier;

import elki.Algorithm;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.datastore.WritableIntegerDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDMIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDList;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.KNNList;
import elki.database.ids.ModifiableDBIDs;
import elki.database.query.QueryBuilder;
import elki.database.query.distance.DistanceQuery;
import elki.database.query.knn.KNNSearcher;
import elki.database.query.range.RangeSearcher;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.progress.IndefiniteProgress;
import elki.logging.progress.StepProgress;
import elki.math.DoubleMinMax;
import elki.math.Mean;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.InvertedOutlierScoreMeta;
import elki.result.outlier.OutlierResult;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;

@Title(value="DWOF: Dynamic Window Outlier Factor")
@Description(value="Algorithm to compute dynamic-window outlier factors in a database based on the neighborhood size parameter 'k'")
@Reference(authors="R. Momtaz, N. Mohssen, M. A. Gowayyed", title="DWOF: A Robust Density-Based Outlier Detection Approach", booktitle="Proc. 6th Iberian Conf. Pattern Recognition and Image Analysis (IbPRIA 2013)", url="https://doi.org/10.1007/978-3-642-38628-2_61", bibkey="DBLP:conf/ibpria/MomtazMG13")
public class DWOF<O>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(DWOF.class);
    protected Distance<? super O> distance;
    protected int kplus;
    protected double delta = 1.1;

    public DWOF(Distance<? super O> distance, int k, double delta) {
        this.distance = distance;
        this.kplus = k + 1;
        this.delta = delta;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{this.distance.getInputTypeRestriction()});
    }

    public OutlierResult run(Relation<O> relation) {
        IndefiniteProgress clusEvalProgress;
        DBIDs ids = relation.getDBIDs();
        QueryBuilder qb = new QueryBuilder(relation, this.distance);
        DistanceQuery distFunc = qb.distanceQuery();
        KNNSearcher knnq = qb.kNNByDBID(this.kplus);
        RangeSearcher rnnQuery = qb.rangeByDBID();
        StepProgress stepProg = LOG.isVerbose() ? new StepProgress("DWOF", 2) : null;
        WritableDoubleDataStore dwofs = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)30, (double)0.0);
        if (stepProg != null) {
            stepProg.beginStep(1, "Initializing objects' Radii", LOG);
        }
        WritableDoubleDataStore radii = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)3, (double)0.0);
        this.initializeRadii(ids, (KNNSearcher<DBIDRef>)knnq, distFunc, radii);
        WritableIntegerDataStore oldSizes = DataStoreUtil.makeIntegerStorage((DBIDs)ids, (int)2, (int)1);
        WritableIntegerDataStore newSizes = DataStoreUtil.makeIntegerStorage((DBIDs)ids, (int)2, (int)1);
        int countUnmerged = relation.size();
        if (stepProg != null) {
            stepProg.beginStep(2, "Clustering-Evaluating Cycles.", LOG);
        }
        IndefiniteProgress indefiniteProgress = clusEvalProgress = LOG.isVerbose() ? new IndefiniteProgress("Evaluating DWOFs", LOG) : null;
        while (countUnmerged > 0) {
            LOG.incrementProcessed((AbstractProgress)clusEvalProgress);
            DBIDIter iter = ids.iter();
            while (iter.valid()) {
                radii.putDouble((DBIDRef)iter, radii.doubleValue((DBIDRef)iter) * this.delta);
                iter.advance();
            }
            WritableDataStore labels = DataStoreUtil.makeStorage((DBIDs)ids, (int)1, ModifiableDBIDs.class);
            this.clusterData(ids, (RangeSearcher<DBIDRef>)rnnQuery, radii, (WritableDataStore<ModifiableDBIDs>)labels);
            WritableIntegerDataStore temp = newSizes;
            newSizes = oldSizes;
            oldSizes = temp;
            countUnmerged = this.updateSizes(ids, (WritableDataStore<ModifiableDBIDs>)labels, newSizes);
            labels.destroy();
            DBIDIter iter2 = ids.iter();
            while (iter2.valid()) {
                double newScore = newSizes.intValue((DBIDRef)iter2) > 0 ? (double)(oldSizes.intValue((DBIDRef)iter2) - 1) / (double)newSizes.intValue((DBIDRef)iter2) : 0.0;
                dwofs.putDouble((DBIDRef)iter2, dwofs.doubleValue((DBIDRef)iter2) + newScore);
                iter2.advance();
            }
        }
        LOG.setCompleted(clusEvalProgress);
        LOG.setCompleted(stepProg);
        DoubleMinMax minmax = new DoubleMinMax();
        DBIDIter iter = relation.iterDBIDs();
        while (iter.valid()) {
            minmax.put(dwofs.doubleValue((DBIDRef)iter));
            iter.advance();
        }
        InvertedOutlierScoreMeta meta = new InvertedOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY);
        MaterializedDoubleRelation rel = new MaterializedDoubleRelation("Dynamic-Window Outlier Factors", ids, (DoubleDataStore)dwofs);
        return new OutlierResult(meta, (DoubleRelation)rel);
    }

    private void initializeRadii(DBIDs ids, KNNSearcher<DBIDRef> knnq, DistanceQuery<O> distFunc, WritableDoubleDataStore radii) {
        FiniteProgress avgDistProgress = LOG.isVerbose() ? new FiniteProgress("Calculating average kNN distances-", ids.size(), LOG) : null;
        double absoluteMinDist = Double.POSITIVE_INFINITY;
        double minAvgDist = Double.POSITIVE_INFINITY;
        Mean mean = new Mean();
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            KNNList iterNeighbors = knnq.getKNN((Object)iter, this.kplus);
            mean.reset();
            DoubleDBIDListIter neighbor1 = iterNeighbors.iter();
            while (neighbor1.valid()) {
                if (!DBIDUtil.equal((DBIDRef)neighbor1, (DBIDRef)iter)) {
                    DoubleDBIDListIter neighbor2 = iterNeighbors.iter();
                    while (neighbor2.valid()) {
                        if (!DBIDUtil.equal((DBIDRef)neighbor1, (DBIDRef)neighbor2) && !DBIDUtil.equal((DBIDRef)neighbor2, (DBIDRef)iter)) {
                            double dist = distFunc.distance((DBIDRef)neighbor1, (DBIDRef)neighbor2);
                            mean.put(dist);
                            if (dist > 0.0 && dist < absoluteMinDist) {
                                absoluteMinDist = dist;
                            }
                        }
                        neighbor2.advance();
                    }
                }
                neighbor1.advance();
            }
            double currentMean = mean.getMean();
            radii.putDouble((DBIDRef)iter, currentMean);
            if (currentMean < minAvgDist) {
                minAvgDist = currentMean;
            }
            LOG.incrementProcessed((AbstractProgress)avgDistProgress);
            iter.advance();
        }
        LOG.ensureCompleted(avgDistProgress);
        iter = ids.iter();
        while (iter.valid()) {
            radii.putDouble((DBIDRef)iter, minAvgDist > 0.0 ? absoluteMinDist * radii.doubleValue((DBIDRef)iter) / minAvgDist : Double.POSITIVE_INFINITY);
            iter.advance();
        }
    }

    private void clusterData(DBIDs ids, RangeSearcher<DBIDRef> rnnQuery, WritableDoubleDataStore radii, WritableDataStore<ModifiableDBIDs> labels) {
        FiniteProgress clustProg = LOG.isVerbose() ? new FiniteProgress("Density-Based Clustering", ids.size(), LOG) : null;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            if (labels.get((DBIDRef)iter) == null) {
                ArrayModifiableDBIDs newCluster = DBIDUtil.newArray();
                newCluster.add((DBIDRef)iter);
                labels.put((DBIDRef)iter, (Object)newCluster);
                LOG.incrementProcessed((AbstractProgress)clustProg);
                ArrayModifiableDBIDs nChain = DBIDUtil.newArray();
                nChain.add((DBIDRef)iter);
                DBIDMIter toGetNeighbors = nChain.iter();
                while (toGetNeighbors.valid()) {
                    double range = radii.doubleValue((DBIDRef)toGetNeighbors);
                    DoubleDBIDList nNeighbors = rnnQuery.getRange((Object)toGetNeighbors, range);
                    DoubleDBIDListIter iter2 = nNeighbors.iter();
                    while (iter2.valid()) {
                        if (!DBIDUtil.equal((DBIDRef)toGetNeighbors, (DBIDRef)iter2)) {
                            if (labels.get((DBIDRef)iter2) == null) {
                                newCluster.add((DBIDRef)iter2);
                                labels.put((DBIDRef)iter2, (Object)newCluster);
                                nChain.add((DBIDRef)iter2);
                                LOG.incrementProcessed((AbstractProgress)clustProg);
                            } else if (labels.get((DBIDRef)iter2) != newCluster) {
                                ModifiableDBIDs toBeDeleted = (ModifiableDBIDs)labels.get((DBIDRef)iter2);
                                newCluster.addDBIDs((DBIDs)toBeDeleted);
                                DBIDMIter iter3 = toBeDeleted.iter();
                                while (iter3.valid()) {
                                    labels.put((DBIDRef)iter3, (Object)newCluster);
                                    iter3.advance();
                                }
                                toBeDeleted.clear();
                            }
                        }
                        iter2.advance();
                    }
                    toGetNeighbors.advance();
                }
            }
            iter.advance();
        }
        LOG.ensureCompleted(clustProg);
    }

    private int updateSizes(DBIDs ids, WritableDataStore<ModifiableDBIDs> labels, WritableIntegerDataStore newSizes) {
        int countUnmerged = 0;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            int newClusterSize = ((ModifiableDBIDs)labels.get((DBIDRef)iter)).size();
            newSizes.putInt((DBIDRef)iter, newClusterSize);
            if (newClusterSize == 1) {
                ++countUnmerged;
            }
            iter.advance();
        }
        return countUnmerged;
    }

    public static class Par<O>
    implements Parameterizer {
        public static final OptionID K_ID = OptionID.getOrCreateOptionID((String)"dwof.k", (String)"Number of neighbors to get for DWOF score outlier detection.");
        public static final OptionID DELTA_ID = OptionID.getOrCreateOptionID((String)"dwof.delta", (String)"Radius increase factor.");
        protected int k;
        protected double delta = 1.1;
        protected Distance<? super O> distance;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, Distance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(K_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_INT)).grab(config, x -> {
                this.k = x;
            });
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(DELTA_ID).setDefaultValue((Object)1.1)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ONE_DOUBLE)).grab(config, x -> {
                this.delta = x;
            });
        }

        public DWOF<O> make() {
            return new DWOF<O>(this.distance, this.k, this.delta);
        }
    }
}

