/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.distance;

import elki.Algorithm;
import elki.data.NumberVector;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBID;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDPair;
import elki.database.ids.HashSetModifiableDBIDs;
import elki.database.query.QueryBuilder;
import elki.database.query.distance.DistanceQuery;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.distance.minkowski.LPNormDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.math.DoubleMinMax;
import elki.math.spacefillingcurves.HilbertSpatialSorter;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.BasicOutlierScoreMeta;
import elki.result.outlier.OutlierResult;
import elki.utilities.datastructures.BitsUtil;
import elki.utilities.datastructures.heap.ComparableMaxHeap;
import elki.utilities.datastructures.heap.ComparatorMinHeap;
import elki.utilities.datastructures.heap.ObjectHeap;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.EnumParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import java.util.Arrays;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Set;
import net.jafama.FastMath;

@Title(value="Fast Outlier Detection in High Dimensional Spaces")
@Description(value="Algorithm to compute outliers using Hilbert space filling curves")
@Reference(authors="F. Angiulli, C. Pizzuti", title="Fast Outlier Detection in High Dimensional Spaces", booktitle="Proc. European Conf. Principles of Knowledge Discovery and Data Mining (PKDD'02)", url="https://doi.org/10.1007/3-540-45681-3_2", bibkey="DBLP:conf/pkdd/AngiulliP02")
public class HilOut<O extends NumberVector>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(HilOut.class);
    private Distance<? super O> distance;
    private int k;
    private int n;
    private int h;
    private double t;
    private Enum<ScoreType> tn;
    private DistanceQuery<O> distq;
    private int capital_n;
    private int n_star;
    private int capital_n_star;
    private int d;
    private double omega_star;

    public HilOut(LPNormDistance distance, int k, int n, int h, Enum<ScoreType> tn) {
        this.distance = distance;
        this.n = n;
        this.k = k - 1;
        this.h = h;
        this.tn = tn;
        this.t = distance.getP();
        this.n_star = 0;
        this.omega_star = 0.0;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public OutlierResult run(Relation<O> relation) {
        int i;
        this.distq = new QueryBuilder(relation, this.distance).distanceQuery();
        this.d = RelationUtil.dimensionality(relation);
        WritableDoubleDataStore hilout_weight = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)4);
        double diameter = 0.0;
        double[][] hbbs = RelationUtil.computeMinMax(relation);
        double[] min = hbbs[0];
        double[] max = hbbs[1];
        for (i = 0; i < this.d; ++i) {
            diameter = Math.max(diameter, max[i] - min[i]);
        }
        i = 0;
        while (i < this.d) {
            double diff = (diameter - (max[i] - min[i])) * 0.5;
            int n = i;
            min[n] = min[n] - diff;
            int n2 = i++;
            max[n2] = max[n2] + diff;
        }
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)("Rescaling dataset by " + 1.0 / diameter + " to fit the unit cube."));
        }
        this.capital_n_star = this.capital_n = relation.size();
        HilbertFeatures h = new HilbertFeatures(relation, min, diameter);
        FiniteProgress progressHilOut = LOG.isVerbose() ? new FiniteProgress("HilOut iterations", this.d + 1, LOG) : null;
        FiniteProgress progressTrueOut = LOG.isVerbose() ? new FiniteProgress("True outliers found", this.n, LOG) : null;
        for (int j = 0; j <= this.d && this.n_star < this.n; ++j) {
            HilFeature entry;
            h.out.clear();
            h.wlb.clear();
            h.initialize(0.5 * (double)j / (double)(this.d + 1));
            this.scan(h, (int)((double)(this.k * this.capital_n) / (double)this.capital_n_star));
            this.trueOutliers(h);
            if (progressTrueOut != null) {
                progressTrueOut.setProcessed(this.n_star, LOG);
            }
            h.top.clear();
            HashSetModifiableDBIDs top_keys = DBIDUtil.newHashSet((int)h.out.size());
            ObjectHeap.UnsortedIter iter = h.out.unsortedIter();
            while (iter.valid()) {
                entry = (HilFeature)iter.get();
                top_keys.add((DBIDRef)entry.id);
                h.top.add(entry);
                iter.advance();
            }
            iter = h.wlb.unsortedIter();
            while (iter.valid()) {
                entry = (HilFeature)iter.get();
                if (!top_keys.contains((DBIDRef)entry.id)) {
                    h.top.add(entry);
                }
                iter.advance();
            }
            LOG.incrementProcessed((AbstractProgress)progressHilOut);
        }
        if (this.n_star < this.n) {
            h.out.clear();
            h.wlb.clear();
            this.scan(h, this.capital_n);
        }
        if (progressHilOut != null) {
            progressHilOut.setProcessed(this.d, LOG);
            progressHilOut.ensureCompleted(LOG);
        }
        if (progressTrueOut != null) {
            progressTrueOut.setProcessed(this.n, LOG);
            progressTrueOut.ensureCompleted(LOG);
        }
        DoubleMinMax minmax = new DoubleMinMax();
        if (this.tn == ScoreType.TOPN) {
            minmax.put(0.0);
            DBIDIter iditer = relation.iterDBIDs();
            while (iditer.valid()) {
                hilout_weight.putDouble((DBIDRef)iditer, 0.0);
                iditer.advance();
            }
            ObjectHeap.UnsortedIter iter = h.out.unsortedIter();
            while (iter.valid()) {
                HilFeature ent = (HilFeature)iter.get();
                minmax.put(ent.ubound);
                hilout_weight.putDouble((DBIDRef)ent.id, ent.ubound);
                iter.advance();
            }
        } else {
            for (ObjectHeap.UnsortedIter ent : h.pf) {
                minmax.put(ent.ubound);
                hilout_weight.putDouble((DBIDRef)ent.id, ent.ubound);
            }
        }
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("HilOut weight", relation.getDBIDs(), (DoubleDataStore)hilout_weight);
        BasicOutlierScoreMeta scoreMeta = new BasicOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY);
        OutlierResult result = new OutlierResult(scoreMeta, (DoubleRelation)scoreResult);
        return result;
    }

    private void scan(HilbertFeatures hf, int k0) {
        int mink0 = Math.min(2 * k0, this.capital_n - 1);
        if (LOG.isDebuggingFine()) {
            LOG.debugFine((CharSequence)("Scanning with k0=" + k0 + " (" + mink0 + ") N*=" + this.capital_n_star));
        }
        for (int i = 0; i < hf.pf.length; ++i) {
            if (hf.pf[i].ubound < this.omega_star) continue;
            if (hf.pf[i].lbound < hf.pf[i].ubound) {
                double omega = hf.fastUpperBound(i);
                if (omega < this.omega_star) {
                    hf.pf[i].ubound = omega;
                } else {
                    int maxcount = hf.top.contains(hf.pf[i]) ? this.capital_n - 1 : mink0;
                    this.innerScan(hf, i, maxcount);
                }
            }
            if (hf.pf[i].ubound > 0.0) {
                hf.updateOUT(i);
            }
            if (hf.pf[i].lbound > 0.0) {
                hf.updateWLB(i);
            }
            if (hf.wlb.size() < this.n) continue;
            this.omega_star = Math.max(this.omega_star, ((HilFeature)((HilbertFeatures)hf).wlb.peek()).lbound);
        }
    }

    private void innerScan(HilbertFeatures hf, int i, int maxcount) {
        NumberVector p = (NumberVector)hf.relation.get((DBIDRef)hf.pf[i].id);
        int a = i;
        int b = i;
        int level = this.h;
        int levela = this.h;
        int levelb = this.h;
        for (int count = 0; count < maxcount; ++count) {
            double delta;
            int mlevel;
            int c;
            if (a == 0) {
                levelb = Math.min(levelb, hf.pf[b].level);
                c = ++b;
            } else if (b >= this.capital_n - 1) {
                levela = Math.min(levela, hf.pf[--a].level);
                c = a;
            } else if (hf.pf[a - 1].level >= hf.pf[b].level) {
                levela = Math.min(levela, hf.pf[--a].level);
                c = a;
            } else {
                levelb = Math.min(levelb, hf.pf[b].level);
                c = ++b;
            }
            if (hf.pf[i].nn_keys.contains((DBIDRef)hf.pf[c].id)) continue;
            hf.pf[i].insert(hf.pf[c].id, this.distq.distance((Object)p, (DBIDRef)hf.pf[c].id), this.k);
            if (hf.pf[i].nn.size() == this.k && (hf.pf[i].sum_nn < this.omega_star || (mlevel = Math.max(levela, levelb)) < level && (delta = hf.minDistLevel(hf.pf[i].id, level = mlevel)) >= ((DoubleDBIDPair)hf.pf[i].nn.peek()).doubleValue())) break;
        }
        double br = hf.boxRadius(i, a - 1, b + 1);
        double newlb = 0.0;
        double newub = 0.0;
        ObjectHeap.UnsortedIter iter = hf.pf[i].nn.unsortedIter();
        while (iter.valid()) {
            DoubleDBIDPair entry = (DoubleDBIDPair)iter.get();
            newub += entry.doubleValue();
            if (entry.doubleValue() <= br) {
                newlb += entry.doubleValue();
            }
            iter.advance();
        }
        if (newlb > hf.pf[i].lbound) {
            hf.pf[i].lbound = newlb;
        }
        if (newub < hf.pf[i].ubound) {
            hf.pf[i].ubound = newub;
        }
    }

    private void trueOutliers(HilbertFeatures h) {
        this.n_star = 0;
        ObjectHeap.UnsortedIter iter = h.out.unsortedIter();
        while (iter.valid()) {
            HilFeature entry = (HilFeature)iter.get();
            if (entry.ubound >= this.omega_star && entry.ubound - entry.lbound < 1.0E-10) {
                ++this.n_star;
            }
            iter.advance();
        }
    }

    public static class Par<O extends NumberVector>
    implements Parameterizer {
        public static final OptionID K_ID = new OptionID("HilOut.k", "Compute up to k next neighbors");
        public static final OptionID N_ID = new OptionID("HilOut.n", "Compute n outliers");
        public static final OptionID H_ID = new OptionID("HilOut.h", "Max. Hilbert-Level");
        public static final OptionID T_ID = new OptionID("HilOut.t", "t of Lt Metric");
        public static final OptionID TN_ID = new OptionID("HilOut.tn", "output of Top n or all elements");
        protected int k = 5;
        protected int n = 10;
        protected int h = 32;
        protected LPNormDistance distfunc;
        protected Enum<ScoreType> tn;

        public void configure(Parameterization config) {
            new IntParameter(K_ID, 5).grab(config, x -> {
                this.k = x;
            });
            new IntParameter(N_ID, 10).grab(config, x -> {
                this.n = x;
            });
            new IntParameter(H_ID, 32).grab(config, x -> {
                this.h = x;
            });
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, LPNormDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distfunc = x;
            });
            new EnumParameter(TN_ID, ScoreType.class, (Enum)ScoreType.TOPN).grab(config, x -> {
                this.tn = x;
            });
        }

        public HilOut<O> make() {
            return new HilOut(this.distfunc, this.k, this.n, this.h, this.tn);
        }
    }

    static final class HilFeature
    implements Comparable<HilFeature> {
        public DBID id;
        public long[] hilbert = null;
        public int level = 0;
        public double ubound = Double.POSITIVE_INFINITY;
        public double lbound = 0.0;
        public ObjectHeap<DoubleDBIDPair> nn;
        public HashSetModifiableDBIDs nn_keys = DBIDUtil.newHashSet();
        public double sum_nn = 0.0;

        public HilFeature(DBID id, ObjectHeap<DoubleDBIDPair> nn) {
            this.id = id;
            this.nn = nn;
        }

        @Override
        public int compareTo(HilFeature o) {
            return BitsUtil.compare((long[])this.hilbert, (long[])o.hilbert);
        }

        protected void insert(DBID id, double dt, int k) {
            if (this.nn.size() < k) {
                DoubleDBIDPair entry = DBIDUtil.newPair((double)dt, (DBIDRef)id);
                this.nn.add((Object)entry);
                this.nn_keys.add((DBIDRef)id);
                this.sum_nn += dt;
            } else {
                DoubleDBIDPair head = (DoubleDBIDPair)this.nn.peek();
                if (dt < head.doubleValue()) {
                    head = (DoubleDBIDPair)this.nn.poll();
                    this.sum_nn -= head.doubleValue();
                    this.nn_keys.remove((DBIDRef)head);
                    DoubleDBIDPair entry = DBIDUtil.newPair((double)dt, (DBIDRef)id);
                    this.nn.add((Object)entry);
                    this.nn_keys.add((DBIDRef)id);
                    this.sum_nn += dt;
                }
            }
        }
    }

    class HilbertFeatures {
        Relation<O> relation;
        HilFeature[] pf;
        double[] min;
        double diameter;
        double shift;
        private Set<HilFeature> top;
        private ObjectHeap<HilFeature> out;
        private ObjectHeap<HilFeature> wlb;

        public HilbertFeatures(Relation<O> relation, double[] min, double diameter) {
            this.relation = relation;
            this.min = min;
            this.diameter = diameter;
            this.pf = new HilFeature[relation.size()];
            int pos = 0;
            DBIDIter iditer = relation.iterDBIDs();
            while (iditer.valid()) {
                this.pf[pos++] = new HilFeature(DBIDUtil.deref((DBIDRef)iditer), (ObjectHeap<DoubleDBIDPair>)new ComparableMaxHeap(HilOut.this.k));
                iditer.advance();
            }
            this.out = new ComparatorMinHeap(HilOut.this.n, (Comparator)new Comparator<HilFeature>(){

                @Override
                public int compare(HilFeature o1, HilFeature o2) {
                    return Double.compare(o1.ubound, o2.ubound);
                }
            });
            this.wlb = new ComparatorMinHeap(HilOut.this.n, (Comparator)new Comparator<HilFeature>(){

                @Override
                public int compare(HilFeature o1, HilFeature o2) {
                    return Double.compare(o1.lbound, o2.lbound);
                }
            });
            this.top = new HashSet<HilFeature>(2 * HilOut.this.n);
        }

        private void initialize(double shift) {
            int i;
            Object[] coord;
            int i2;
            this.shift = shift;
            if (HilOut.this.h >= 32) {
                long scale = Long.MAX_VALUE;
                for (int i3 = 0; i3 < this.pf.length; ++i3) {
                    NumberVector obj = (NumberVector)this.relation.get((DBIDRef)this.pf[i3].id);
                    long[] coord2 = new long[HilOut.this.d];
                    for (int dim = 0; dim < HilOut.this.d; ++dim) {
                        coord2[dim] = (long)(this.getDimForObject(obj, dim) * 0.5 * 9.223372036854776E18);
                    }
                    this.pf[i3].hilbert = HilbertSpatialSorter.coordinatesToHilbert((long[])coord2, (int)HilOut.this.h, (int)1);
                }
            } else if (HilOut.this.h >= 16) {
                int scale = Integer.MAX_VALUE;
                for (i2 = 0; i2 < this.pf.length; ++i2) {
                    NumberVector obj = (NumberVector)this.relation.get((DBIDRef)this.pf[i2].id);
                    coord = new int[HilOut.this.d];
                    for (int dim = 0; dim < HilOut.this.d; ++dim) {
                        coord[dim] = (int)(this.getDimForObject(obj, dim) * 0.5 * 2.147483647E9);
                    }
                    this.pf[i2].hilbert = HilbertSpatialSorter.coordinatesToHilbert((int[])coord, (int)HilOut.this.h, (int)1);
                }
            } else if (HilOut.this.h >= 8) {
                int scale = 65535;
                for (i2 = 0; i2 < this.pf.length; ++i2) {
                    NumberVector obj = (NumberVector)this.relation.get((DBIDRef)this.pf[i2].id);
                    coord = new short[HilOut.this.d];
                    for (int dim = 0; dim < HilOut.this.d; ++dim) {
                        coord[dim] = (short)(this.getDimForObject(obj, dim) * 0.5 * 65535.0);
                    }
                    this.pf[i2].hilbert = HilbertSpatialSorter.coordinatesToHilbert((short[])coord, (int)HilOut.this.h, (int)16);
                }
            } else {
                int scale = 0xFFFFFF;
                for (i2 = 0; i2 < this.pf.length; ++i2) {
                    NumberVector obj = (NumberVector)this.relation.get((DBIDRef)this.pf[i2].id);
                    coord = new byte[HilOut.this.d];
                    for (int dim = 0; dim < HilOut.this.d; ++dim) {
                        coord[dim] = (byte)(this.getDimForObject(obj, dim) * 0.5 * 1.6777215E7);
                    }
                    this.pf[i2].hilbert = HilbertSpatialSorter.coordinatesToHilbert((byte[])coord, (int)HilOut.this.h, (int)24);
                }
            }
            Arrays.sort(this.pf);
            for (i = 0; i < this.pf.length - 1; ++i) {
                this.pf[i].level = this.minRegLevel(i, i + 1);
            }
            HilOut.this.capital_n_star = 0;
            for (i = 0; i < this.pf.length; ++i) {
                if (!(this.pf[i].ubound >= HilOut.this.omega_star)) continue;
                HilOut.this.capital_n_star++;
            }
        }

        private void updateOUT(int i) {
            if (this.out.size() < HilOut.this.n) {
                this.out.add((Object)this.pf[i]);
            } else {
                HilFeature head = (HilFeature)this.out.peek();
                if (this.pf[i].ubound > head.ubound) {
                    this.out.replaceTopElement((Object)this.pf[i]);
                }
            }
        }

        private void updateWLB(int i) {
            if (this.wlb.size() < HilOut.this.n) {
                this.wlb.add((Object)this.pf[i]);
            } else {
                HilFeature head = (HilFeature)this.wlb.peek();
                if (this.pf[i].lbound > head.lbound) {
                    this.wlb.replaceTopElement((Object)this.pf[i]);
                }
            }
        }

        private double fastUpperBound(int i) {
            int pre = i;
            int post = i;
            while (post - pre < HilOut.this.k) {
                int post_level;
                int pre_level = pre - 1 >= 0 ? this.pf[pre - 1].level : -2;
                int n = post_level = post < HilOut.this.capital_n - 1 ? this.pf[post].level : -2;
                if (post_level >= pre_level) {
                    ++post;
                    continue;
                }
                --pre;
            }
            return (double)HilOut.this.k * this.maxDistLevel(this.pf[i].id, this.minRegLevel(pre, post));
        }

        private double minDistLevel(DBID id, int level) {
            NumberVector obj = (NumberVector)this.relation.get((DBIDRef)id);
            double r = 1.0 / (double)(1 << level - 1);
            double dist = Double.POSITIVE_INFINITY;
            for (int dim = 0; dim < HilOut.this.d; ++dim) {
                double p_m_r = this.getDimForObject(obj, dim) % r;
                dist = Math.min(dist, Math.min(p_m_r, r - p_m_r));
            }
            return dist * this.diameter;
        }

        private double maxDistLevel(DBID id, int level) {
            double dist;
            NumberVector obj = (NumberVector)this.relation.get((DBIDRef)id);
            double r = 1.0 / (double)(1 << level - 1);
            if (HilOut.this.t == 1.0) {
                dist = 0.0;
                for (int dim = 0; dim < HilOut.this.d; ++dim) {
                    double p_m_r = this.getDimForObject(obj, dim) % r;
                    dist += Math.max(p_m_r, r - p_m_r);
                }
            } else if (HilOut.this.t == 2.0) {
                dist = 0.0;
                for (int dim = 0; dim < HilOut.this.d; ++dim) {
                    double p_m_r = this.getDimForObject(obj, dim) % r;
                    double a = Math.max(p_m_r, r - p_m_r);
                    dist += a * a;
                }
                dist = Math.sqrt(dist);
            } else if (!Double.isInfinite(HilOut.this.t)) {
                dist = 0.0;
                for (int dim = 0; dim < HilOut.this.d; ++dim) {
                    double p_m_r = this.getDimForObject(obj, dim) % r;
                    dist += FastMath.pow((double)Math.max(p_m_r, r - p_m_r), (double)HilOut.this.t);
                }
                dist = FastMath.pow((double)dist, (double)(1.0 / HilOut.this.t));
            } else {
                dist = Double.NEGATIVE_INFINITY;
                for (int dim = 0; dim < HilOut.this.d; ++dim) {
                    double p_m_r = this.getDimForObject(obj, dim) % r;
                    dist = Math.max(dist, Math.max(p_m_r, r - p_m_r));
                }
            }
            return dist * this.diameter;
        }

        private int numberSharedLevels(long[] a, long[] b) {
            int i = 0;
            int j = a.length - 1;
            while (i < a.length) {
                long diff = a[j] ^ b[j];
                if (diff != 0L) {
                    int expected = a.length * 64 - HilOut.this.d * HilOut.this.h;
                    return (BitsUtil.numberOfLeadingZeros((long)diff) + i * 64 - expected) / HilOut.this.d;
                }
                ++i;
                --j;
            }
            return HilOut.this.h - 1;
        }

        private int minRegLevel(int a, int b) {
            return this.numberSharedLevels(this.pf[a].hilbert, this.pf[b].hilbert);
        }

        private int maxRegLevel(int ref, int q) {
            return this.numberSharedLevels(this.pf[ref].hilbert, this.pf[q].hilbert) + 1;
        }

        private double boxRadius(int i, int a, int b) {
            int level;
            if (a < 0) {
                if (b >= this.pf.length) {
                    return Double.POSITIVE_INFINITY;
                }
                level = this.maxRegLevel(i, b);
            } else {
                level = b >= this.pf.length ? this.maxRegLevel(i, a) : Math.max(this.maxRegLevel(i, a), this.maxRegLevel(i, b));
            }
            return this.minDistLevel(this.pf[i].id, level);
        }

        private double getDimForObject(NumberVector obj, int dim) {
            return (obj.doubleValue(dim) - this.min[dim]) / this.diameter + this.shift;
        }
    }

    public static enum ScoreType {
        ALL,
        TOPN;

    }
}

