/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.lof;

import elki.Algorithm;
import elki.data.NumberVector;
import elki.data.type.CombinedTypeInformation;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.ArrayModifiableDBIDs;
import elki.database.ids.DBIDArrayMIter;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.NumberVectorDistance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.math.DoubleMinMax;
import elki.math.MathUtil;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.OutlierResult;
import elki.result.outlier.QuotientOutlierScoreMeta;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import elki.utilities.optionhandling.parameters.RandomParameter;
import elki.utilities.random.RandomFactory;
import java.util.ArrayList;
import java.util.List;
import java.util.Random;
import net.jafama.FastMath;

@Title(value="Approximate LOCI: Fast Outlier Detection Using the Local Correlation Integral")
@Description(value="Algorithm to compute outliers based on the Local Correlation Integral")
@Reference(authors="S. Papadimitriou, H. Kitagawa, P. B. Gibbons, C. Faloutsos", title="LOCI: Fast Outlier Detection Using the Local Correlation Integral", booktitle="Proc. 19th IEEE Int. Conf. on Data Engineering (ICDE '03)", url="https://doi.org/10.1109/ICDE.2003.1260802", bibkey="DBLP:conf/icde/PapadimitriouKGF03")
public class ALOCI<V extends NumberVector>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(ALOCI.class);
    private NumberVectorDistance<? super V> distance;
    private int nmin;
    private int alpha;
    private int g;
    private RandomFactory rnd;

    public ALOCI(NumberVectorDistance<? super V> distance, int nmin, int alpha, int g, RandomFactory rnd) {
        this.distance = distance;
        this.nmin = nmin;
        this.alpha = alpha;
        this.g = g;
        this.rnd = rnd;
    }

    public OutlierResult run(Relation<V> relation) {
        int i;
        int dim = RelationUtil.dimensionality(relation);
        Random random = this.rnd.getSingleThreadedRandom();
        FiniteProgress progressPreproc = LOG.isVerbose() ? new FiniteProgress("Build aLOCI quadtress", this.g, LOG) : null;
        double[][] hbbs = RelationUtil.computeMinMax(relation);
        double[] min = hbbs[0];
        double[] max = hbbs[1];
        double maxd = 0.0;
        for (i = 0; i < dim; ++i) {
            maxd = MathUtil.max((double)maxd, (double)(max[i] - min[i]));
        }
        i = 0;
        while (i < dim) {
            double diff = (maxd - (max[i] - min[i])) * 0.5;
            int n = i;
            min[n] = min[n] - diff;
            int n2 = i++;
            max[n2] = max[n2] + diff;
        }
        ArrayList<ALOCIQuadTree> qts = new ArrayList<ALOCIQuadTree>(this.g);
        double[] nshift = new double[dim];
        ALOCIQuadTree qt = new ALOCIQuadTree(min, max, nshift, this.nmin, relation);
        qts.add(qt);
        LOG.incrementProcessed((AbstractProgress)progressPreproc);
        for (int shift = 1; shift < this.g; ++shift) {
            double[] svec = new double[dim];
            for (int i2 = 0; i2 < dim; ++i2) {
                svec[i2] = random.nextDouble() * (max[i2] - min[i2]);
            }
            qt = new ALOCIQuadTree(min, max, svec, this.nmin, relation);
            qts.add(qt);
            LOG.incrementProcessed((AbstractProgress)progressPreproc);
        }
        LOG.ensureCompleted(progressPreproc);
        FiniteProgress progressLOCI = LOG.isVerbose() ? new FiniteProgress("Compute aLOCI scores", relation.size(), LOG) : null;
        WritableDoubleDataStore mdef_norm = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)4);
        DoubleMinMax minmax = new DoubleMinMax();
        NumberVectorDistance<? super V> distFunc = this.distance;
        DBIDIter iditer = relation.iterDBIDs();
        while (iditer.valid()) {
            NumberVector obj = (NumberVector)relation.get((DBIDRef)iditer);
            double maxmdefnorm = 0.0;
            int l = 0;
            while (true) {
                Node ci = null;
                for (int i3 = 0; i3 < this.g; ++i3) {
                    Node ci2 = ((ALOCIQuadTree)qts.get(i3)).findClosestNode(obj, l);
                    if (ci2.getLevel() != l || ci != null && !(distFunc.distance((NumberVector)ci, obj) > distFunc.distance((NumberVector)ci2, obj))) continue;
                    ci = ci2;
                }
                if (ci == null) break;
                Node cj = null;
                for (int i4 = 0; i4 < this.g; ++i4) {
                    Node cj2 = ((ALOCIQuadTree)qts.get(i4)).findClosestNode(ci, l - this.alpha);
                    if (cj != null && cj2.getLevel() < cj.getLevel() || cj != null && !(distFunc.distance((NumberVector)cj, (NumberVector)ci) > distFunc.distance((NumberVector)cj2, (NumberVector)ci))) continue;
                    cj = cj2;
                }
                if (cj != null) {
                    double mdefnorm = ALOCI.calculate_MDEF_norm(cj, ci);
                    maxmdefnorm = MathUtil.max((double)maxmdefnorm, (double)mdefnorm);
                }
                ++l;
            }
            mdef_norm.putDouble((DBIDRef)iditer, maxmdefnorm);
            minmax.put(maxmdefnorm);
            LOG.incrementProcessed((AbstractProgress)progressLOCI);
            iditer.advance();
        }
        LOG.ensureCompleted(progressLOCI);
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("aLOCI normalized MDEF", relation.getDBIDs(), (DoubleDataStore)mdef_norm);
        QuotientOutlierScoreMeta scoreMeta = new QuotientOutlierScoreMeta(minmax.getMin(), minmax.getMax(), 0.0, Double.POSITIVE_INFINITY);
        OutlierResult result = new OutlierResult(scoreMeta, (DoubleRelation)scoreResult);
        return result;
    }

    private static double calculate_MDEF_norm(Node sn, Node cg) {
        long sq = sn.getSquareSum(cg.getLevel() - sn.getLevel());
        if (sq == (long)sn.getCount()) {
            return 0.0;
        }
        long cb = sn.getCubicSum(cg.getLevel() - sn.getLevel());
        double n_hat = (double)sq / (double)sn.getCount();
        double sig_n_hat = Math.sqrt(cb * (long)sn.getCount() - sq * sq) / (double)sn.getCount();
        if (sig_n_hat < Double.MIN_NORMAL) {
            return 0.0;
        }
        double mdef = n_hat - (double)cg.getCount();
        return mdef / sig_n_hat;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{new CombinedTypeInformation(new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD, this.distance.getInputTypeRestriction()})});
    }

    public static class Par<O extends NumberVector>
    implements Parameterizer {
        public static final OptionID NMIN_ID = new OptionID("loci.nmin", "Minimum neighborhood size to be considered.");
        public static final OptionID ALPHA_ID = new OptionID("loci.alpha", "Scaling factor for averaging neighborhood");
        public static final OptionID GRIDS_ID = new OptionID("loci.g", "The number of Grids to use.");
        public static final OptionID SEED_ID = new OptionID("loci.seed", "The seed to use for initializing Random.");
        protected int nmin = 0;
        protected int alpha = 4;
        protected int g = 1;
        protected RandomFactory rnd;
        protected NumberVectorDistance<? super O> distance;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, NumberVectorDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(NMIN_ID, 20).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.nmin = x;
            });
            ((IntParameter)new IntParameter(GRIDS_ID, 1).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.g = x;
            });
            ((IntParameter)new IntParameter(ALPHA_ID, 4).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_DOUBLE)).grab(config, x -> {
                this.alpha = x;
            });
            new RandomParameter(SEED_ID).grab(config, x -> {
                this.rnd = x;
            });
        }

        public ALOCI<O> make() {
            return new ALOCI<O>(this.distance, this.nmin, this.alpha, this.g, this.rnd);
        }
    }

    static class Node
    implements NumberVector {
        final int code;
        final int count;
        final int level;
        List<Node> children;
        Node parent = null;
        double[] center;

        protected Node(int code, double[] center, int count, int level, List<Node> children) {
            this.code = code;
            this.center = center;
            this.count = count;
            this.level = level;
            this.children = children;
            if (children != null) {
                for (Node child : children) {
                    child.parent = this;
                }
            }
        }

        public int getLevel() {
            return this.level;
        }

        public int getCount() {
            return this.count;
        }

        public long getSquareSum(int levels) {
            if (levels <= 0 || this.children == null) {
                return (long)this.count * (long)this.count;
            }
            long agg = 0L;
            for (Node child : this.children) {
                agg += child.getSquareSum(levels - 1);
            }
            return agg;
        }

        public long getCubicSum(int levels) {
            if (levels <= 0 || this.children == null) {
                return (long)this.count * (long)this.count * (long)this.count;
            }
            long agg = 0L;
            for (Node child : this.children) {
                agg += child.getCubicSum(levels - 1);
            }
            return agg;
        }

        public int getDimensionality() {
            return this.center.length;
        }

        public double doubleValue(int dimension) {
            return this.center[dimension];
        }

        public long longValue(int dimension) {
            return (long)this.center[dimension];
        }

        public double[] toArray() {
            return (double[])this.center.clone();
        }
    }

    static class ALOCIQuadTree {
        private double[] shift;
        private double[] min;
        private double[] width;
        private int nmin;
        Node root;
        private Relation<? extends NumberVector> relation;

        public ALOCIQuadTree(double[] min, double[] max, double[] shift, int nmin, Relation<? extends NumberVector> relation) {
            assert (min.length <= 32) : "Quadtrees are only supported for up to 32 dimensions";
            this.shift = shift;
            this.nmin = nmin;
            this.min = min;
            this.width = new double[min.length];
            for (int d = 0; d < min.length; ++d) {
                this.width[d] = max[d] - min[d];
                if (!(this.width[d] <= 0.0)) continue;
                this.width[d] = 1.0;
            }
            double[] center = new double[min.length];
            for (int d = 0; d < min.length; ++d) {
                center[d] = min[d] + shift[d] + this.width[d] * (shift[d] < this.width[d] * 0.5 ? 0.5 : -0.5);
            }
            this.relation = relation;
            ArrayModifiableDBIDs ids = DBIDUtil.newArray((DBIDs)relation.getDBIDs());
            ArrayList<Node> children = new ArrayList<Node>();
            this.bulkLoad((double[])min.clone(), (double[])max.clone(), children, ids, 0, ids.size(), 0, 0, 0);
            this.root = new Node(0, center, ids.size(), -1, children);
        }

        private void bulkLoad(double[] lmin, double[] lmax, List<Node> children, ArrayModifiableDBIDs ids, int start, int end, int dim, int level, int code) {
            if (dim == 0) {
                int d;
                DBIDArrayMIter iter = ids.iter();
                iter.seek(start);
                NumberVector first = (NumberVector)this.relation.get((DBIDRef)iter);
                iter.advance();
                boolean degenerate = true;
                block0: while (iter.getOffset() < end) {
                    NumberVector other = (NumberVector)this.relation.get((DBIDRef)iter);
                    for (d = 0; d < lmin.length; ++d) {
                        if (!(Math.abs(first.doubleValue(d) - other.doubleValue(d)) > 1.0E-15)) continue;
                        degenerate = false;
                        break block0;
                    }
                    iter.advance();
                }
                if (degenerate) {
                    double[] center = new double[lmin.length];
                    for (d = 0; d < lmin.length; ++d) {
                        center[d] = lmin[d] * 0.5 + lmax[d] * 0.5 + this.shift[d];
                        if (!(center[d] > this.min[d] + this.width[d])) continue;
                        int n = d;
                        center[n] = center[n] - this.width[d];
                    }
                    children.add(new Node(code, center, end - start, level, null));
                    return;
                }
            }
            if (dim == lmin.length) {
                double[] center = new double[lmin.length];
                for (int d = 0; d < lmin.length; ++d) {
                    center[d] = lmin[d] * 0.5 + lmax[d] * 0.5 + this.shift[d];
                    if (!(center[d] > this.min[d] + this.width[d])) continue;
                    int n = d;
                    center[n] = center[n] - this.width[d];
                }
                if (end - start < this.nmin) {
                    children.add(new Node(code, center, end - start, level, null));
                    return;
                }
                ArrayList<Node> newchildren = new ArrayList<Node>();
                this.bulkLoad(lmin, lmax, newchildren, ids, start, end, 0, level + 1, 0);
                children.add(new Node(code, center, end - start, level, newchildren));
                return;
            }
            DBIDArrayMIter siter = ids.iter();
            DBIDArrayMIter eiter = ids.iter();
            siter.seek(start);
            eiter.seek(end - 1);
            while (siter.getOffset() < eiter.getOffset()) {
                if (this.getShiftedDim((NumberVector)this.relation.get((DBIDRef)siter), dim, level) <= 0.5) {
                    siter.advance();
                    continue;
                }
                if (this.getShiftedDim((NumberVector)this.relation.get((DBIDRef)eiter), dim, level) > 0.5) {
                    eiter.retract();
                    continue;
                }
                ids.swap(siter.getOffset(), eiter.getOffset() - 1);
                siter.advance();
                eiter.retract();
            }
            int spos = siter.getOffset();
            if (start < spos) {
                double tmp = lmax[dim];
                lmax[dim] = lmax[dim] * 0.5 + lmin[dim] * 0.5;
                this.bulkLoad(lmin, lmax, children, ids, start, spos, dim + 1, level, code);
                lmax[dim] = tmp;
            }
            if (spos < end) {
                double tmp = lmin[dim];
                lmin[dim] = lmax[dim] * 0.5 + lmin[dim] * 0.5;
                this.bulkLoad(lmin, lmax, children, ids, spos, end, dim + 1, level, code | 1 << dim);
                lmin[dim] = tmp;
            }
        }

        private double getShiftedDim(NumberVector obj, int dim, int level) {
            double pos = obj.doubleValue(dim) + this.shift[dim];
            pos = (pos - this.min[dim]) / this.width[dim] * (double)(1 + level);
            return pos - FastMath.floor((double)pos);
        }

        public Node findClosestNode(NumberVector vec, int tlevel) {
            Node cur = this.root;
            for (int level = 0; level <= tlevel && cur.children != null; ++level) {
                int code = 0;
                for (int d = 0; d < this.min.length; ++d) {
                    if (!(this.getShiftedDim(vec, d, level) > 0.5)) continue;
                    code |= 1 << d;
                }
                boolean found = false;
                for (Node child : cur.children) {
                    if (child.code != code) continue;
                    cur = child;
                    found = true;
                    break;
                }
                if (!found) break;
            }
            return cur;
        }
    }
}

