/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.lof;

import elki.Algorithm;
import elki.data.type.CombinedTypeInformation;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.KNNList;
import elki.database.query.QueryBuilder;
import elki.database.query.knn.KNNSearcher;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.math.DoubleMinMax;
import elki.math.MathUtil;
import elki.math.MeanVariance;
import elki.math.statistics.distribution.NormalDistribution;
import elki.math.statistics.kernelfunctions.GaussianKernelDensityFunction;
import elki.math.statistics.kernelfunctions.KernelDensityFunction;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.OutlierResult;
import elki.result.outlier.ProbabilisticOutlierScore;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.exceptions.AbortException;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.ParameterException;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.WrongParameterValueException;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import elki.utilities.optionhandling.parameters.Parameter;

@Title(value="KDEOS: Kernel Density Estimator Outlier Score")
@Reference(authors="Erich Schubert, Arthur Zimek, Hans-Peter Kriegel", title="Generalized Outlier Detection with Flexible Kernel Density Estimates", booktitle="Proc. 14th SIAM International Conference on Data Mining (SDM 2014)", url="https://doi.org/10.1137/1.9781611973440.63", bibkey="DBLP:conf/sdm/SchubertZK14")
public class KDEOS<O>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(KDEOS.class);
    private static final double CUTOFF = 1.0E-20;
    protected Distance<? super O> distance;
    protected KernelDensityFunction kernel;
    protected int kmin;
    protected int kmax;
    protected double scale;
    protected double minBandwidth = 1.0E-6;
    protected int idim = -1;

    public KDEOS(Distance<? super O> distance, int kmin, int kmax, KernelDensityFunction kernel, double minBandwidth, double scale, int idim) {
        this.distance = distance;
        this.kmin = kmin;
        this.kmax = kmax;
        this.kernel = kernel;
        this.minBandwidth = minBandwidth;
        this.scale = scale;
        this.idim = idim;
    }

    public TypeInformation[] getInputTypeRestriction() {
        TypeInformation res = this.distance.getInputTypeRestriction();
        res = this.idim == 0 ? res : new CombinedTypeInformation(new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD, res});
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{res});
    }

    public OutlierResult run(Relation<O> rel) {
        DBIDs ids = rel.getDBIDs();
        KNNSearcher knnq = new QueryBuilder(rel, this.distance).precomputed().kNNByDBID(this.kmax + 1);
        WritableDataStore densities = DataStoreUtil.makeStorage((DBIDs)ids, (int)3, double[].class);
        this.estimateDensities(rel, (KNNSearcher<DBIDRef>)knnq, ids, (WritableDataStore<double[]>)densities);
        WritableDoubleDataStore kofs = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)30);
        DoubleMinMax minmax = new DoubleMinMax();
        this.computeOutlierScores((KNNSearcher<DBIDRef>)knnq, ids, (WritableDataStore<double[]>)densities, kofs, minmax);
        MaterializedDoubleRelation scoreres = new MaterializedDoubleRelation("Kernel Density Estimation Outlier Scores", ids, (DoubleDataStore)kofs);
        ProbabilisticOutlierScore meta = new ProbabilisticOutlierScore(minmax.getMin(), minmax.getMax());
        return new OutlierResult(meta, (DoubleRelation)scoreres);
    }

    protected void estimateDensities(Relation<O> rel, KNNSearcher<DBIDRef> knnq, DBIDs ids, WritableDataStore<double[]> densities) {
        int dim = this.dimensionality(rel);
        int knum = this.kmax + 1 - this.kmin;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            densities.put((DBIDRef)iter, (Object)new double[knum]);
            iter.advance();
        }
        FiniteProgress prog = LOG.isVerbose() ? new FiniteProgress("Computing densities", ids.size(), LOG) : null;
        double iminbw = this.minBandwidth > 0.0 ? 1.0 / (this.minBandwidth * this.scale) : Double.POSITIVE_INFINITY;
        DBIDIter iter2 = ids.iter();
        while (iter2.valid()) {
            KNNList neighbors = knnq.getKNN((Object)iter2, this.kmax + 1);
            int idx = 0;
            double sum = 0.0;
            DoubleDBIDListIter kneighbor = neighbors.iter();
            for (int k = 1; k <= this.kmax && kneighbor.valid(); ++k) {
                sum += kneighbor.doubleValue();
                if (k >= this.kmin) {
                    double ibw = Math.min((double)k / (sum * this.scale), iminbw);
                    double sca = MathUtil.powi((double)ibw, (int)dim);
                    DoubleDBIDListIter neighbor = neighbors.iter();
                    while (neighbor.valid()) {
                        double dens = sca < Double.POSITIVE_INFINITY ? sca * this.kernel.density(neighbor.doubleValue() * ibw) : (neighbor.doubleValue() == 0.0 ? 1.0 : 0.0);
                        double[] dArray = (double[])densities.get((DBIDRef)neighbor);
                        int n = idx;
                        dArray[n] = dArray[n] + dens;
                        if (dens < 1.0E-20) break;
                        neighbor.advance();
                    }
                    ++idx;
                }
                kneighbor.advance();
            }
            LOG.incrementProcessed((AbstractProgress)prog);
            iter2.advance();
        }
        LOG.ensureCompleted(prog);
    }

    private int dimensionality(Relation<O> rel) {
        if (this.idim >= 0) {
            return this.idim;
        }
        Relation<O> frel = rel;
        int dim = RelationUtil.dimensionality(frel);
        if (dim < 1) {
            throw new AbortException("When using KDEOS with non-vectorspace data, the intrinsic dimensionality parameter must be set!");
        }
        return dim;
    }

    protected void computeOutlierScores(KNNSearcher<DBIDRef> knnq, DBIDs ids, WritableDataStore<double[]> densities, WritableDoubleDataStore kdeos, DoubleMinMax minmax) {
        int knum = this.kmax + 1 - this.kmin;
        FiniteProgress prog = LOG.isVerbose() ? new FiniteProgress("Computing KDEOS scores", ids.size(), LOG) : null;
        double[][] scratch = new double[knum][this.kmax + 5];
        MeanVariance mv = new MeanVariance();
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            double[] dens = (double[])densities.get((DBIDRef)iter);
            KNNList neighbors = knnq.getKNN((Object)iter, this.kmax + 1);
            if (scratch[0].length < neighbors.size()) {
                scratch = new double[knum][neighbors.size() + 5];
            }
            int i = 0;
            DoubleDBIDListIter neighbor = neighbors.iter();
            while (neighbor.valid()) {
                double[] ndens = (double[])densities.get((DBIDRef)neighbor);
                for (int k = 0; k < knum; ++k) {
                    scratch[k][i] = ndens[k];
                }
                neighbor.advance();
                ++i;
            }
            assert (i == neighbors.size());
            double score = 0.0;
            for (int i2 = 0; i2 < knum; ++i2) {
                mv.reset();
                for (int j = 0; j < neighbors.size(); ++j) {
                    mv.put(scratch[i2][j]);
                }
                double mean = mv.getMean();
                double stddev = mv.getSampleStddev();
                if (!(stddev > 0.0)) continue;
                score += (mean - dens[i2]) / stddev;
            }
            score /= (double)knum;
            score = NormalDistribution.standardNormalCDF((double)score);
            minmax.put(score);
            kdeos.put((DBIDRef)iter, score);
            LOG.incrementProcessed((AbstractProgress)prog);
            iter.advance();
        }
        LOG.ensureCompleted(prog);
    }

    public static class Par<O>
    implements Parameterizer {
        public static final OptionID KERNEL_ID = new OptionID("kdeos.kernel", "Kernel density function to use.");
        public static final OptionID KERNEL_MIN_ID = new OptionID("kdeos.kernel.minbw", "Minimum bandwidth for kernel density estimation.");
        public static final OptionID KERNEL_SCALE_ID = new OptionID("kdeos.kernel.scale", "Scaling factor for the kernel function.");
        public static final OptionID KMIN_ID = new OptionID("kdeos.k.min", "Minimum value of k to analyze.");
        public static final OptionID KMAX_ID = new OptionID("kdeos.k.max", "Maximum value of k to analyze.");
        public static final OptionID IDIM_ID = new OptionID("kdeos.idim", "Intrinsic dimensionality of this data set. Use -1 for using the true data dimensionality, but values such as 0-2 often offer better performance.");
        protected Distance<? super O> distance;
        protected KernelDensityFunction kernel;
        protected int kmin;
        protected int kmax;
        protected double scale;
        protected double minBandwidth = 0.0;
        protected int idim = -1;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, Distance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            new ObjectParameter(KERNEL_ID, KernelDensityFunction.class, GaussianKernelDensityFunction.class).grab(config, x -> {
                this.kernel = x;
            });
            IntParameter kminP = (IntParameter)new IntParameter(KMIN_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT);
            kminP.grab(config, x -> {
                this.kmin = x;
            });
            IntParameter kmaxP = (IntParameter)new IntParameter(KMAX_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT);
            kmaxP.grab(config, x -> {
                this.kmax = x;
            });
            if (this.kmin > this.kmax) {
                config.reportError((ParameterException)new WrongParameterValueException((Parameter)kminP, "must be at most", (Parameter)kmaxP, ""));
            }
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(KERNEL_SCALE_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).setDefaultValue((Object)0.25)).grab(config, x -> {
                this.scale = x * (this.kernel != null ? this.kernel.canonicalBandwidth() : 1.0);
            });
            ((DoubleParameter)((DoubleParameter)new DoubleParameter(KERNEL_MIN_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_DOUBLE)).setOptional(true)).grab(config, x -> {
                this.minBandwidth = x;
            });
            new IntParameter(IDIM_ID, 1).grab(config, x -> {
                this.idim = x;
            });
        }

        public KDEOS<O> make() {
            return new KDEOS<O>(this.distance, this.kmin, this.kmax, this.kernel, this.minBandwidth, this.scale, this.idim);
        }
    }
}

