/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.lof;

import elki.Algorithm;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.DoubleDBIDListIter;
import elki.database.ids.KNNList;
import elki.database.query.QueryBuilder;
import elki.database.query.knn.KNNSearcher;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.Relation;
import elki.distance.Distance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.logging.progress.StepProgress;
import elki.math.DoubleMinMax;
import elki.math.MathUtil;
import elki.outlier.OutlierAlgorithm;
import elki.result.outlier.OutlierResult;
import elki.result.outlier.QuotientOutlierScoreMeta;
import elki.utilities.Priority;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;

@Title(value="LOF: Local Outlier Factor")
@Description(value="Algorithm to compute density-based local outlier factors in a database based on the neighborhood size parameter 'k'")
@Reference(authors="Markus M. Breunig, Hans-Peter Kriegel, Raymond Ng, J\u00f6rg Sander", title="LOF: Identifying Density-Based Local Outliers", booktitle="Proc. 2nd ACM SIGMOD Int. Conf. on Management of Data (SIGMOD'00)", url="https://doi.org/10.1145/342009.335388", bibkey="DBLP:conf/sigmod/BreunigKNS00")
@Priority(value=200)
public class LOF<O>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(LOF.class);
    protected Distance<? super O> distance;
    protected int kplus;

    public LOF(int k, Distance<? super O> distance) {
        this.distance = distance;
        this.kplus = k + 1;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{this.distance.getInputTypeRestriction()});
    }

    public OutlierResult run(Relation<O> relation) {
        StepProgress stepprog = LOG.isVerbose() ? new StepProgress("LOF", 3) : null;
        DBIDs ids = relation.getDBIDs();
        LOG.beginStep(stepprog, 1, "Materializing nearest-neighbor sets.");
        KNNSearcher knnq = new QueryBuilder(relation, this.distance).precomputed().kNNByDBID(this.kplus);
        LOG.beginStep(stepprog, 2, "Computing Local Reachability Densities (LRD).");
        WritableDoubleDataStore lrds = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)3);
        this.computeLRDs((KNNSearcher<DBIDRef>)knnq, ids, lrds);
        LOG.beginStep(stepprog, 3, "Computing Local Outlier Factors (LOF).");
        WritableDoubleDataStore lofs = DataStoreUtil.makeDoubleStorage((DBIDs)ids, (int)30);
        DoubleMinMax lofminmax = new DoubleMinMax();
        this.computeLOFScores((KNNSearcher<DBIDRef>)knnq, ids, (DoubleDataStore)lrds, lofs, lofminmax);
        LOG.setCompleted(stepprog);
        MaterializedDoubleRelation scoreResult = new MaterializedDoubleRelation("Local Outlier Factor", ids, (DoubleDataStore)lofs);
        QuotientOutlierScoreMeta scoreMeta = new QuotientOutlierScoreMeta(lofminmax.getMin(), lofminmax.getMax(), 0.0, Double.POSITIVE_INFINITY, 1.0);
        return new OutlierResult(scoreMeta, (DoubleRelation)scoreResult);
    }

    private void computeLRDs(KNNSearcher<DBIDRef> knnq, DBIDs ids, WritableDoubleDataStore lrds) {
        FiniteProgress lrdsProgress = LOG.isVerbose() ? new FiniteProgress("Local Reachability Densities (LRD)", ids.size(), LOG) : null;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            double lrd = this.computeLRD(knnq, iter);
            lrds.putDouble((DBIDRef)iter, lrd);
            LOG.incrementProcessed((AbstractProgress)lrdsProgress);
            iter.advance();
        }
        LOG.ensureCompleted(lrdsProgress);
    }

    protected double computeLRD(KNNSearcher<DBIDRef> knnq, DBIDIter curr) {
        KNNList neighbors = knnq.getKNN((Object)curr, this.kplus);
        double sum = 0.0;
        int count = 0;
        DoubleDBIDListIter neighbor = neighbors.iter();
        while (neighbor.valid()) {
            if (!DBIDUtil.equal((DBIDRef)curr, (DBIDRef)neighbor)) {
                KNNList neighborsNeighbors = knnq.getKNN((Object)neighbor, this.kplus);
                sum += MathUtil.max((double)neighbor.doubleValue(), (double)neighborsNeighbors.getKNNDistance());
                ++count;
            }
            neighbor.advance();
        }
        return sum > 0.0 ? (double)count / sum : Double.POSITIVE_INFINITY;
    }

    private void computeLOFScores(KNNSearcher<DBIDRef> knnq, DBIDs ids, DoubleDataStore lrds, WritableDoubleDataStore lofs, DoubleMinMax lofminmax) {
        FiniteProgress progressLOFs = LOG.isVerbose() ? new FiniteProgress("Local Outlier Factor (LOF) scores", ids.size(), LOG) : null;
        DBIDIter iter = ids.iter();
        while (iter.valid()) {
            double lof = this.computeLOFScore(knnq, (DBIDRef)iter, lrds);
            lofs.putDouble((DBIDRef)iter, lof);
            lofminmax.put(lof);
            LOG.incrementProcessed((AbstractProgress)progressLOFs);
            iter.advance();
        }
        LOG.ensureCompleted(progressLOFs);
    }

    protected double computeLOFScore(KNNSearcher<DBIDRef> knnq, DBIDRef cur, DoubleDataStore lrds) {
        double lrdp = lrds.doubleValue(cur);
        if (Double.isInfinite(lrdp)) {
            return 1.0;
        }
        double sum = 0.0;
        int count = 0;
        KNNList neighbors = knnq.getKNN((Object)cur, this.kplus);
        DoubleDBIDListIter neighbor = neighbors.iter();
        while (neighbor.valid()) {
            if (!DBIDUtil.equal((DBIDRef)cur, (DBIDRef)neighbor)) {
                sum += lrds.doubleValue((DBIDRef)neighbor);
                ++count;
            }
            neighbor.advance();
        }
        return sum / (lrdp * (double)count);
    }

    public static class Par<O>
    implements Parameterizer {
        public static final OptionID K_ID = new OptionID("lof.k", "The number of nearest neighbors (not including the query point) of an object to be considered for computing its LOF score.");
        protected Distance<? super O> distance;
        protected int k = 2;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, Distance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(K_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.k = x;
            });
        }

        public LOF<O> make() {
            return new LOF<O>(this.k, this.distance);
        }
    }
}

