/*
 * Decompiled with CFR 0.152.
 */
package elki.outlier.subspace;

import elki.data.NumberVector;
import elki.data.type.SimpleTypeInformation;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStore;
import elki.database.datastore.DataStoreUtil;
import elki.database.datastore.DoubleDataStore;
import elki.database.datastore.WritableDataStore;
import elki.database.datastore.WritableDoubleDataStore;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.ids.KNNHeap;
import elki.database.ids.ModifiableDBIDs;
import elki.database.query.similarity.SimilarityQuery;
import elki.database.relation.DoubleRelation;
import elki.database.relation.MaterializedDoubleRelation;
import elki.database.relation.MaterializedRelation;
import elki.database.relation.Relation;
import elki.logging.Logging;
import elki.logging.progress.AbstractProgress;
import elki.logging.progress.FiniteProgress;
import elki.math.DoubleMinMax;
import elki.math.Mean;
import elki.math.linearalgebra.Centroid;
import elki.math.linearalgebra.VMath;
import elki.outlier.OutlierAlgorithm;
import elki.result.Metadata;
import elki.result.outlier.BasicOutlierScoreMeta;
import elki.result.outlier.OutlierResult;
import elki.result.textwriter.TextWriteable;
import elki.result.textwriter.TextWriterStream;
import elki.similarity.SharedNearestNeighborSimilarity;
import elki.similarity.Similarity;
import elki.utilities.datastructures.BitsUtil;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.DoubleParameter;
import elki.utilities.optionhandling.parameters.Flag;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;

@Title(value="SOD: Subspace outlier degree")
@Description(value="Outlier Detection in Axis-Parallel Subspaces of High Dimensional Data")
@Reference(authors="Hans-Peter Kriegel, Peer Kr\u00f6ger, Erich Schubert, Arthur Zimek", title="Outlier Detection in Axis-Parallel Subspaces of High Dimensional Data", booktitle="Proc. Pacific-Asia Conf. on Knowledge Discovery and Data Mining (PAKDD 2009)", url="https://doi.org/10.1007/978-3-642-01307-2_86", bibkey="DBLP:conf/pakdd/KriegelKSZ09")
public class SOD<V extends NumberVector>
implements OutlierAlgorithm {
    private static final Logging LOG = Logging.getLogger(SOD.class);
    private int knn;
    private double alpha;
    private Similarity<V> similarityFunction;
    private boolean models;

    public SOD(int knn, double alpha, Similarity<V> similarityFunction, boolean models) {
        this.knn = knn;
        this.alpha = alpha;
        this.similarityFunction = similarityFunction;
        this.models = models;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public OutlierResult run(Relation<V> relation) {
        SimilarityQuery snnInstance = this.similarityFunction.instantiate(relation);
        FiniteProgress progress = LOG.isVerbose() ? new FiniteProgress("Assigning Subspace Outlier Degree", relation.size(), LOG) : null;
        WritableDoubleDataStore sod_scores = DataStoreUtil.makeDoubleStorage((DBIDs)relation.getDBIDs(), (int)4);
        WritableDataStore sod_models = this.models ? DataStoreUtil.makeStorage((DBIDs)relation.getDBIDs(), (int)4, SODModel.class) : null;
        DoubleMinMax minmax = new DoubleMinMax();
        DBIDIter iter = relation.iterDBIDs();
        while (iter.valid()) {
            double[] center;
            DBIDs neighborhood = this.getNearestNeighbors(relation, snnInstance, (DBIDRef)iter);
            long[] weightVector = null;
            double sod = 0.0;
            if (neighborhood.size() > 0) {
                center = Centroid.make(relation, (DBIDs)neighborhood).getArrayRef();
                double[] variances = SOD.computePerDimensionVariances(relation, center, neighborhood);
                double expectationOfVariance = Mean.of((double[])variances);
                weightVector = BitsUtil.zero((int)variances.length);
                for (int d = 0; d < variances.length; ++d) {
                    if (!(variances[d] < this.alpha * expectationOfVariance)) continue;
                    BitsUtil.setI((long[])weightVector, (int)d);
                }
                sod = this.subspaceOutlierDegree((NumberVector)relation.get((DBIDRef)iter), center, weightVector);
            } else {
                center = ((NumberVector)relation.get((DBIDRef)iter)).toArray();
            }
            if (sod_models != null) {
                sod_models.put((DBIDRef)iter, (Object)new SODModel(center, weightVector));
            }
            sod_scores.putDouble((DBIDRef)iter, sod);
            minmax.put(sod);
            LOG.incrementProcessed((AbstractProgress)progress);
            iter.advance();
        }
        LOG.ensureCompleted(progress);
        BasicOutlierScoreMeta meta = new BasicOutlierScoreMeta(minmax.getMin(), minmax.getMax());
        OutlierResult sodResult = new OutlierResult(meta, (DoubleRelation)new MaterializedDoubleRelation("Subspace Outlier Degree", relation.getDBIDs(), (DoubleDataStore)sod_scores));
        if (sod_models != null) {
            Metadata.hierarchyOf((Object)sodResult).addChild((Object)new MaterializedRelation("Subspace Outlier Model", new SimpleTypeInformation(SODModel.class), relation.getDBIDs(), (DataStore)sod_models));
        }
        return sodResult;
    }

    private DBIDs getNearestNeighbors(Relation<V> relation, SimilarityQuery<V> simQ, DBIDRef queryObject) {
        KNNHeap nearestNeighbors = DBIDUtil.newHeap((int)this.knn);
        DBIDIter iter = relation.iterDBIDs();
        while (iter.valid()) {
            if (!DBIDUtil.equal((DBIDRef)iter, (DBIDRef)queryObject)) {
                nearestNeighbors.insert(-simQ.similarity(queryObject, (DBIDRef)iter), (DBIDRef)iter);
            }
            iter.advance();
        }
        return nearestNeighbors.unorderedIterator().addTo((ModifiableDBIDs)DBIDUtil.newArray((int)nearestNeighbors.size()));
    }

    private static double[] computePerDimensionVariances(Relation<? extends NumberVector> relation, double[] center, DBIDs neighborhood) {
        int dim = center.length;
        double[] variances = new double[dim];
        DBIDIter iter = neighborhood.iter();
        while (iter.valid()) {
            NumberVector databaseObject = (NumberVector)relation.get((DBIDRef)iter);
            int d = 0;
            while (d < dim) {
                double deviation = databaseObject.doubleValue(d) - center[d];
                int n = d++;
                variances[n] = variances[n] + deviation * deviation;
            }
            iter.advance();
        }
        return VMath.timesEquals((double[])variances, (double)(1.0 / (double)neighborhood.size()));
    }

    private double subspaceOutlierDegree(V queryObject, double[] center, long[] weightVector) {
        double sqrDist = 0.0;
        int card = 0;
        int d = BitsUtil.nextSetBit((long[])weightVector, (int)0);
        while (d >= 0) {
            double delta = queryObject.doubleValue(d) - center[d];
            sqrDist += delta * delta;
            ++card;
            d = BitsUtil.nextSetBit((long[])weightVector, (int)(d + 1));
        }
        return sqrDist > 0.0 ? Math.sqrt(sqrDist) / (double)card : 0.0;
    }

    public static class Par<V extends NumberVector>
    implements Parameterizer {
        public static final OptionID KNN_ID = new OptionID("sod.knn", "The number of most snn-similar objects to use as reference set for learning the subspace properties.");
        public static final OptionID ALPHA_ID = new OptionID("sod.alpha", "The multiplier for the discriminance value for discerning small from large variances.");
        public static final OptionID SIM_ID = new OptionID("sod.similarity", "The similarity function used for the neighborhood set.");
        public static final OptionID MODELS_ID = new OptionID("sod.models", "Report the models computed by SOD (default: report only scores).");
        private int knn = 1;
        private double alpha = 1.1;
        private Similarity<V> similarityFunction;
        private boolean models = false;

        public void configure(Parameterization config) {
            new ObjectParameter(SIM_ID, Similarity.class, SharedNearestNeighborSimilarity.class).grab(config, x -> {
                this.similarityFunction = x;
            });
            ((IntParameter)new IntParameter(KNN_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.knn = x;
            });
            ((DoubleParameter)new DoubleParameter(ALPHA_ID, 1.1).addConstraint((ParameterConstraint)CommonConstraints.GREATER_THAN_ZERO_DOUBLE)).grab(config, x -> {
                this.alpha = x;
            });
            new Flag(MODELS_ID).grab(config, x -> {
                this.models = x;
            });
        }

        public SOD<V> make() {
            return new SOD<V>(this.knn, this.alpha, this.similarityFunction, this.models);
        }
    }

    public static class SODModel
    implements TextWriteable {
        private double[] center;
        private long[] weightVector;

        public SODModel(double[] center, long[] weightVector) {
            this.center = center;
            this.weightVector = weightVector;
        }

        public void writeToText(TextWriterStream out, String label) {
            out.commentPrintLn((CharSequence)(this.getClass().getSimpleName() + ":"));
            out.commentPrintLn((CharSequence)("relevant attributes (starting with 0): " + BitsUtil.toString((long[])this.weightVector, (String)", ", (int)0)));
            out.commentPrintLn((CharSequence)("center of neighborhood: " + this.center.toString()));
            out.commentPrintSeparator();
        }
    }
}

