/*
 * This file is part of ELKI:
 * Environment for Developing KDD-Applications Supported by Index-Structures
 *
 * Copyright (C) 2022
 * ELKI Development Team
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package elki.algorithm.statistics;

import elki.Algorithm;
import elki.data.NumberVector;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.Database;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.math.MathUtil;
import elki.math.scales.LinearScale;
import elki.result.ResultUtil;
import elki.result.ScalesResult;
import elki.utilities.documentation.Description;

/**
 * Pseudo "algorithm" that computes the global min/max for a relation across all
 * attributes.
 *
 * FIXME: this should become part of relation metadata.
 *
 * @author Erich Schubert
 * @since 0.5.0
 */
@Description("Setup a scaling so that all dimensions have the same value range.")
public class AddUniformScale implements Algorithm {
  /**
   * Constructor.
   */
  public AddUniformScale() {
    super();
  }

  /**
   * Run the algorithm on all vector relations of a database.
   * 
   * @param database Database
   * @return Empty (scales are attached to the relations)
   */
  public Void run(Database database) {
    for(Relation<?> rel : database.getRelations()) {
      if(TypeUtil.NUMBER_VECTOR_FIELD.isAssignableFromType(rel.getDataTypeInformation())) {
        @SuppressWarnings("unchecked")
        Relation<? extends NumberVector> vrel = (Relation<? extends NumberVector>) rel;
        ResultUtil.addChildResult(rel, run(vrel));
      }
    }
    return null;
  }

  /**
   * Add scales to a single vector relation.
   *
   * @param rel Relation
   * @return Scales
   */
  private ScalesResult run(Relation<? extends NumberVector> rel) {
    double[][] mms = RelationUtil.computeMinMax(rel);
    int dim = mms[0].length;
    double delta = 0.;
    for(int d = 0; d < dim; d++) {
      double del = mms[1][d] - mms[0][d];
      delta = del > delta ? del : delta;
    }
    if(delta < Double.MIN_NORMAL) {
      delta = 1.;
    }
    int log10res = (int) Math.ceil(Math.log10(delta / (LinearScale.MAXTICKS - 1)));
    double res = MathUtil.powi(10, log10res);
    double target = Math.ceil(delta / res) * res; // Target width
    LinearScale[] scales = new LinearScale[dim];
    for(int d = 0; d < dim; d++) {
      double mid = (mms[0][d] + mms[1][d] - target) * .5;
      double min = Math.floor(mid / res) * res;
      double max = Math.ceil((mid + target) / res) * res;
      scales[d] = new LinearScale(min, max);
    }
    return new ScalesResult(scales);
  }

  @Override
  public TypeInformation[] getInputTypeRestriction() {
    return TypeUtil.array(TypeUtil.NUMBER_VECTOR_FIELD);
  }
}
