/*
 * This file is part of ELKI:
 * Environment for Developing KDD-Applications Supported by Index-Structures
 *
 * Copyright (C) 2022
 * ELKI Development Team
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 */
package elki.evaluation.index;

import java.util.ArrayList;
import java.util.Collection;

import elki.database.Database;
import elki.evaluation.Evaluator;
import elki.index.tree.IndexTree;
import elki.result.CollectionResult;
import elki.result.Metadata;
import elki.result.ResultUtil;
import elki.utilities.pairs.Pair;

/**
 * Simple index analytics, which includes the toString() dump of index
 * information.
 * 
 * @author Erich Schubert
 * @since 0.4.0
 * 
 * @opt nodefillcolor LemonChiffon
 * @navhas - create - IndexMetaResult
 */
public class IndexStatistics implements Evaluator {
  /**
   * Constructor.
   */
  public IndexStatistics() {
    super();
  }

  @Override
  public void processNewResult(Object newResult) {
    Database db = ResultUtil.findDatabase(newResult);
    Collection<String> header = null;
    final ArrayList<IndexTree<?, ?>> indexes = ResultUtil.filterResults(newResult, IndexTree.class);
    if(indexes == null || indexes.isEmpty()) {
      return;
    }
    for(IndexTree<?, ?> index : indexes) {
      header = new ArrayList<>();
      header.add(index.toString());
    }
    Collection<Pair<String, String>> col = new ArrayList<>();
    IndexMetaResult analysis = new IndexMetaResult(col, header);
    Metadata.hierarchyOf(db).addChild(analysis);
  }

  /**
   * Result class.
   * 
   * @author Erich Schubert
   */
  public static class IndexMetaResult extends CollectionResult<Pair<String, String>> {
    /**
     * Constructor.
     * 
     * @param col key value pairs
     * @param header header
     */
    public IndexMetaResult(Collection<Pair<String, String>> col, Collection<String> header) {
      super(col, header);
      Metadata.of(this).setLongName("Index Statistics");
    }
  }
}
