/*
 * Decompiled with CFR 0.152.
 */
package elki.algorithm;

import elki.Algorithm;
import elki.data.NumberVector;
import elki.data.model.CorrelationAnalysisSolution;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.ids.DBIDUtil;
import elki.database.ids.DBIDs;
import elki.database.query.QueryBuilder;
import elki.database.relation.Relation;
import elki.database.relation.RelationUtil;
import elki.distance.NumberVectorDistance;
import elki.distance.minkowski.EuclideanDistance;
import elki.logging.Logging;
import elki.math.linearalgebra.Centroid;
import elki.math.linearalgebra.LinearEquationSystem;
import elki.math.linearalgebra.VMath;
import elki.math.linearalgebra.pca.PCAFilteredResult;
import elki.math.linearalgebra.pca.PCAResult;
import elki.math.linearalgebra.pca.PCARunner;
import elki.math.linearalgebra.pca.filter.EigenPairFilter;
import elki.math.linearalgebra.pca.filter.PercentageEigenPairFilter;
import elki.utilities.Priority;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.io.FormatUtil;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.Flag;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import elki.utilities.random.RandomFactory;
import java.text.NumberFormat;
import java.util.Locale;

@Title(value="Dependency Derivator: Deriving numerical inter-dependencies on data")
@Description(value="Derives an equality-system describing dependencies between attributes in a correlation-cluster")
@Reference(authors="Elke Achtert, Christian B\u00f6hm, Hans-Peter Kriegel, Peer Kr\u00f6ger, Arthur Zimek", title="Deriving Quantitative Dependencies for Correlation Clusters", booktitle="Proc. 12th Int. Conf. on Knowledge Discovery and Data Mining (KDD '06)", url="https://doi.org/10.1145/1150402.1150408", bibkey="DBLP:conf/kdd/AchtertBKKZ06")
@Priority(value=-5)
public class DependencyDerivator<V extends NumberVector>
implements Algorithm {
    private static final Logging LOG = Logging.getLogger(DependencyDerivator.class);
    private NumberVectorDistance<? super V> distance;
    private final int sampleSize;
    private final PCARunner pca;
    private final EigenPairFilter filter;
    private final NumberFormat nf;
    private final boolean randomsample;

    public DependencyDerivator(NumberVectorDistance<? super V> distance, NumberFormat nf, PCARunner pca, EigenPairFilter filter, int sampleSize, boolean randomsample) {
        this.distance = distance;
        this.nf = nf;
        this.pca = pca;
        this.filter = filter;
        this.sampleSize = sampleSize;
        this.randomsample = randomsample;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public CorrelationAnalysisSolution run(Relation<V> relation) {
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)"retrieving database objects...");
        }
        Centroid centroid = Centroid.make(relation, (DBIDs)relation.getDBIDs());
        NumberVector.Factory factory = RelationUtil.getNumberVectorFactory(relation);
        NumberVector centroidDV = factory.newNumberVector(centroid.getArrayRef());
        Object ids = this.sampleSize == 0 ? relation.getDBIDs() : (this.randomsample ? DBIDUtil.randomSample((DBIDs)relation.getDBIDs(), (int)this.sampleSize, (RandomFactory)RandomFactory.DEFAULT) : new QueryBuilder(relation, this.distance).cheapOnly().kNNByObject(this.sampleSize).getKNN((Object)centroidDV, this.sampleSize));
        return this.generateModel(relation, (DBIDs)ids, centroid.getArrayRef());
    }

    public CorrelationAnalysisSolution generateModel(Relation<V> db, DBIDs ids) {
        return this.generateModel(db, ids, Centroid.make(db, (DBIDs)ids).getArrayRef());
    }

    public CorrelationAnalysisSolution generateModel(Relation<V> relation, DBIDs ids, double[] centroid) {
        if (LOG.isDebuggingFine()) {
            LOG.debugFine((CharSequence)"PCA...");
        }
        PCAResult epairs = this.pca.processIds(ids, relation);
        int numstrong = this.filter.filter(epairs.getEigenvalues());
        PCAFilteredResult pcares = new PCAFilteredResult(epairs.getEigenPairs(), numstrong, 1.0, 0.0);
        double[][] transposedWeakEigenvectors = pcares.getWeakEigenvectors();
        double[][] transposedStrongEigenvectors = pcares.getStrongEigenvectors();
        if (transposedWeakEigenvectors.length == 0) {
            return new CorrelationAnalysisSolution(null, relation, VMath.transpose((double[][])transposedStrongEigenvectors), new double[0][], pcares.similarityMatrix(), centroid);
        }
        if (LOG.isDebugging()) {
            StringBuilder msg = new StringBuilder(1000);
            FormatUtil.formatTo((StringBuilder)msg.append("Strong Eigenvectors:\n"), (double[][])transposedStrongEigenvectors, (String)" [", (String)"]\n", (String)", ", (NumberFormat)this.nf);
            FormatUtil.formatTo((StringBuilder)msg.append("\nWeak Eigenvectors:\n"), (double[][])transposedWeakEigenvectors, (String)" [", (String)"]\n", (String)", ", (NumberFormat)this.nf);
            FormatUtil.formatTo((StringBuilder)msg.append("\nEigenvalues:\n"), (double[])pcares.getEigenvalues(), (String)", ", (NumberFormat)this.nf);
            LOG.debugFine((CharSequence)msg.toString());
        }
        double[] b = VMath.times((double[][])transposedWeakEigenvectors, (double[])centroid);
        if (LOG.isDebugging()) {
            StringBuilder msg = new StringBuilder(1000);
            FormatUtil.formatTo((StringBuilder)msg.append("Centroid:\n"), (double[])centroid, (String)", ", (NumberFormat)this.nf);
            FormatUtil.formatTo((StringBuilder)msg.append("\ntEV * Centroid\n"), (double[])b, (String)", ", (NumberFormat)this.nf);
            LOG.debugFine((CharSequence)msg.toString());
        }
        double[][] gaussJordan = new double[transposedWeakEigenvectors.length][transposedWeakEigenvectors[0].length + 1];
        VMath.setMatrix((double[][])gaussJordan, (int)0, (int)transposedWeakEigenvectors.length, (int)0, (int)transposedWeakEigenvectors[0].length, (double[][])transposedWeakEigenvectors);
        VMath.setCol((double[][])gaussJordan, (int)transposedWeakEigenvectors[0].length, (double[])b);
        if (LOG.isDebuggingFiner()) {
            LOG.debugFiner((CharSequence)("Gauss-Jordan-Elimination of " + FormatUtil.format((double[][])gaussJordan, (String)" [", (String)"]\n", (String)", ", (NumberFormat)this.nf)));
        }
        LinearEquationSystem lq = new LinearEquationSystem(VMath.copy((double[][])transposedWeakEigenvectors), b);
        lq.solveByTotalPivotSearch();
        CorrelationAnalysisSolution sol = new CorrelationAnalysisSolution(lq, relation, VMath.transpose((double[][])transposedStrongEigenvectors), VMath.transpose((double[][])transposedWeakEigenvectors), pcares.similarityMatrix(), centroid);
        if (LOG.isDebuggingFine()) {
            LOG.debugFine((CharSequence)("Solution:\n" + "Standard deviation " + sol.getStandardDeviation() + lq.equationsToString(this.nf.getMaximumFractionDigits())));
        }
        return sol;
    }

    public static class Par<V extends NumberVector>
    implements Parameterizer {
        public static final OptionID DEPENDENCY_DERIVATOR_RANDOM_SAMPLE_ID = new OptionID("derivator.randomSample", "Flag to use random sample (use knn query around centroid, if flag is not set).");
        public static final OptionID OUTPUT_ACCURACY_ID = new OptionID("derivator.accuracy", "Threshold for output accuracy fraction digits.");
        public static final OptionID SAMPLE_SIZE_ID = new OptionID("derivator.sampleSize", "Threshold for the size of the random sample to use. Default value is size of the complete dataset.");
        protected NumberVectorDistance<? super V> distance;
        protected int outputAccuracy = 0;
        protected int sampleSize = 0;
        protected boolean randomSample = false;
        protected PCARunner pca = null;
        protected EigenPairFilter filter;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, NumberVectorDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(OUTPUT_ACCURACY_ID, 4).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ZERO_INT)).grab(config, x -> {
                this.outputAccuracy = x;
            });
            ((IntParameter)((IntParameter)new IntParameter(SAMPLE_SIZE_ID).setOptional(true)).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.sampleSize = x;
            });
            new Flag(DEPENDENCY_DERIVATOR_RANDOM_SAMPLE_ID).grab(config, x -> {
                this.randomSample = x;
            });
            new ObjectParameter(PCARunner.Par.PCARUNNER_ID, PCARunner.class, PCARunner.class).grab(config, x -> {
                this.pca = x;
            });
            new ObjectParameter(EigenPairFilter.PCA_EIGENPAIR_FILTER, EigenPairFilter.class, PercentageEigenPairFilter.class).grab(config, x -> {
                this.filter = x;
            });
        }

        public DependencyDerivator<V> make() {
            NumberFormat nf = NumberFormat.getInstance(Locale.US);
            nf.setMaximumFractionDigits(this.outputAccuracy);
            nf.setMinimumFractionDigits(this.outputAccuracy);
            return new DependencyDerivator<V>(this.distance, nf, this.pca, this.filter, this.sampleSize, this.randomSample);
        }
    }
}

