/*
 * Decompiled with CFR 0.152.
 */
package elki.clustering.optics;

import elki.Algorithm;
import elki.algorithm.KNNJoin;
import elki.clustering.optics.ClusterOrder;
import elki.clustering.optics.OPTICSTypeAlgorithm;
import elki.data.NumberVector;
import elki.data.spatial.SpatialComparable;
import elki.data.type.TypeInformation;
import elki.data.type.TypeUtil;
import elki.database.datastore.DataStore;
import elki.database.datastore.WritableDataStore;
import elki.database.ids.DBID;
import elki.database.ids.DBIDIter;
import elki.database.ids.DBIDRef;
import elki.database.ids.DBIDs;
import elki.database.ids.KNNList;
import elki.database.relation.Relation;
import elki.distance.SpatialPrimitiveDistance;
import elki.distance.minkowski.EuclideanDistance;
import elki.index.tree.IndexTreePath;
import elki.index.tree.LeafEntry;
import elki.index.tree.spatial.SpatialDirectoryEntry;
import elki.index.tree.spatial.rstarvariants.AbstractRStarTree;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluDirectoryEntry;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluEntry;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluNode;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluTree;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluTreeFactory;
import elki.index.tree.spatial.rstarvariants.deliclu.DeLiCluTreeIndex;
import elki.logging.Logging;
import elki.logging.progress.FiniteProgress;
import elki.math.MathUtil;
import elki.result.Metadata;
import elki.utilities.ClassGenericsUtil;
import elki.utilities.datastructures.heap.UpdatableHeap;
import elki.utilities.documentation.Description;
import elki.utilities.documentation.Reference;
import elki.utilities.documentation.Title;
import elki.utilities.exceptions.AbortException;
import elki.utilities.optionhandling.OptionID;
import elki.utilities.optionhandling.Parameterizer;
import elki.utilities.optionhandling.constraints.CommonConstraints;
import elki.utilities.optionhandling.constraints.ParameterConstraint;
import elki.utilities.optionhandling.parameterization.Parameterization;
import elki.utilities.optionhandling.parameters.IntParameter;
import elki.utilities.optionhandling.parameters.ObjectParameter;
import it.unimi.dsi.fastutil.ints.IntSet;
import java.util.ArrayList;
import java.util.List;

@Title(value="DeliClu: Density-Based Hierarchical Clustering")
@Description(value="Hierachical algorithm to find density-connected sets in a database based on the parameter 'minpts'.")
@Reference(authors="Elke Achtert, Christian B\u00f6hm, Peer Kr\u00f6ger", title="DeLiClu: Boosting Robustness, Completeness, Usability, and Efficiency of Hierarchical Clustering by a Closest Pair Ranking", booktitle="Proc. 10th Pacific-Asia Conf. on Knowledge Discovery and Data Mining (PAKDD 2006)", url="https://doi.org/10.1007/11731139_16", bibkey="DBLP:conf/pakdd/AchtertBK06")
public class DeLiClu<V extends NumberVector>
implements OPTICSTypeAlgorithm {
    private static final Logging LOG = Logging.getLogger(DeLiClu.class);
    protected SpatialPrimitiveDistance<? super V> distance;
    protected DeLiCluTreeFactory<? super V> indexer;
    private UpdatableHeap<SpatialObjectPair> heap;
    private int minpts;

    public DeLiClu(DeLiCluTreeFactory<? super V> indexer, SpatialPrimitiveDistance<? super V> distance, int minpts) {
        this.distance = distance;
        this.indexer = indexer;
        this.minpts = minpts;
    }

    public TypeInformation[] getInputTypeRestriction() {
        return TypeUtil.array((TypeInformation[])new TypeInformation[]{TypeUtil.NUMBER_VECTOR_FIELD});
    }

    public ClusterOrder run(Relation<V> relation) {
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)"Building DeLiClu index");
        }
        DeLiCluTreeIndex index = this.indexer.instantiate(relation);
        index.initialize();
        if (LOG.isVerbose()) {
            LOG.verbose((CharSequence)"Performing kNN join");
        }
        WritableDataStore<KNNList> knns = new KNNJoin(this.distance, this.minpts).run((AbstractRStarTree<?, ?, ?>)index, relation.getDBIDs());
        DBIDs ids = relation.getDBIDs();
        int size = ids.size();
        FiniteProgress progress = LOG.isVerbose() ? new FiniteProgress("DeLiClu", size, LOG) : null;
        ClusterOrder clusterOrder = new ClusterOrder(ids);
        Metadata.of((Object)clusterOrder).setLongName("DeLiClu Cluster Order");
        this.heap = new UpdatableHeap();
        DBIDIter startID = ids.iter();
        clusterOrder.add((DBIDRef)startID, Double.POSITIVE_INFINITY, null);
        int numHandled = 1;
        index.setHandled((DBIDRef)startID, (NumberVector)relation.get((DBIDRef)startID));
        DeLiCluEntry rootEntry = (DeLiCluEntry)index.getRootEntry();
        SpatialObjectPair spatialObjectPair = new SpatialObjectPair(0.0, rootEntry, rootEntry, true);
        this.heap.add((Object)spatialObjectPair);
        while (numHandled < size) {
            if (this.heap.isEmpty()) {
                throw new AbortException("DeLiClu heap was empty when it shouldn't have been.");
            }
            SpatialObjectPair dataPair = (SpatialObjectPair)this.heap.poll();
            if (dataPair.isExpandable) {
                this.expandNodes((DeLiCluTree)index, dataPair, (DataStore<KNNList>)knns);
                continue;
            }
            LeafEntry e1 = (LeafEntry)dataPair.entry1;
            LeafEntry e2 = (LeafEntry)dataPair.entry2;
            DBID e1id = e1.getDBID();
            IndexTreePath path = index.setHandled((DBIDRef)e1id, (NumberVector)relation.get((DBIDRef)e1id));
            if (path == null) {
                throw new RuntimeException("snh: parent(" + e1id + ") = null!!!");
            }
            clusterOrder.add((DBIDRef)e1id, dataPair.distance, (DBIDRef)e2.getDBID());
            ++numHandled;
            this.reinsertExpanded((DeLiCluTree)index, (IndexTreePath<DeLiCluEntry>)path, (DataStore<KNNList>)knns);
            if (progress == null) continue;
            progress.setProcessed(numHandled, LOG);
        }
        LOG.ensureCompleted(progress);
        return clusterOrder;
    }

    private void expandNodes(DeLiCluTree index, SpatialObjectPair nodePair, DataStore<KNNList> knns) {
        DeLiCluNode node1 = (DeLiCluNode)index.getNode(((SpatialDirectoryEntry)nodePair.entry1).getPageID());
        DeLiCluNode node2 = (DeLiCluNode)index.getNode(((SpatialDirectoryEntry)nodePair.entry2).getPageID());
        if (node1.isLeaf()) {
            this.expandLeafNodes(node1, node2, knns);
        } else {
            this.expandDirNodes(node1, node2);
        }
        index.setExpanded(nodePair.entry2, nodePair.entry1);
    }

    private void expandDirNodes(DeLiCluNode node1, DeLiCluNode node2) {
        if (LOG.isDebuggingFinest()) {
            LOG.debugFinest((CharSequence)("ExpandDirNodes: " + node1.getPageID() + " + " + node2.getPageID()));
        }
        int numEntries_1 = node1.getNumEntries();
        int numEntries_2 = node2.getNumEntries();
        SpatialPrimitiveDistance<? super V> distFunction = this.distance;
        for (int i = 0; i < numEntries_1; ++i) {
            DeLiCluEntry entry1 = (DeLiCluEntry)node1.getEntry(i);
            if (!entry1.hasUnhandled()) continue;
            for (int j = 0; j < numEntries_2; ++j) {
                DeLiCluEntry entry2 = (DeLiCluEntry)node2.getEntry(j);
                if (!entry2.hasHandled()) continue;
                this.heap.add((Object)new SpatialObjectPair(distFunction.minDist((SpatialComparable)entry1, (SpatialComparable)entry2), entry1, entry2, true));
            }
        }
    }

    private void expandLeafNodes(DeLiCluNode node1, DeLiCluNode node2, DataStore<KNNList> knns) {
        if (LOG.isDebuggingFinest()) {
            LOG.debugFinest((CharSequence)("ExpandLeafNodes: " + node1.getPageID() + " + " + node2.getPageID()));
        }
        int numEntries_1 = node1.getNumEntries();
        int numEntries_2 = node2.getNumEntries();
        SpatialPrimitiveDistance<? super V> distFunction = this.distance;
        for (int i = 0; i < numEntries_1; ++i) {
            DeLiCluEntry entry1 = (DeLiCluEntry)node1.getEntry(i);
            if (!entry1.hasUnhandled()) continue;
            for (int j = 0; j < numEntries_2; ++j) {
                DeLiCluEntry entry2 = (DeLiCluEntry)node2.getEntry(j);
                if (!entry2.hasHandled()) continue;
                double distance = distFunction.minDist((SpatialComparable)entry1, (SpatialComparable)entry2);
                double reach = MathUtil.max((double)distance, (double)((KNNList)knns.get((DBIDRef)((LeafEntry)entry2).getDBID())).getKNNDistance());
                this.heap.add((Object)new SpatialObjectPair(reach, entry1, entry2, false));
            }
        }
    }

    private void reinsertExpanded(DeLiCluTree index, IndexTreePath<DeLiCluEntry> path, DataStore<KNNList> knns) {
        int l = 0;
        for (IndexTreePath it = path; it != null; it = it.getParentPath()) {
            ++l;
        }
        ArrayList<IndexTreePath<DeLiCluEntry>> p = new ArrayList<IndexTreePath<DeLiCluEntry>>(l - 1);
        IndexTreePath it = path;
        while (it.getParentPath() != null) {
            p.add(it);
            it = it.getParentPath();
        }
        assert (p.size() == l - 1);
        DeLiCluEntry rootEntry = (DeLiCluEntry)it.getEntry();
        this.reinsertExpanded(index, p, l - 2, rootEntry, knns);
    }

    private void reinsertExpanded(DeLiCluTree index, List<IndexTreePath<DeLiCluEntry>> path, int pos, DeLiCluEntry parentEntry, DataStore<KNNList> knns) {
        DeLiCluNode parentNode = (DeLiCluNode)index.getNode((Object)parentEntry);
        DeLiCluEntry entry2 = (DeLiCluEntry)path.get(pos).getEntry();
        SpatialPrimitiveDistance<? super V> distFunction = this.distance;
        if (entry2 instanceof LeafEntry) {
            assert (pos == 0);
            for (int i = 0; i < parentNode.getNumEntries(); ++i) {
                DeLiCluEntry entry1 = (DeLiCluEntry)parentNode.getEntry(i);
                if (entry1.hasHandled()) continue;
                double distance = distFunction.minDist((SpatialComparable)entry1, (SpatialComparable)entry2);
                double reach = MathUtil.max((double)distance, (double)((KNNList)knns.get((DBIDRef)((LeafEntry)entry2).getDBID())).getKNNDistance());
                this.heap.add((Object)new SpatialObjectPair(reach, entry1, entry2, false));
            }
            return;
        }
        IntSet expanded = index.getExpanded(entry2);
        for (int i = 0; i < parentNode.getNumEntries(); ++i) {
            DeLiCluDirectoryEntry entry1 = (DeLiCluDirectoryEntry)parentNode.getEntry(i);
            if (!expanded.contains(entry1.getPageID())) {
                double distance = distFunction.minDist((SpatialComparable)entry1, (SpatialComparable)entry2);
                this.heap.add((Object)new SpatialObjectPair(distance, (DeLiCluEntry)entry1, entry2, true));
                continue;
            }
            this.reinsertExpanded(index, path, pos - 1, (DeLiCluEntry)entry1, knns);
        }
    }

    public int getMinPts() {
        return this.minpts;
    }

    public static class Par<V extends NumberVector>
    implements Parameterizer {
        public static final OptionID MINPTS_ID = new OptionID("deliclu.minpts", "Threshold for minimum number of points within a cluster.");
        protected int minpts = 0;
        protected DeLiCluTreeFactory<? super V> indexer;
        protected SpatialPrimitiveDistance<? super V> distance;

        public void configure(Parameterization config) {
            new ObjectParameter(Algorithm.Utils.DISTANCE_FUNCTION_ID, SpatialPrimitiveDistance.class, EuclideanDistance.class).grab(config, x -> {
                this.distance = x;
            });
            ((IntParameter)new IntParameter(MINPTS_ID).addConstraint((ParameterConstraint)CommonConstraints.GREATER_EQUAL_ONE_INT)).grab(config, x -> {
                this.minpts = x;
            });
            Class clz = ClassGenericsUtil.uglyCastIntoSubclass(DeLiCluTreeFactory.class);
            this.indexer = (DeLiCluTreeFactory)config.tryInstantiate(clz);
        }

        public DeLiClu<V> make() {
            return new DeLiClu<V>(this.indexer, this.distance, this.minpts);
        }
    }

    public static class SpatialObjectPair
    implements Comparable<SpatialObjectPair> {
        DeLiCluEntry entry1;
        DeLiCluEntry entry2;
        boolean isExpandable;
        double distance;

        public SpatialObjectPair(double distance, DeLiCluEntry entry1, DeLiCluEntry entry2, boolean isExpandable) {
            this.distance = distance;
            this.entry1 = entry1;
            this.entry2 = entry2;
            this.isExpandable = isExpandable;
        }

        @Override
        public int compareTo(SpatialObjectPair other) {
            return Double.compare(this.distance, other.distance);
        }

        public String toString() {
            return !this.isExpandable ? this.entry1 + " - " + this.entry2 : "n_" + this.entry1 + " - n_" + this.entry2;
        }

        public boolean equals(Object obj) {
            if (!SpatialObjectPair.class.isInstance(obj)) {
                return false;
            }
            SpatialObjectPair other = (SpatialObjectPair)obj;
            return this.entry1.equals(other.entry1) && (!this.isExpandable || this.entry2.equals(other.entry2));
        }

        public int hashCode() {
            return !this.isExpandable ? this.entry1.hashCode() : (int)(2654435761L * (long)this.entry1.hashCode() + (long)this.entry2.hashCode());
        }
    }
}

