package io.github.embeddedkafka.ops

import java.net.InetSocketAddress
import java.nio.file.Path

import io.github.embeddedkafka.{EmbeddedKafkaConfig, EmbeddedServer, EmbeddedZ}
import org.apache.zookeeper.server.{ServerCnxnFactory, ZooKeeperServer}

/**
  * Trait for ZooKeeper-related actions.
  */
trait ZooKeeperOps {
  private[embeddedkafka] def startZooKeeper(
      zooKeeperPort: Int,
      zkLogsDir: Path
  ): ServerCnxnFactory = {
    val tickTime = 2000

    val zkServer = new ZooKeeperServer(
      zkLogsDir.toFile,
      zkLogsDir.toFile,
      tickTime
    )

    val factory = ServerCnxnFactory.createFactory
    factory.configure(new InetSocketAddress("localhost", zooKeeperPort), 1024)
    factory.startup(zkServer)
    factory
  }
}

/**
  * [[ZooKeeperOps]] extension relying on `RunningServersOps` for
  * keeping track of running [[EmbeddedZ]] instances.
  */
trait RunningZooKeeperOps {
  this: ZooKeeperOps with RunningServersOps =>

  /**
    * Starts a Zookeeper instance in memory, storing logs in a specific location.
    *
    * @param zkLogsDir the path for the Zookeeper logs
    * @param config    an implicit [[EmbeddedKafkaConfig]]
    * @return          an [[EmbeddedZ]] server
    */
  def startZooKeeper(
      zkLogsDir: Path
  )(implicit config: EmbeddedKafkaConfig): EmbeddedZ = {
    val factory =
      EmbeddedZ(startZooKeeper(config.zooKeeperPort, zkLogsDir), zkLogsDir)
    runningServers.add(factory)
    factory
  }

  /**
    * Stops all in memory Zookeeper instances, preserving the logs directories.
    */
  def stopZooKeeper(): Unit =
    runningServers.stopAndRemove(isEmbeddedZ, clearLogs = false)

  private def isEmbeddedZ(server: EmbeddedServer): Boolean =
    server.isInstanceOf[EmbeddedZ]

  private[embeddedkafka] def zookeeperPort(zk: EmbeddedZ): Int =
    zookeeperPort(zk.factory)
  private def zookeeperPort(fac: ServerCnxnFactory): Int =
    fac.getLocalPort
}
