package com.videogo.wificonfig;

import android.content.Context;

import com.ezviz.sdk.configwifi.EZWiFiConfigManager;
import com.ezviz.sdk.configwifi.ap.ApConfigParam;
import com.ezviz.sdk.configwifi.common.ConfigParamAbstract;
import com.ezviz.sdk.configwifi.common.EZConfigWifiCallback;
import com.ezviz.sdk.configwifi.EZConfigWifiErrorEnum;
import com.ezviz.sdk.configwifi.EZConfigWifiInfoEnum;
import com.ezviz.sdk.configwifi.finder.DeviceFindCallback;
import com.ezviz.sdk.configwifi.finder.DeviceFindParam;
import com.ezviz.sdk.configwifi.mixedconfig.MixedConfigParam;
import com.ezviz.sdk.configwifi.mixedconfig.MixedConfigParamOld;
import com.hikvision.wifi.configuration.DeviceDiscoveryListener;
import com.videogo.openapi.EZConstants;
import com.videogo.openapi.EZOpenSDKListener;
import com.videogo.openapi.BaseAPI;
import com.videogo.util.LogUtil;

/**
 * 配网api
 *
 * @author dingwei3
 * @date 2018/5/11
 */
public class ConfigWifiSdkManager {
    
    private static final String TAG = ConfigWifiSdkManager.class.getSimpleName();

    /**
     * AP配网接口,默认为"EZVIZ_"+设备序列号,设备AP热点密码默认为"EZVIZ_"+设备验证码,默认自动连接设备热点，需要获取可扫描wifi的权限

     * @param ssid WiFi的ssid
     * @param password WiFi的密码
     * @param deviceSerial 设备序列号
     * @param verifyCode 设备验证码
     * @param apConfigCallback 结果回调
     * @since 5.0.0
     */
    public static void startAPConfigWifiWithSsid(final String ssid, final String password, String deviceSerial, final String verifyCode, final APWifiConfig.APConfigCallback apConfigCallback) {
        LogUtil.i(TAG, "Enter startAPConfigWifiWithSsid 1");
        startAPConfigWifiWithSsid(ssid, password, deviceSerial, verifyCode, "EZVIZ_" + deviceSerial, "EZVIZ_"+ verifyCode, true, null, apConfigCallback);
    }

    /**
     * AP配网接口
     * @param ssid WiFi的ssid
     * @param password WiFi的密码
     * @param deviceSerial 设备序列号
     * @param verifyCode 设备验证码
     * @param deviceWifiSsid 设备热点名称前缀，可传空，默认为"EZVIZ_"
     * @param deviceWifiPwd 设备热点密码前缀,可传空，默认为"EZVIZ_"
     * @param isAutoConnectDeviceHotSpot 是否自动连接设备热点
     * @param apiUrl 指定去哪个平台查询
     * @param callback 结果回调
     * @since 5.0.0
     */
    public static void startAPConfigWifiWithSsid(final String ssid, final String password, String deviceSerial, final String verifyCode, final String deviceWifiSsid, final String deviceWifiPwd,
                                                 boolean isAutoConnectDeviceHotSpot, final String apiUrl, final APWifiConfig.APConfigCallback callback) {
        LogUtil.i(TAG, "Enter startAPConfigWifiWithSsid 0");
        // step1 准备参数
        final ApConfigParam param = new ApConfigParam();
        param.routerWifiSsid = ssid;
        param.routerWifiPwd = password;
        param.deviceSerial = deviceSerial;
        param.deviceVerifyCode = verifyCode;
        param.deviceHotspotSsid = deviceWifiSsid;
        param.deviceHotspotPwd = deviceWifiPwd;
        param.autoConnect = isAutoConnectDeviceHotSpot;

        // step2 启动配网
        EZWiFiConfigManager.startAPConfig(BaseAPI.mApplication, param, new EZConfigWifiCallback() {
            @Override
            public void onInfo(int code, String message) {
                super.onInfo(code, message);
                // 相对比老版本的变更：
                // 老版本以配置发送给设备作为成功标志
                // 新版本还以平台设备已上线状态作为成功标志
                if (code == EZConfigWifiInfoEnum.CONNECTING_SENT_CONFIGURATION_TO_DEVICE.code){
                    callback.reportInfo(EZConfigWifiInfoEnum.CONNECTING_SENT_CONFIGURATION_TO_DEVICE);
                    startQueryDeviceStatusFromPlatform(param);
                }
            }
            @Override
            public void onError(int code, String description) {
                super.onInfo(code, description);
                callback.OnError(code);
                if (code == EZConfigWifiErrorEnum.CONFIG_TIMEOUT.code){
                    DeviceFinderFromPlatform.getInstance().stop();
                }
            }
            private void startQueryDeviceStatusFromPlatform(final ConfigParamAbstract configParam){
                DeviceFindParam findParam = new DeviceFindParam();
                findParam.serial = configParam.deviceSerial;
                findParam.wifiName = configParam.routerWifiSsid;
                findParam.wifiPwd = configParam.routerWifiPwd;
                DeviceFinderFromPlatform.getInstance().setCallback(new DeviceFindCallback() {
                    @Override
                    public void onFind(String deviceSerial) {
                        callback.reportInfo(EZConfigWifiInfoEnum.CONNECTED_TO_PLATFORM);
                    }

                    @Override
                    public void onTimeout(String deviceSerial) {
                        callback.reportError(EZConfigWifiErrorEnum.CONFIG_TIMEOUT);
                    }
                });
                DeviceFinderFromPlatform.getInstance().setApiUrl(apiUrl);
                DeviceFinderFromPlatform.getInstance().start(findParam);
            }
        });
    }

    /**
     * 停止AP配网
     * @since 5.0.0
     */
    public static void stopAPConfigWifiWithSsid() {
        LogUtil.i(TAG, "Enter stopAPConfigWifiWithSsid");
        EZWiFiConfigManager.stopAPConfig();
    }

    /**
     * 启动声波或SmartConfig配网
     * @since 5.0.0
     */
    public static void startConfigWifi(Context context, String deviceSerial, final String ssid, final String password,
                                       int mode, final EZOpenSDKListener.EZStartConfigWifiCallback callback) {
        startConfigWifi(context, deviceSerial, ssid, password, mode, null, callback);
    }

    /**
     * 启动声波或SmartConfig配网
     * @since 5.0.0
     */
    public static void startConfigWifi(Context context, String deviceSerial, final String ssid, final String password,
                                       int mode, final String apiUrl, final EZOpenSDKListener.EZStartConfigWifiCallback callback) {
        LogUtil.i(TAG, "Enter startConfigWifi  EZConfigWifiCallback");

        // step1 准备参数
        final MixedConfigParam param = new MixedConfigParam();
        param.routerWifiSsid = ssid;
        param.routerWifiPwd = password;
        param.deviceSerial = deviceSerial;
        param.mode = mode;
        // step2 启动配网
        EZWiFiConfigManager.startMixedConfig(BaseAPI.mApplication, param, new EZConfigWifiCallback() {
            @Override
            public void onInfo(int code, String message) {
                super.onInfo(code, message);
                if (code == EZConfigWifiInfoEnum.CONNECTED_TO_WIFI.code){
                    callback.onStartConfigWifiCallback(param.deviceSerial, EZConstants.EZWifiConfigStatus.DEVICE_WIFI_CONNECTED);
                    startQueryDeviceStatusFromPlatform(param);
                }
            }
            @Override
            public void onError(int code, String description) {
                super.onError(code, description);
                callback.onError(code, description);
                if(code == EZConfigWifiErrorEnum.CONFIG_TIMEOUT.code){
                    DeviceFinderFromPlatform.getInstance().stop();
                    callback.onStartConfigWifiCallback(param.deviceSerial, EZConstants.EZWifiConfigStatus.TIME_OUT);
                }
            }
            private void startQueryDeviceStatusFromPlatform(final ConfigParamAbstract configParam){
                DeviceFindParam findParam = new DeviceFindParam();
                findParam.serial = configParam.deviceSerial;
                findParam.wifiName = configParam.routerWifiSsid;
                findParam.wifiPwd = configParam.routerWifiPwd;
                DeviceFinderFromPlatform.getInstance().setCallback(new DeviceFindCallback() {
                    @Override
                    public void onFind(String deviceSerial) {
                        callback.onStartConfigWifiCallback(configParam.deviceSerial, EZConstants.EZWifiConfigStatus.DEVICE_PLATFORM_REGISTED);
                    }

                    @Override
                    public void onTimeout(String deviceSerial) {
                        callback.onStartConfigWifiCallback(configParam.deviceSerial, EZConstants.EZWifiConfigStatus.TIME_OUT);
                    }
                });
                DeviceFinderFromPlatform.getInstance().setApiUrl(apiUrl);
                DeviceFinderFromPlatform.getInstance().start(findParam);
            }
        });
    }

    /**
     * SmartConfig配网
     * @since 5.0.0
     */
    public static boolean startConfigWifi(Context context, final String ssid, final String password,
                                   final DeviceDiscoveryListener listener) {
        LogUtil.i(TAG, "Enter startConfigWifi  DeviceDiscoveryListener");
        // step1 准备参数
        MixedConfigParamOld param = new MixedConfigParamOld();
        param.routerWifiSsid = ssid;
        param.routerWifiPwd = password;
        // step2 启动配网
        EZWiFiConfigManager.startMixedConfigOld(BaseAPI.mApplication, param, listener);
        return true;
    }

    /**
     * 停止声波和SmartConfig配网
     * @since 5.0.0
     */
    public static void stopConfigWifi() {
        LogUtil.i(TAG, "Enter stopConfigWifi");
        EZWiFiConfigManager.stopMixedConfig();
    }

}
