package com.langchainbeam;

import org.apache.beam.sdk.transforms.PTransform;
import org.apache.beam.sdk.transforms.ParDo;
import org.apache.beam.sdk.values.PCollection;

import com.langchainbeam.model.EmbeddingModelOptions;
import com.langchainbeam.model.EmbeddingOutput;

/**
 * A {@link PTransform} that integrates embedding models with Apache Beam.
 * 
 * <p>
 * This transform converts a {@link PCollection} of input strings into a
 * {@link PCollection} of {@link EmbeddingOutput} objects, where each
 * {@link EmbeddingOutput}
 * contains the vector representation generated by the embedding model.
 * </p>
 * 
 * <p>
 * The transform leverages an embedding model to generate
 * embedding vectors for each element in the input {@code PCollection<String>}.
 * The resulting {@code PCollection<EmbeddingOutput>} contains the processed
 * embedding vectors, which can be used for tasks such as similarity search,
 * clustering, or machine learning model input.
 * </p>
 * 
 * <p>
 * An {@link EmbeddingModelHandler} is utilized to manage the embedding process.
 * This handler requires {@link EmbeddingModelOptions} as input, specifying
 * model-specific configurations such as the embedding model name, embedding
 * dimensions, and other parameters necessary for generating the embeddings.
 * </p>
 * 
 * <h3>Example Usage:</h3>
 * 
 * <pre>{@code
 * OpenAiEmbeddingModelOptions modelOptions = OpenAiEmbeddingModelOptions.builder()
 *         .apikey(apiKey)
 *         .modelName("text-embedding-3-small")
 *         .build();
 * 
 * EmbeddingModelHandler handler = new EmbeddingModelHandler(modelOptions);
 * 
 * PipelineOptions options = PipelineOptionsFactory.create();
 * 
 * // Create Beam pipeline
 * Pipeline p = Pipeline.create(options);
 * 
 * p.apply(TextIO.read().from("/home/ganesh/Downloads/product_reviews.csv")) // Load data
 *         .apply(LangchainBeamEmbedding.embed(handler)) // Apply the LangchainBeamEmbedding Transform
 *         .apply(ParDo.of(new DoFn<EmbeddingOutput, Void>() {
 * 
 *             @ProcessElement
 *             public void processElement(@Element EmbeddingOutput out) {
 *                 System.out.println("Embedding vector: " + Arrays.toString(out.getEmbedding().vector()));
 *             }
 *         }));
 * 
 * p.run();
 * }</pre>
 */
public class LangchainBeamEmbedding extends PTransform<PCollection<String>, PCollection<EmbeddingOutput>> {

    private final EmbeddingModelHandler handler;

    private LangchainBeamEmbedding(EmbeddingModelHandler modelHandler) {
        this.handler = modelHandler;
    }

    /**
     * Applies the embedding transformation to the input PCollection.
     */
    @Override
    public PCollection<EmbeddingOutput> expand(PCollection<String> input) {
        return input.apply("Embedding transform", ParDo.of(new EmbeddingDoFn(handler)));
    }

    /**
     * Creates and returns a new instance of {@code LangchainBeamEmbedding}.
     * 
     * @param modelHandler the handler used for interacting with the embedding
     *                     model. Must not be {@code null}.
     * @return a configured instance of the embedding transform.
     * @throws IllegalArgumentException if {@code modelHandler} is {@code null}.
     * 
     *                                  <p>
     *                                  Example:
     * 
     *                                  <pre>{@code
     * EmbeddingModelHandler handler = new EmbeddingModelHandler(...);
     * LangchainBeamEmbedding.embed(handler);
     * }</pre>
     */
    public static LangchainBeamEmbedding embed(EmbeddingModelHandler modelHandler) {
        if (modelHandler == null) {
            throw new IllegalArgumentException("Handler cannot be null");
        }
        return new LangchainBeamEmbedding(modelHandler);
    }

}
