package com.langchainbeam.model;

import java.io.Serializable;
import java.util.Objects;

import org.apache.beam.sdk.coders.DefaultCoder;
import org.apache.beam.sdk.coders.SerializableCoder;

import lombok.Builder;

/**
 * A data class representing the output of an embedding transformation.
 * 
 * <p>
 * {@link EmbeddingOutput} encapsulates the embedding vector and its associated
 * input string, providing a structured representation of the result of an
 * embedding model's processing.
 * </p>
 * 
 * <p>
 * This class is primarily used in conjunction with Beam pipelines to pass
 * embedding results downstream for further processing, such as similarity
 * search or writing to vector database
 * </p>
 * 
 * <h3>Components:</h3>
 * <ul>
 * <li>{@link BeamEmbedding} - Holds the vector representation generated by the
 * embedding model.</li>
 * <li>{@code inputElement} - The input string corresponding to the generated
 * embedding.</li>
 * </ul>
 * 
 * <h3>Example Usage:</h3>
 * 
 * <pre>{@code
 * BeamEmbedding embedding = new BeamEmbedding(new float[] { 0.1, 0.2, 0.3 });
 * EmbeddingOutput output = EmbeddingOutput.builder()
 *         .embedding(embedding)
 *         .inputElement("example input")
 *         .build();
 * 
 * System.out.println("Embedding vector: " + Arrays.toString(output.getEmbedding().vector()));
 * System.out.println("Input element: " + output.getInputElement());
 * }</pre>
 */
@DefaultCoder(SerializableCoder.class)
public class EmbeddingOutput implements Serializable {

    private final BeamEmbedding embedding;
    private final String inputElement;

    /**
     * Constructs an {@link EmbeddingOutput} instance.
     * 
     * @param embedding    the embedding vector generated by the model
     * @param inputElement the input string associated with the embedding
     */
    @Builder
    public EmbeddingOutput(BeamEmbedding embedding, String inputElement) {
        this.embedding = embedding;
        this.inputElement = inputElement;
    }

    /**
     * Returns the embedding vector generated by the model.
     * 
     * @return the {@link BeamEmbedding} containing the vector
     */
    public BeamEmbedding getEmbedding() {
        return embedding;
    }

    /**
     * Returns the input string associated with the embedding.
     * 
     * @return the input string
     */
    public String getInputElement() {
        return inputElement;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        EmbeddingOutput that = (EmbeddingOutput) o;
        return Objects.equals(embedding, that.embedding) &&
                Objects.equals(inputElement, that.inputElement);
    }

    @Override
    public int hashCode() {
        return Objects.hash(embedding, inputElement);
    }
}
