package io.github.ibony.ats.android;

import io.github.ibony.ats.common.command.Command;
import io.github.ibony.ats.common.utf7.ByteUtil;
import io.github.ibony.ats.common.utf7.CharsetProvider;
import io.github.ibony.ats.utils.FileUtil;
import io.github.ibony.ats.utils.Xml;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.dom4j.Element;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.List;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/***
 * Created by Bony on 2017-09-13.
 * 操作信息监控 uiautomator events
 ***/
public class Device {
    private String NAME;
    private String SERIAL;
    private int WIDTH;
    private int HEIGHT;
    private String ActivityName;
    private boolean DUMP_COMPRESSED = true;
    private final Log logger = LogFactory.getLog(Device.class);
    private ActivityChangeListener activityChangeListener = null;

    /**
     * 构造函数
     *
     * @param serial 设备ID
     * @param name   设备名称
     */
    public Device(String serial, String name) {
        this.NAME = name;
        this.SERIAL = serial;
        this.ActivityName = NAME + SERIAL.replaceAll("\\.", "").replaceAll(":", "-") + "Activity.xml";
        initSize();
    }

    public Device() {
    }

    public String getAppVersionName(App app) {
        return getAppVersionName(app.getPackageName());
    }

    public int getAppVersionCode(App app) {
        return getAppVersionCode(app.getPackageName());
    }

    public String getAppVersionName(String packageName) {
        String result = adbCommand(String.format("adb shell \"dumpsys package %s |grep versionName\"", packageName)).readClose();
        Matcher matcher_tmp = Pattern.compile("versionName=(\\S+)").matcher(result);
        if (matcher_tmp.find()) {
            return matcher_tmp.group(1);
        } else return "";
    }

    public int getAppVersionCode(String packageName) {
        String result = adbCommand(String.format("adb shell \"dumpsys package %s |grep versionCode\"", packageName)).readClose();
        Matcher matcher_tmp = Pattern.compile("versionCode=(\\S+)").matcher(result);
        if (matcher_tmp.find()) {
            try {
                return Integer.parseInt(matcher_tmp.group(1));
            } catch (Exception e) {
                e.printStackTrace();
                return 0;
            }
        } else return 0;
    }

    /**
     * 输入文字（不支持中文）
     *
     * @param content 文字
     */
    public void inputTextEn(String content) {
        adbCommand("adb shell input text " + content).readClose();
    }

    /**
     * 输入文字（支持中文）
     * 1.安裝Utf7Ime.apk，
     * 2.设置UTF7 IMEI for UI Testing为默认输入法(最好先获取当前输入法，输入完之后再还原输入法)
     * adb shell settings put secure default_input_method jp.jun_nama.test.utf7ime/.Utf7ImeService
     *
     * @param content 输入内容
     */
    public void inputTextZh(String content) {
        adbCommand("adb shell input text \\\"" +
                new String(ByteUtil.getBytes(
                        content,
                        new CharsetProvider().charsetForName("X-MODIFIED-UTF-7")
                ), Charset.forName("US-ASCII")) + "\\\""
        ).readClose();
    }

    /**
     * 初始化支持中文输入
     *
     * @return 初始化结果
     */
    public boolean initTextZh() {
        if (!isSupportInputTextZh()) {
            try {
                InputStream inputStream = this.getClass().getClassLoader().getResourceAsStream("inputting_method/Utf7Ime.apk");
                FileUtil.copy(
                        inputStream,
                        new FileOutputStream("Utf7Ime.apk")
                );
            } catch (FileNotFoundException e) {
                e.printStackTrace();
            }
            installApp("Utf7Ime.apk");
        }
        setDefaultInputMethod("jp.jun_nama.test.utf7ime/.Utf7ImeService");
        return isSupportInputTextZh();
    }

    /**
     * 判断是否支持中文
     *
     * @return 判断结果
     */
    public boolean isSupportInputTextZh() {
        List<String> ps = adbCommand("adb shell pm list packages -3").readLinesClose();
        return ps.toString().contains("jp.jun_nama.test.utf7ime");
    }

    /**
     * 设置默认输入法
     *
     * @param mainActivity 默认输入法的入口
     */
    public void setDefaultInputMethod(String mainActivity) {
        adbCommand("adb shell settings put secure default_input_method " + mainActivity).readClose();
    }

    /**
     * 判断设备设备连接正常
     *
     * @return false 连接正常，true 连接失败
     */
    public boolean isConnect() {
        String res = adbCommand("adb get-state").readClose();
        if (res.contains(String.format("error: device '%s' not found", getSerial()))) return false;
        return true;
    }

    /**
     * 点击坐标
     *
     * @param x 坐标x
     * @param y 坐标y
     */
    public void click(int x, int y) {
        adbCommand("adb shell input tap " + x + " " + y).readClose();
    }

    /**
     * 点击坐标
     *
     * @param x 坐标x
     * @param y 坐标y
     */
    public void click(double x, double y) {
        adbCommand("adb shell input tap " + x + " " + y).readClose();
    }

    /**
     * 滑动坐标
     *
     * @param x0 起始坐标x
     * @param y0 起始坐标y
     * @param x1 结束坐标x
     * @param y1 结束坐标y
     */
    public void swipe(int x0, int y0, int x1, int y1) {
        adbCommand("adb shell input swipe " + x0 + " " + y0 + " " + x1 + " " + y1).readClose();
    }

    /**
     * 滑动坐标
     *
     * @param x0 起始坐标x
     * @param y0 起始坐标y
     * @param x1 结束坐标x
     * @param y1 结束坐标y
     */
    public void swipe(double x0, double y0, double x1, double y1) {
        adbCommand("adb shell input swipe " + x0 + " " + y0 + " " + x1 + " " + y1).readClose();
    }

    /**
     * 点击模拟按键
     *
     * @param code 模拟code
     */
    public void clickKeyEvent(KeyCode code) {
        adbCommand("adb shell input keyevent " + code.code).readClose();
    }

    public void clickBack() {
        clickKeyEvent(KeyCode.KEYCODE_BACK);
    }

    public void clickEnter() {
        clickKeyEvent(KeyCode.KEYCODE_ENTER);
    }

    public void clickHome() {
        clickKeyEvent(KeyCode.KEYCODE_HOME);
    }

    public void clickMenu() {
        clickKeyEvent(KeyCode.KEYCODE_MENU);
    }

    public void clickCall() {
    }

    /**
     * 向左滑
     *
     * @param d 滑动比例
     */
    public void swipeToLeft(double d) {
        if (d > 1) {
            new Exception("value out of range").printStackTrace();
            d = 1;
        }
        d = 1 - d;
        swipe(
                WIDTH - (WIDTH * d * 0.5),
                HEIGHT / 2,
                WIDTH * d * 0.5,
                HEIGHT / 2
        );
    }

    /**
     * 向左滑
     */
    public void swipeToLeft() {
        swipeToLeft(1);
    }

    /**
     * 向右滑
     *
     * @param d 滑动比例
     */
    public void swipeToRight(double d) {
        if (d > 1) {
            new Exception("value out of range").printStackTrace();
            d = 1;
        }
        d = 1 - d;
        swipe(
                WIDTH * d * 0.5,
                HEIGHT / 2,
                WIDTH - (WIDTH * d * 0.5),
                HEIGHT / 2
        );
    }

    /**
     * 向右滑
     */
    public void swipeToRight() {
        swipeToRight(1);
    }

    /**
     * 向上滑
     *
     * @param d 滑动比例
     */
    public void swipeToUp(double d) {
        if (d > 1) {
            new Exception("value out of range").printStackTrace();
            d = 1;
        }
        d = 1 - d;
        swipe(
                WIDTH / 2,
                HEIGHT - (HEIGHT * d * 0.5),
                WIDTH / 2,
                HEIGHT * d * 0.5
        );
    }

    /**
     * 向上滑
     */
    public void swipeToUp() {
        swipeToUp(1);
    }

    /**
     * 向下滑
     *
     * @param d 滑动比例
     */
    public void swipeToDown(double d) {
        if (d > 1) {
            new Exception("value out of range").printStackTrace();
            d = 1;
        }
        d = 1 - d;
        swipe(
                WIDTH / 2,
                HEIGHT * d * 0.5,
                WIDTH / 2,
                HEIGHT - (HEIGHT * d * 0.5)
        );
    }

    /**
     * 向上滑
     *
     */
    public void swipeToDown() {
        swipeToDown(1);
    }


    /**
     * isDUMP_COMPRESSED
     * @return 判断结果
     */
    public boolean isDUMP_COMPRESSED() {
        return DUMP_COMPRESSED;
    }

    public void setDumpCompressed(boolean DUMP_COMPRESSED) {
        this.DUMP_COMPRESSED = DUMP_COMPRESSED;
    }

    /**
     * 安装路径下的apk
     *
     * @param appPath apk路径
     * @return 安装结果
     */
    public boolean installApp(String appPath) {
        boolean flag = false;
        List<String> results = adbCommand("adb install -r " + appPath).readLinesClose();
        if (results.get(results.size() - 1).equals("Success")) {
            flag = true;
        }
        return flag;
    }

    /**
     * 安装APP
     *
     * @param app app对象
     */
    public void installApp(App app) {
        if (app.getLocalPath() != null) {
            adbCommand("adb install -r " + app.getLocalPath());
        }
    }


    public void setDefaultInputMethod() {
        adbCommand("adb shell settings put secure default_input_method jp.jun_nama.test.utf7ime/.Utf7ImeService");
    }


    /**
     * 根据mainActivity对象，启动app
     *
     * @param mainActivity mainActivity
     */
    public void startApp(String mainActivity) {
        adbCommand("adb shell am start -n " + mainActivity).readClose();
    }

    /**
     * 根据APP对象，启动app
     *
     * @param app app
     */
    public void startApp(App app) {
        if (app.getPackageName() != null && app.getMainActivity() != null) {
            adbCommand("adb shell am start -n " + app.getPackageName());
        }
    }

    /**
     * 根据APP包名，APP停止运行
     *
     * @param packagename packagename
     */
    public void stopApp(String packagename) {
        adbCommand("adb shell am force-stop " + packagename).readClose();
    }

    /**
     * 根据APP对象，APP停止运行
     *
     * @param app app
     */
    public void stopApp(App app) {
        if (app.getPackageName() != null) {
            adbCommand("adb shell am force-stop " + app.getPackageName()).readClose();
        }
    }

    /**
     *
     * 停止并清除APP数据
     * @param packageName 包名
     */
    public void clearApp(String packageName) {
        adbCommand("adb shell pm clear " + packageName).readClose();
    }

    /**
     * 获取设备当前界面布局xml文件
     *
     * @param FileName 获取之后存放的地址
     * @return 获取结果
     */
    public boolean getActivityXml(String FileName) {
        return getActivityXml("CurrentActivity", FileName);
    }

    /**
     *
     * 获取设备当前界面布局xml文件
     *
     * @param sourceName 源名称
     * @param FileName   获取之后存放的地址
     * @return 结果
     */
    public boolean getActivityXml(String sourceName, String FileName) {
        String dumpContent;
        if (isDUMP_COMPRESSED()) {
            dumpContent = adbCommand("adb shell uiautomator dump --compressed /data/local/tmp/" + sourceName + ".xml").readClose();
        } else {
            dumpContent = adbCommand("adb shell uiautomator dump /data/local/tmp/" + sourceName + ".xml").readClose();
        }
        if (dumpContent.contains("ERROR: null root node returned by UiTestAutomationBridge.")) {
            if (Driver.DEBUG) logger.debug(dumpContent);
            return false;
        }
        adbCommand("adb pull /data/local/tmp/" + sourceName + ".xml " + FileName);
        adbCommand("adb shell rm -r /data/local/tmp/" + sourceName + ".xml");
        return true;
    }

    public String getActivityXml() {
        String dumpContent;
        if (isDUMP_COMPRESSED()) {
            dumpContent = adbCommand("adb shell uiautomator dump --compressed /sdcard/window_dump.xml").readClose();
        } else {
            dumpContent = adbCommand("adb shell uiautomator dump /sdcard/window_dump.xml").readClose();
        }
        if (dumpContent.contains("ERROR: null root node returned by UiTestAutomationBridge.")) {
            if (Driver.DEBUG) logger.debug(dumpContent);
            return null;
        }
        String result = adbCommand("adb shell cat /sdcard/window_dump.xml").readClose();
        adbCommand("adb shell rm -r /sdcard/window_dump.xml").readClose();
        if (result.contains("No such file or directory")) return null;
        return result;
    }


    /**
     * 支持webView元素获取
     */
    public void WdHub() {
        adbCommand("res/bin/chromedriver.exe --url-base=wd/hub --port=9515").close();
        adbCommand("adb shell \"am force-stop uiautomator\"").close();
    }

    public void openUrl(String url) {
        adbCommand("adb shell am start -a android.intent.action.VIEW -d " + url);
    }

    /**
     * 获取元素列表
     *
     * @param type 属性类型
     * @param value 属性值
     * @return 元素列表
     */
    public ArrayList<DElement> getIElementAll(String type, String value) {
//        if (!getActivityXml(ActivityName)) return null;
        String xmlContent = getActivityXml();
        if (xmlContent == null) return null;
        ArrayList<DElement> IElements = new ArrayList();
        for (Element element : new Xml(xmlContent).getElementAll("node")) {
            IElements.add(new DElement(this, element, null));
        }
        return IElements;
    }

    /**
     * 获取元素树
     *
     * @return 结果
     */
    public DElement getIElementTree() {
//        if (!getActivityXml(ActivityName)) return null;
        String xmlContent = getActivityXml();
        if (xmlContent == null) return null;
        Element rootElement = new Xml(xmlContent).getDocument().getRootElement();
        return Element2DElement(rootElement, null);
    }

    /**
     * 获取元素根
     *
     * @return 元素
     */
    public DElement getRootDElement() {
//        if (!getActivityXml(ActivityName)) return null;
        String xmlContent = getActivityXml();
        if (xmlContent == null) return null;
        Element rootElement = new Xml(xmlContent).getDocument().getRootElement();
        return Element2DElement(rootElement, null);
    }

    /**
     * 普通元素对象转为APP元素对象
     *
     * @param element 元素
     * @param dElement 元素
     * @return 结果元素
     */
    public DElement Element2DElement(Element element, DElement dElement) {
        DElement iElement = new DElement(this, element, dElement);
        for (Element e : (List<Element>) element.elements()) iElement.add(Element2DElement(e, iElement));
        return iElement;
    }

    /**
     * 根据属性获取元素
     *
     * @param type 属性类型
     * @param value 属性值
     * @return 元素
     */
    public DElement findTypeElement(String type, String value) {
        DElement root = getRootDElement();
        if (root == null) return null;
        for (DElement element : root.getElements()) {
            if (element.getType(type).equals(value)) {
                return element;
            }
            DElement dElement1 = element.findTypeElement(type, value);
            if (dElement1 != null) {
                return dElement1;
            }
        }
        return null;
    }

    /**
     * 根据元素属性获取元素
     * @param type 属性类
     * @param value 属性值
     * @return 元素对象
     */
    public DElement findTypeContainElement(String type, String value) {
        DElement root = getRootDElement();
        if (root == null) return null;
        for (DElement element : root.getElements()) {
            if (element.getType(type).contains(value)) {
                return element;
            }
        }
        return null;
    }

    /**
     * 根据属性判断元素是否存在
     *
     * @param type  元素属性类别
     * @param value 属性值
     * @return 判断结果
     */
    public boolean isTypeElement(String type, String value) {
        if (findTypeElement(type, value) == null) return false;
        else return true;
    }

    /**
     * 根据index元素是否存在
     *
     * @param indexPath 索引路径
     * @return boolean 是否存在
     */
    public boolean isIndexXpathElement(int[] indexPath) {
        if (findIndexPathElement(indexPath) == null) return false;
        else return true;
    }

    /**
     * 根据属性获取元素群
     *
     * @param type  元素属性类别
     * @param value 元素属性值
     * @return 元素对象
     */
    public ArrayList<DElement> findTypeElements(String type, String value) {
        ArrayList<DElement> elements = new ArrayList<>();
        for (DElement element : getRootDElement().getElements()) {
            if (element.getType(type).equals(value)) {
                elements.add(element);
            }
            DElement dElement1 = element.findTypeElement(type, value);
            if (dElement1 != null) {
                dElement1.setParent(element);
                elements.add(element);
            }
        }
        return elements;
    }

    /**
     * 根据Index路径获取元素
     *
     * @param indexPath 索引路径
     * @return 元素对象
     */
    public DElement findIndexPathElement(int[] indexPath) {
        DElement dElement = getIElementTree();
        for (int index : indexPath) {
            dElement = dElement.getElements().get(index);
        }
        return dElement;
    }

    /**
     * 获取指定位置的元素
     *
     * @param root  根节点
     * @param index 索引值
     * @return 元素对象
     */
    public Element getIndexElement(Element root, int index) {
        Element element = null;
        element = (Element) root.elements().get(index);
        return element;
    }

    /**
     * 捕获屏幕（获取屏幕截图）
     *
     * @param path 获取之后存放的地址
     */
    public void getScreenCapture(String path) {
        adbCommand("adb shell screencap -p /data/local/tmp/ScreenCapture.png").readClose();
        pull("/data/local/tmp/ScreenCapture.png", path);
        adbCommand("adb shell rm /data/local/tmp/ScreenCapture.png").readClose();
    }

    /**
     * 捕获屏幕（获取屏幕截图）
     */
    public void getScreenCapture() {
        getScreenCapture(getName() + "_ScreenCapture.jpg");
    }


    /**
     * 捕获屏幕（获取屏幕截图）,并获取区域截屏
     * @param element 元素
     * @param path 存储路径
     */
    public void getScreenCaptureShear(DElement element, String path) {
        getScreenCaptureShear(element.getX0(), element.getY0(), element.getX1(), element.getY1(), path);
    }

    /**
     * 捕获屏幕（获取屏幕截图）,并获取控件截屏
     * @param x0 开始坐标X
     * @param y0 开始坐标Y
     * @param x1 结束坐标X
     * @param y1 结束坐标Y
     * @param path 路径
     */
    public void getScreenCaptureShear(int x0, int y0, int x1, int y1, String path) {
        String tmp_path = getName() + "_ScreenCapture.jpg";
        try {
            getScreenCapture(tmp_path);
            BufferedImage image = ImageIO.read(new File(tmp_path));
            ImageIO.write(image.getSubimage(x0, y0, x1, y1), "jpg", new File(path));// 写图片
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取当前前台的Activity
     * @return 前台的Activity
     */
    public String getFocusedPackageActivity() {
        String result = adbCommand("adb shell \"dumpsys window |grep mCurrentFocus\"").readClose();
        Pattern pattern = Pattern.compile("(\\S+/\\S+)");
        Matcher m = pattern.matcher(result);
        String str = "";
        if (m.find()) {
            str = m.group(0);
        }
        return str.replaceAll("\\{", "").replaceAll("}", "");
    }

    public Choreographer getChoreographer() {
        return new Choreographer(this);
    }


    /**
     * 设置指定文件权限位最大
     *
     * @param filePath 文件路径
     */
    public void chmod_R(String filePath) {
        chmod(filePath, 7777);
    }

    /**
     * 指定文件设置指定的权限
     *
     * @param targetPath 路径
     * @param us 权限
     */
    public void chmod(String targetPath, int us) {
        adbCommand(String.format("adb shell chmod -R %s %s", us, targetPath)).readClose();
    }

    public String getName() {
        return NAME;
    }

    public void setName(String name) {
        this.NAME = name;
    }

    public String getSerial() {
        return SERIAL;
    }

    public String setSerial(String serial) {
        return this.SERIAL = serial;
    }

    public int getWidth() {
        return WIDTH;
    }

    public int getHeight() {
        return HEIGHT;
    }

    public void setWidth(int width) {
        this.WIDTH = width;
    }

    public void setHeight(int height) {
        this.HEIGHT = height;
    }

    /**
     * 初始化设备大小(设备尺寸)
     */
    public void initSize() {
        Matcher matcher_tmp;
        String Request = adbCommand("adb shell \"dumpsys window displays |grep \\\"init\\\"\"").readClose();
        matcher_tmp = Pattern.compile("init=(\\S+)").matcher(Request);
        while (matcher_tmp.find()) {
            Request = matcher_tmp.group(1);
            WIDTH = Integer.parseInt(Request.split("x")[0]);
            HEIGHT = Integer.parseInt(Request.split("x")[1]);
        }
    }


    /**
     * 推送文件到设备
     *
     * @param currenPath 本机地址
     * @param targetPath 设备目标地址
     */
    public void push(String currenPath, String targetPath) {
        adbCommand(String.format("adb push %s %s", currenPath, targetPath)).readClose();
    }

    /**
     * 获取文件到本地
     *
     * @param currenPath 远程路径
     * @param targetPath 本地路径
     */
    public void pull(String currenPath, String targetPath) {
        try {
//            Thread.sleep(1000);
            adbCommand(String.format("adb pull %s %s", currenPath, targetPath)).readClose();
//            Thread.sleep(1000);
        } catch (Exception e) {
            e.printStackTrace();
        }
    }

    /**
     * 获取CPU版本
     *
     * @return CPU版本
     */
    public String getCPU() {
        String result = this.adbCommand("adb shell getprop ro.product.cpu.abi").readClose();
        if (result.contains("error")) {
            return null;
        } else {
            return result.trim().replaceAll("\r|\n", "");
        }
    }

    public DElement findElementByBounds(String value) {
        return findTypeElement(DElementType.BOUNDS, value);
    }

    public DElement findElementByChecked(String value) {
        return findTypeElement(DElementType.CHECKED, value);
    }

    public DElement findElementByClass(String value) {
        return findTypeElement(DElementType.CLASS, value);
    }

    public DElement findElementByContentDesc(String value) {
        return findTypeElement(DElementType.CONTENT_DESC, value);
    }

    public DElement findElementByText(String value) {
        return findTypeElement(DElementType.TEXT, value);
    }

    public DElement findElementByResourceId(String value) {
        return findTypeElement(DElementType.RESOURCE_ID, value);
    }

    public DElement findElementByPackage(String value) {
        return findTypeElement(DElementType.PACKAGE, value);
    }

    public DElement findElementBySelected(String value) {
        return findTypeElement(DElementType.SELECTED, value);
    }

    public DElement findElementByEnabled(String value) {
        return findTypeElement(DElementType.ENABLED, value);
    }

    public DElement findElementByLongClickable(String value) {
        return findTypeElement(DElementType.LONG_CLICKABLE, value);
    }

    /**
     * 获取SDK版本
     *
     * @return SDK版本
     */
    public String getSDK() {
        String result = this.adbCommand("adb shell getprop ro.build.version.sdk").readClose();
        if (result.contains("error")) {
            return null;
        } else {
            return result.trim().replaceAll("\r|\n", "");
        }
    }


    /**
     * adb命令发送方法
     *
     * @param adbStr 命令串
     * @return Command对象
     */
    public Command adbCommand(String adbStr) {
        return Driver.command(getAdbCommandStr(adbStr));
    }

    /**
     * adb命令发送方法
     *
     * @param adbStr 命令串
     * @return Command对象
     */
    public Command adbCommand(String[] adbStr) {
        return Driver.command(adbStr);
    }

    /**
     * 设备命令串格式化
     *
     * @param adbStr 源命令
     * @return 格式化结果
     */
    public String getAdbCommandStr(String adbStr) {
        adbStr = adbStr.replaceAll("adb", "adb -s " + this.getSerial());
        return adbStr;
    }

    public String toString() {
        return NAME + "_" + SERIAL;
    }

    /**
     * 判断设备是否root
     *
     * @return 判断结果
     */
    public boolean isRooted() {
        BufferedReader reader = null;
        Process process = null;
        PrintStream outputStream = null;
        boolean flag = false;

        try {
            process = Runtime.getRuntime().exec("adb shell");
            outputStream = new PrintStream(new BufferedOutputStream(process.getOutputStream(), 8192));
            outputStream.println("ls /data/");
            outputStream.flush();

            reader = new BufferedReader(new InputStreamReader(process.getInputStream()));

            if (reader.readLine() != null) {
                flag = true;
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            if (outputStream != null) {
                outputStream.close();
            }
            if (process != null) {
                process.destroy();
            }
            if (reader != null) {
                try {
                    reader.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        }
        return flag;
    }

    /**
     * getProcessInformationList
     * @param keyword keyword
     * @return ProcessInformation列表
     */
    public List<ProcessInformation> getProcessInformationList(String keyword) {
        return ProcessInformation.getProcessInformationList(this, keyword);
    }

    /**
     * 添加界面变化监听器
     *
     * @param activityChangeListener ActivityChangeListener
     */
    public void addActivityChangeListener(ActivityChangeListener activityChangeListener) {
        this.activityChangeListener = activityChangeListener;
        this.activityChangeListener.start(this);
    }

    /**
     * 清除界面变化监听器
     */
    public void cleanActivityChangeListener() {
        this.activityChangeListener.stop();
    }


    /**
     * 杀掉进程，需要root权限
     *
     * @param processInformation 进程
     */
    public void destructionProcessInformation(ProcessInformation processInformation) {
        adbCommand("adb shell kill " + processInformation.getPID()).readClose();
    }
}
