package io.github.ibony.ats.android.fps;

import io.github.ibony.ats.android.Device;
import io.github.ibony.ats.common.FixSizeLinkedList;
import io.github.ibony.ats.common.command.Command;
import io.github.ibony.ats.common.command.CommandListener;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.*;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * TODO 命令数据解释：
 * TODO 第一部分数据t1 -- when the app started to draw （开始绘制图像的瞬时时间）
 * TODO 第二部分数据t2 -- the vsync immediately preceding SF submitting the frame to the h/w （VSYNC信令将软件SF帧传递给硬件HW之前的垂直同步时间），也就是对应上面所说的软件Vsync
 * TODO 第三部分数据t3 -- timestamp immediately after SF submitted that frame to the h/w （SF将帧传递给HW的瞬时时间，及完成绘制的瞬时时间）
 * TODO 将第i行和第i-1行t2相减，即可得到第i帧的绘制耗时
 *
 * adb shell dumpsys SurfaceFlinger --latency [window name]
 * prints some information about the last 128 frames displayed in
 * that window.
 * The data returned looks like this:
 * 16954612
 * 7657467895508   7657482691352   7657493499756
 * 7657484466553   7657499645964   7657511077881
 * 7657500793457   7657516600576   7657527404785
 * (...)
 *
 * The first line is the refresh period (here 16.95 ms), it is followed
 * by 128 lines w/ 3 timestamps in nanosecond each:
 * A) when the app started to draw
 * B) the vsync immediately preceding SF submitting the frame to the h/w
 * C) timestamp immediately after SF submitted that frame to the h/w
 *
 * The difference between the 1st and 3rd timestamp is the frame-latency.
 * An interesting data is when the frame latency crosses a refresh period
 * boundary, this can be calculated this way:
 *
 * ceil((C - A) / refresh-period)
 *
 * (each time the number above changes, we have a "jank").
 * If this happens a lot during an animation, the animation appears
 * janky, even if it runs at 60 fps in average.
 *
 * We use the special "SurfaceView" window name because the statistics for
 * the activity's main window are not updated when the main web content is
 * composited into a SurfaceView.
 * print (self._focuse_name)
 *
 */
public abstract class SurfaceFlinger {
    private static final Log logger = LogFactory.getLog(Device.class);
    Device device;
    String activityRecord;
    Command adbCommand;
    boolean ACQUISITION_STATUS = false;
    int idealFPS = 60;//理想的帧率
    Map<String, Float> callMap = new HashMap();
    LinkedBlockingQueue<long[]> queue = new LinkedBlockingQueue();
    float refreshPeriod = 0;
    FixSizeLinkedList<long[]> instantFrames;
    float fps;//刷新率
    float jank;//掉帧

    /**
     * 构造函数
     *
     * @param device         设备对象
     * @param activityRecord 界面名称
     */
    public SurfaceFlinger(Device device, String activityRecord) {
        this.device = device;
        this.activityRecord = activityRecord;
        init();
    }

    /**
     * 构造函数
     *
     * @param device         设备对象
     * @param activityRecord 界面名称
     * @param idealFPS       理想的帧率
     */
    public SurfaceFlinger(Device device, String activityRecord, int idealFPS) {
        this.idealFPS = idealFPS;
        this.device = device;
        this.activityRecord = activityRecord;
        init();
    }

    /**
     * 初始化全局变量
     */
    void init() {
        instantFrames = new FixSizeLinkedList<>(idealFPS);
        adbCommand = device.adbCommand("adb shell");
        adbCommand.setCommandListener(new CommandListener() {
            @Override
            public void handler(String message) {
                try {
                    long[] arr = Arrays.stream(message.split("\t")).mapToLong(Long::valueOf).toArray();
                    queue.put(arr);
                } catch (Exception e) {
//                    e.printStackTrace();
                }
            }
        });
    }

    public void setActivityRecord(String activityRecord) {
        this.activityRecord = activityRecord;
    }

    public void setActivity(String activity) {
        this.activityRecord = activity + "#0";
    }

    /**
     * 启动
     */
    public void start() {
        ACQUISITION_STATUS = true;
        new Thread(new DumpDataRunnable()).start();
        new Thread(new CalculationRunnable()).start();

    }

    /**
     * 启动
     */
    public void stop() {
        ACQUISITION_STATUS = false;
        adbCommand.close();
    }

    /**
     * 预处理筛选可计算数据
     *
     * @param message 每一行的数据详情
     */
    private void pretreatment(long[] message) {
        if (message.length >= 3) {
            if (message[0] == 0 && message[1] == 0 && message[2] == 0) return;
            if (message[1] == 9223372036854775807f) return;
            //判断链表是否存在内容
            if (instantFrames.size() > 0) {
                if (message[1] > instantFrames.getLast()[1]) {
                    instantFrames.add(message);
                    getCalculation();
                    callMap.put("endFrame1", (float) message[0]);
                    callMap.put("endFrame2", (float) message[1]);
                    callMap.put("endFrame3", (float) message[2]);
                    callMap.put("fps", fps);
//                    callMap.put("jank", jank);
                    callMap.put("refreshPeriod", refreshPeriod);
                    callData(callMap);
                }
            } else instantFrames.add(message);
        } else {
            refreshPeriod = (float) message[0] / (float) 1e6;
        }
    }

    /**
     * fps与jank的核心计算部分
     */
    public void getCalculation() {
        float totalFrame = 0;
        float totalTime = 0;
        jank = 0;
        for (long[] frame : instantFrames) {
            int index = instantFrames.indexOf(frame);
            if (index >= 1) {
                long[] startFrame = instantFrames.get(index - 1);
                float intervalTime = (float) (frame[1] - startFrame[1]);
                intervalTime = (intervalTime / (float) 1e6);
//                if (intervalTime > refreshPeriod) jank++;
                intervalTime = (float) (Math.ceil(intervalTime / refreshPeriod) * refreshPeriod);
                if (intervalTime > 1 && intervalTime < 160) {//排除帧索引计算间隔时间大于160毫秒的数据
                    totalFrame++;
                    totalTime += intervalTime;
                }
            }
        }
        fps = totalFrame / (totalTime / 1000);
        fps = (int) (fps * 100) / (float) 100.00;
    }

    /**
     * 获取活动的界面记录
     * @param device 设备对象
     * @param pn     包名
     * @return 活动列表
     */
    public static LinkedList<String> getRecordActivityNameList(Device device, String pn) {
        LinkedList<String> RecordActivityNameList = new LinkedList<>();
        List<String> rs = device.adbCommand("adb shell \"dumpsys SurfaceFlinger --list |grep " + pn + "\"").readLinesClose();
        for (String s : rs) {
            if (s.contains("\t")) continue;
            Pattern pattern = Pattern.compile("(\\S+/\\S+#+[0-9]*)");
            Matcher m = pattern.matcher(s);
            if (m.find()) {
                RecordActivityNameList.add(m.group(0));
            }
        }
        return RecordActivityNameList;
    }

    public abstract void callData(Map<String, Float> data);

    /**
     * 数据收集线程
     */
    private class DumpDataRunnable implements Runnable {
        public void run() {
            adbCommand.put("dumpsys SurfaceFlinger --latency-clear");
            while (ACQUISITION_STATUS) {
                try {
                    Thread.sleep(10);
                } catch (InterruptedException e) {
                    e.printStackTrace();
                }
                adbCommand.put("dumpsys SurfaceFlinger --latency " + activityRecord);
            }
        }
    }

    /**
     * 数据处理线程
     */
    private class CalculationRunnable implements Runnable {
        public void run() {
            while (ACQUISITION_STATUS) {
                if (!queue.isEmpty()) {
                    pretreatment(queue.poll());
                }
            }
        }
    }
}
