package io.github.ibony.ats.android.minicap;

import io.github.ibony.ats.android.Device;
import io.github.ibony.ats.common.command.Command;
import io.github.ibony.ats.utils.FileUtil;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.*;
import java.net.Socket;
import java.util.Queue;
import java.util.concurrent.LinkedBlockingQueue;

/**
 * Created by 80071482 on 2017-06-05.
 * 可执行下载地址：https://github.com/openstf/minicap/tree/master/jni/minicap-shared/aosp/libs
 */
public abstract class MiniCap {
    private static final Log logger = LogFactory.getLog(MiniCap.class);
    private Device device;
    private Command minicapServer = new Command();
    private int ROTATE, PORT;
    private Socket SOCKET;
    private boolean SCREEN_SYNCHRONIZATION_OUPUT = false;
    private int readBannerBytes = 0;
    private int bannerLength = 2;
    private int readFrameBytes = 0;
    private int frameBodyLength = 0;
    private Queue<byte[]> DATA_QUEUE = new LinkedBlockingQueue<byte[]>();
    private Queue<byte[]> IMGDATA_QUEUE = new LinkedBlockingQueue<byte[]>();
    private InputStream ATREAM = null;
    private int len = 4096;
    private int realLen;
    private Header header = new Header();
    private Thread frame;
    private Thread image_converter;
    private Thread image_out;
    private byte[] frameBody = new byte[0];

    public  void imageCollectorStartChange(){};

    public  void imageCollectorStopChange(){};

    public  void imageConverterStartChange(){};

    public  void imageConverterStopChange(){};

    public  void imageOutChange(BufferedImage image){};

    public  void imageOutStopChange(){};

    public  void getImageStopChange(){};

    public  void getImageStartChange(){};

    /**
     * 获取设备数据传输状态
     *
     * @return 结果
     */
    public boolean getScreenSynchronizationState() {
        return SCREEN_SYNCHRONIZATION_OUPUT;
    }

    /**
     * 设置设备数据传输状态
     * @param state ScreenSynchronizationState
     */
    public void setScreenSynchronizationState(boolean state) {
        SCREEN_SYNCHRONIZATION_OUPUT = state;
    }

    //参数初始化
    void initArg() {
        DATA_QUEUE.clear();
        IMGDATA_QUEUE.clear();
        readBannerBytes = 0;
        bannerLength = 2;
        readFrameBytes = 0;
        frameBodyLength = 0;
        frameBody = new byte[0];
    }

    /**
     * 初始化MiniCap连接
     *
     * @param device     设备对象
     * @param port       链接端口
     * @param rotate     图像旋转角度 0,90,180,270
     * @param definition 图像清晰度（清晰度对数据流大小存在影响，负相关，同理影响传输速度） 1-100
     * @return MiniCap对象
     */
    public MiniCap initMinicap(Device device, int port, int rotate, int definition) {
        initArg();
        this.device = device;
        this.PORT = port;
        this.ROTATE = rotate;
        device.initSize();
        String sourcePath = String.format("minicap/bin/%s/minicap", device.getCPU());//.replaceAll("//",File.separator);
        String sourceSoPath = String.format("minicap/shared/android-%s/%s/minicap.so", device.getSDK(), device.getCPU());//.replaceAll("//",File.separator);
        pushMinicapFile(sourcePath, sourceSoPath);
        setScreenSize(ROTATE, definition, device.getWidth(), device.getHeight());//projection 屏幕的尺寸
        startMiniCap(ROTATE, definition, device.getWidth(), device.getHeight());//启动 minicap工具
        startMiniCapServer();//启动 minicap服务
        return this;
    }

    private void pushMinicapFile(String sourcePath, String sourceSoPath) {
        try {
            logger.info(sourcePath);
            InputStream sourceInStream = this.getClass().getClassLoader().getResourceAsStream(sourcePath);
//            System.out.println(sourceInStream);
            FileUtil.copy(
                    sourceInStream,
                    new FileOutputStream("minicap")
            );
            logger.info(sourceSoPath);
            InputStream sourceSoInStream = this.getClass().getClassLoader().getResourceAsStream(sourceSoPath);
//            System.out.println(sourceSoInStream);
            FileUtil.copy(
                    sourceSoInStream,
                    new FileOutputStream("minicap.so")
            );
        } catch (FileNotFoundException e) {
            e.printStackTrace();
        }
        String goal_path = "/data/local/tmp";
        device.push("minicap", goal_path);
        device.push("minicap.so", goal_path);
        device.chmod_R(String.format("%s/minicap", goal_path));
    }

    //projection 屏幕的尺寸
    void setScreenSize(int rotate, int definition, int width, int high) {
        device.adbCommand(String.format("adb shell LD_LIBRARY_PATH=/data/local/tmp /data/local/tmp/minicap -P %sx%s@%sx%s/%s -Q %s -t", width, high, width, high, rotate, definition)).readClose();
    }

    //启动 minicap工具
    void startMiniCap(int rotate, int definition, int width, int high) {
        minicapServer = device.adbCommand(String.format("adb shell LD_LIBRARY_PATH=/data/local/tmp /data/local/tmp/minicap -P %sx%s@%sx%s/%s -Q %s", width, high, width, high, rotate, definition));
//        minicapServer.readClose();
    }

    //启动 minicap服务
    void startMiniCapServer() {
        device.adbCommand(String.format("adb forward tcp:%s localabstract:minicap", PORT)).readClose();
    }

    void startSocket() {
        try {
            SOCKET = new Socket("localhost", PORT);
            ATREAM = SOCKET.getInputStream();
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    void stopSocket() {
        if (SOCKET != null && SOCKET.isConnected()) {
            try {
                SOCKET.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        if (ATREAM != null) {
            try {
                ATREAM.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
        new Thread() {
            public void run() {
                if (minicapServer != null) {
                    minicapServer.readClose();
                }
            }
        }.start();
    }


    public void start() {
        SCREEN_SYNCHRONIZATION_OUPUT = false;
        frame = new Thread(new ImageCollector());
        image_converter = new Thread(new ImageConverter());
        image_out = new Thread(new ImageOut());
        frame.start();
        image_converter.start();
        image_out.start();
        getImageStartChange();
    }

    public void stop() {
        SCREEN_SYNCHRONIZATION_OUPUT = true;
        try {
            Thread.sleep(200);
        } catch (InterruptedException e) {
            e.printStackTrace();
        }
        stopSocket();
        getImageStopChange();
    }

    /**
     * 数据收集器
     */
    class ImageCollector implements Runnable {
        public void run() {
            int i = 0;
            boolean flag = true;
            imageCollectorStartChange();
            while (flag) {
                try {
                    startSocket();
                    Thread.sleep(200);
                    while (!SCREEN_SYNCHRONIZATION_OUPUT) {
                        getBufferadDate();
                    }
                    flag = false;
                } catch (Exception e) {
                    flag = true;
                    if (i == 2) break;//重试次数限制
                    i++;
                }
            }
            SCREEN_SYNCHRONIZATION_OUPUT = true;
            imageCollectorStopChange();
        }
    }

    /**
     * 图像生成器
     */
    class ImageConverter implements Runnable {
        public void run() {
            imageConverterStartChange();
            while (!SCREEN_SYNCHRONIZATION_OUPUT) {
                getImegaData();
            }
            imageConverterStopChange();
        }
    }

    /**
     * 图像处理器
     */
    class ImageOut implements Runnable {
        public void run() {
            while (!SCREEN_SYNCHRONIZATION_OUPUT) {
                if (!IMGDATA_QUEUE.isEmpty()) {
                    new Thread(new Runnable() {
                        public void run() {
                            BufferedImage bufferedImage = createImageFromByte(IMGDATA_QUEUE.poll());
                            if (bufferedImage != null) {
                                imageOutChange(bufferedImage);
                            }
                        }
                    }).start();
                }
            }
            try {
                Thread.sleep(500);
            } catch (InterruptedException e) {
            }
            imageOutStopChange();
        }
    }

    // byte数组处理
    byte[] subByteArray(byte[] byte1, int start, int end) {
        byte[] byte2 = new byte[0];
        try {
            byte2 = new byte[end - start];
        } catch (NegativeArraySizeException e) {
            e.printStackTrace();
        }
        System.arraycopy(byte1, start, byte2, 0, end - start);
        return byte2;
    }

    // // byte数组处理：合并
    static byte[] byteMerger(byte[] byte_1, byte[] byte_2) {
        byte[] byte_3 = new byte[byte_1.length + byte_2.length];
        System.arraycopy(byte_1, 0, byte_3, 0, byte_1.length);
        System.arraycopy(byte_2, 0, byte_3, byte_1.length, byte_2.length);
        return byte_3;
    }

    // 获取ImageByte
    BufferedImage createImageFromByte(byte[] binaryData) {
        if (binaryData == null) return null;
        BufferedImage bufferedImage = null;
        InputStream in = new ByteArrayInputStream(binaryData);
        try {
            bufferedImage = ImageIO.read(in);
            if (bufferedImage == null) {
                System.out.println("bufferedImage is null!");
            }
        } catch (IOException e) {
            e.printStackTrace();
        } finally {
            if (in != null) {
                try {
                    in.close();
                } catch (IOException e) {
                    // TODO Auto-generated catch block
                    e.printStackTrace();
                }
            }
        }
        return bufferedImage;
    }

    //图像数据获取
    void getBufferadDate() throws Exception {
        byte[] buffer = new byte[len];
        realLen = ATREAM.read(buffer);
        if (buffer.length != realLen) {
            buffer = subByteArray(buffer, 0, realLen);
        }
        DATA_QUEUE.add(buffer);
    }


    //获取转化图片数据
    void getImegaData() {
        if (!DATA_QUEUE.isEmpty()) {
            byte[] buffer = DATA_QUEUE.poll();
            int len = buffer.length;
            for (int cursor = 0; cursor < len; ) {
                int byte10 = buffer[cursor] & 0xff;
                if (readBannerBytes < bannerLength) {
                    cursor = parserHeader(cursor, byte10);
                } else if (readFrameBytes < 4) {
                    // 第二次的缓冲区中前4位数字和为frame的缓冲区大小
                    frameBodyLength += (byte10 << (readFrameBytes * 8)) >>> 0;
                    cursor += 1;
                    readFrameBytes += 1;//解析图片大小
                } else {
                    if (len - cursor >= frameBodyLength) {
                        byte[] subByte = subByteArray(buffer, cursor, cursor + frameBodyLength);
                        frameBody = byteMerger(frameBody, subByte);
                        if ((frameBody[0] != -1) || frameBody[1] != -40) {
                            System.out.println(String.format("Frame body does not startScreenProjection with JPG header"));
                            return;
                        }
                        final byte[] finalBytes = subByteArray(frameBody, 0, frameBody.length);
                        IMGDATA_QUEUE.add(finalBytes);
                        cursor += frameBodyLength;
                        restore();
                    } else {
//                      System.out.println("所需数据大小 : " + frameBodyLength);
                        byte[] subByte = subByteArray(buffer, cursor, len);
                        frameBody = byteMerger(frameBody, subByte);
                        frameBodyLength -= (len - cursor);
                        readFrameBytes += (len - cursor);
                        cursor = len;
                    }
                }
            }
        }
    }

    int parserHeader(int cursor, int byte10) {
        switch (readBannerBytes) {
            case 0:
                // version
                header.setVersion(byte10);
                break;
            case 1:
                // length
                bannerLength = byte10;
                header.setLength(byte10);
                break;
            case 2:
            case 3:
            case 4:
            case 5:
                // pid
                int pid = header.getPid();
                pid += (byte10 << ((readBannerBytes - 2) * 8)) >>> 0;
                header.setPid(pid);
                break;
            case 6:
            case 7:
            case 8:
            case 9:
                // real width
                int realWidth = header.getReadWidth();
                realWidth += (byte10 << ((readBannerBytes - 6) * 8)) >>> 0;
                header.setReadWidth(realWidth);
                break;
            case 10:
            case 11:
            case 12:
            case 13:
                // real height
                int realHeight = header.getReadHeight();
                realHeight += (byte10 << ((readBannerBytes - 10) * 8)) >>> 0;
                header.setReadHeight(realHeight);
                break;
            case 14:
            case 15:
            case 16:
            case 17:
                // virtual width
                int virtualWidth = header.getVirtualWidth();
                virtualWidth += (byte10 << ((readBannerBytes - 14) * 8)) >>> 0;
                header.setVirtualWidth(virtualWidth);
                break;
            case 18:
            case 19:
            case 20:
            case 21:
                // virtual height
                int virtualHeight = header.getVirtualHeight();
                virtualHeight += (byte10 << ((readBannerBytes - 18) * 8)) >>> 0;
                header.setVirtualHeight(virtualHeight);
                break;
            case 22:
                // orientation
                header.setOrientation(byte10 * 90);
                break;
            case 23:
                // quirks
                header.setQuirks(byte10);
                break;
        }
        cursor += 1;
        readBannerBytes += 1;
        if (readBannerBytes == bannerLength) {
            System.out.println(header.toString());
        }
        return cursor;
    }

    void restore() {
        frameBodyLength = 0;
        readFrameBytes = 0;
        frameBody = new byte[0];
    }
}