package io.github.ibony.ats.report;

import io.github.ibony.ats.utils.FileUtil;
import org.json.JSONObject;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created by bony on 2020-07-23.
 */
public class Report {
    private String reportName;
    private String REPORT_TEMPLATE_PATH = "res" + File.separator + "report_template";
//    private String REPORT_TEMPLATE_INDEX_PATH = REPORT_TEMPLATE_PATH + File.separator + "index.html";

    private String REPORT_TEMPLATE_INDEX_PATH = "report_template/index.html";
//    private String REPORT_TEMPLATE_MODEL_PATH = "report_template/model.html";
    private String REPORT_ROOT = "report";
    private String REPORT_PATH = REPORT_ROOT + File.separator + "TEST_" + System.currentTimeMillis();
    private String REPORT_LOG_PATH = REPORT_PATH + File.separator + "logs";
    private String REPORT_ANNEX_ROOT = REPORT_PATH + File.separator + "annex" + File.separator;
    private String REPORT_INDEX_HTML_PATH = REPORT_PATH + File.separator + "index.html";
//    private String REPORT_MODEL_HTML_PATH = REPORT_PATH + File.separator + "model.html";
    private String REPORT_INDEX_HTML_CONTENT = "";
    private String reportIntro = "";//报告简介
    private List<String> logFileNames = new ArrayList<>();//日志文件名称
    private List<CaseInfo> caseInfoList = new ArrayList<>();
    private List<ModuleStats> moduleStatsList = new ArrayList<>();
    private long totalNumber;
    private long passNumber;
    private long failNumber;
    AddCaseListener addCaseListener=null;

    public void setAddCaseListener(AddCaseListener addCaseListener) {
        this.addCaseListener = addCaseListener;
    }

    /**
     * 测试报告构造函数
     *
     * @param reportName 报告名称
     * @param rootPath   报告跟命令
     * @param dirName    报告文件夹
     */
    public Report(String rootPath, String dirName, String reportName) {
        REPORT_ROOT = rootPath;
        REPORT_PATH = REPORT_ROOT + File.separator + dirName;
        REPORT_ANNEX_ROOT = REPORT_PATH + File.separator + "annex" + File.separator;
        REPORT_LOG_PATH = REPORT_PATH + File.separator + "logs";
        REPORT_INDEX_HTML_PATH = REPORT_PATH + File.separator + "index.html";
        initReport(reportName);
    }

    /**
     * 测试报告构造函数
     *
     * @param dirName    文件名称
     * @param reportName 报告名称
     */
    public Report(String dirName, String reportName) {
        REPORT_PATH = REPORT_ROOT + File.separator + dirName;
        REPORT_ANNEX_ROOT = REPORT_PATH + File.separator + "annex" + File.separator;
        REPORT_LOG_PATH = REPORT_PATH + File.separator + "logs";
        REPORT_INDEX_HTML_PATH = REPORT_PATH + File.separator + "index.html";
        initReport(reportName);
    }

    /**
     * 测试报告构造函数
     *
     * @param reportName 报告名称
     */
    public Report(String reportName) {
        initReport(reportName);
    }


    /**
     * 测试报告构造函数
     */
    public Report() {
        initReport("TEST_" + System.currentTimeMillis() + "-测试报告");
    }

    /**
     * 测试报告初始化
     *
     * @param reportName 报告名称
     */
    private void initReport(String reportName) {
        this.reportName = reportName;
        this.totalNumber = 0;
        this.passNumber = 0;
        this.failNumber = 0;
        FileUtil.checkNew(new File(REPORT_PATH));
        FileUtil.checkNew(new File(REPORT_ANNEX_ROOT));
        FileUtil.checkNew(new File(REPORT_LOG_PATH));
        try {
            FileUtil.writeContent(REPORT_INDEX_HTML_PATH, FileUtil.readJarFileContent(
                    this.getClass().getClassLoader().getResourceAsStream(REPORT_TEMPLATE_INDEX_PATH)));
        } catch (IOException e) {
            e.printStackTrace();
        }
//        try {
//            FileUtil.writeContent(REPORT_MODEL_HTML_PATH, FileUtil.readJarFileContent(
//                    this.getClass().getClassLoader().getResourceAsStream(REPORT_TEMPLATE_MODEL_PATH)));
//        } catch (IOException e) {
//            e.printStackTrace();
//        }
        try {
            REPORT_INDEX_HTML_CONTENT = FileUtil.readContent(REPORT_INDEX_HTML_PATH);
            String reportContent = REPORT_INDEX_HTML_CONTENT.replaceAll("\\{\\{report_info\\}\\}", this.toString());
            FileUtil.writeContent(REPORT_INDEX_HTML_PATH, reportContent);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * 往测试报告中添加一条测试用例信息
     *
     * @param caseInfo 用例信息
     */
    public void addCaseInfo(CaseInfo caseInfo) {
        caseInfoList.add(caseInfo);
        calcModuleStats(caseInfo);
        upReportCaseData();
        if(this.addCaseListener!=null){
            this.addCaseListener.handler(caseInfo);
        }
    }

    /**
     * 往测试报告中添加一条测试用例信息
     *
     * @param caseInfo          用例信息
     * @param annexRelativePath 附件相对地址(截图、日志)
     * @return annexAbsolutelyPath 附件绝对对地址(截图、日志)
     */
    public String addCaseInfo(CaseInfo caseInfo, String annexRelativePath) {
        String htmlAnnexPath = "annex" + File.separator + annexRelativePath;
        htmlAnnexPath = htmlAnnexPath.replaceAll("\\\\", "\\\\\\\\");
        caseInfo.setAnnexPath(htmlAnnexPath);
        caseInfoList.add(caseInfo);
        calcModuleStats(caseInfo);
        upReportCaseData();
        if(this.addCaseListener!=null){
            this.addCaseListener.handler(caseInfo);
        }
        String annexAbsolutelyPath = REPORT_ANNEX_ROOT + annexRelativePath;
        return annexAbsolutelyPath;
    }

    /**
     * 关于模块的统计更新
     *
     * @param caseInfo 用例信息
     */
    private void calcModuleStats(CaseInfo caseInfo) {
        List<String> nameList = moduleStatsList.stream().map(ModuleStats::getName).collect(Collectors.toList());
        if (nameList.contains(caseInfo.getModule())) {
            for (ModuleStats moduleStats : moduleStatsList) {
                if (caseInfo.getModule().equals(moduleStats.getName())) {
                    moduleStats.calcAdd(caseInfo);
                }
            }
        } else {
            moduleStatsList.add(new ModuleStats(caseInfo));
        }
    }

    /**
     * 往测试报告中添加一条测试用例信息
     *
     * @param logFileName 日志文件名称
     * @return annexAbsolutelyPath 日志文件绝对对地址
     */
    public String addLogFileName(String logFileName) throws IOException {
        this.logFileNames.add(logFileName.replaceAll("\\\\", "\\\\\\\\"));
        String logPath = REPORT_LOG_PATH + File.separator + logFileName;
        File logFile = new File(logPath);
        if (!logFile.exists()) {
            logFile.createNewFile();
        }
        return logPath;
    }

    /**
     * 更新测试报告中的报告
     */
    private void upReportCaseData() {
        try {
            String reportContent = REPORT_INDEX_HTML_CONTENT.replaceAll("\\{\\{report_info\\}\\}", this.toString());
            FileUtil.writeContent(REPORT_INDEX_HTML_PATH, reportContent);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    public String getReportPath() {
        return REPORT_PATH;
    }

    public String getReportAnnexRoot() {
        return REPORT_ANNEX_ROOT;
    }

    /**
     * 设置报告简介
     *
     * @param intro 简介信息
     */
    public void setIntro(String intro) {
        this.reportIntro = intro;
        try {
            String reportContent = REPORT_INDEX_HTML_CONTENT.replaceAll("\\{\\{report_info\\}\\}", this.toString());
            FileUtil.writeContent(REPORT_INDEX_HTML_PATH, reportContent);
        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * @return 返回case列表
     */
    public List<CaseInfo> getCaseList() {
        return caseInfoList;
    }

    JSONObject toJsonObject() {
        JSONObject json = new JSONObject();
        json.put("reportIntro", reportIntro);
        json.put("reportName", reportName);
        json.put("totalNumber", totalNumber);
        json.put("passNumber", passNumber);
        json.put("failNumber", failNumber);
        json.put("caseList", caseInfoList.toArray());
        json.put("moduleStatsList", moduleStatsList.toArray());
        json.put("logFileNames", logFileNames.toArray());
        return json;
    }

    @Override
    public String toString() {
        return toJsonObject().toString();
    }

    /**
     * @return 报告名称
     */
    public String getReportName() {
        return reportName;
    }

    /**
     * @return 屏保简介
     */
    public String getReportIntro() {
        return reportIntro;
    }

    /**
     * @return 模块统计详情
     */
    public List<ModuleStats> getModuleStatsList() {
        return moduleStatsList;
    }

    /**
     * @return case总数
     */
    public long getTotalNumber() {
        return totalNumber;
    }

    /**
     * @return 通过总数
     */
    public long getPassNumber() {
        return passNumber;
    }

    /**
     * @return 失败总数
     */
    public long getFailNumber() {
        return failNumber;
    }
}
