package io.github.ibony.ats.utils;

import org.apache.tools.zip.ZipEntry;
import org.apache.tools.zip.ZipFile;
import org.apache.tools.zip.ZipFile;
import org.apache.commons.compress.archivers.sevenz.SevenZArchiveEntry;
import org.apache.commons.compress.archivers.sevenz.SevenZFile;

import java.io.*;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;

/**
 * 文件操作类
 * Created by bony on 2017-05-06.
 */
public class FileUtil {

    /**
     * 读文件
     *
     * @param path 路径
     * @return String 内容
     * @throws IOException IO异常抛出
     */
    public static String readContent(String path) throws IOException {
//        System.out.println("read path:" + path);
        ArrayList<String> contentlist = FileUtil.readLines(path);
        String ret = "";
        for (String content : contentlist) {
            ret += content;
        }
        return ret;
    }

    /**
     * 读文件
     *
     * @param path 路径
     * @return String 内容
     * @throws IOException IO异常抛出
     */
    public static String readContent(File path) throws IOException {
        ArrayList<String> contentlist = FileUtil.readLines(path);
        String ret = "";
        for (String content : contentlist) {
            ret += content;
        }
        return ret;
    }

    /**
     * 分行读文件到list
     *
     * @param path 路径
     * @return ArrayList  行内容
     * @throws IOException IO异常抛出
     */
    public static ArrayList<String> readLines(String path) throws IOException {
        File file = new File(path);
        return readLines(file);
    }

    /**
     * 分行读文件到list
     *
     * @param file 文件
     * @return ArrayList  行内容
     * @throws IOException IO异常抛出
     */
    public static ArrayList<String> readLines(File file) throws IOException {
        ArrayList<String> List;
        List = new ArrayList<String>();
        BufferedReader bufferedReader = new BufferedReader(new FileReader(file));
        String read = null;
        while ((read = bufferedReader.readLine()) != null) {
            List.add(read + "\n");
        }
        bufferedReader.close();
        return List;
    }

    /**
     * 覆盖写文件
     *
     * @param path    文件路径
     * @param content 内容
     * @throws IOException IO异常抛出
     */
    public static void writeContent(String path, String content) throws IOException {
        write(path, content, false);
    }

    /**
     * 追加写文件
     *
     * @param path 文件路径
     * @param line 内容
     * @throws IOException IO异常抛出
     */
    public static void writeLine(String path, String line) throws IOException {
        write(path, line, true);
    }

    public static void write(String path, String content, boolean append) throws IOException {
        File file = new File(path);
        if (!file.exists()) {
            file.createNewFile();
        }
        FileWriter fileWritter = new FileWriter(file, append);
        fileWritter.write(content);
        fileWritter.flush();
        if (fileWritter != null) {
            fileWritter.close();
        }
    }

    /**
     * 获取文件夹下的文件列表名称
     *
     * @param path 文件夹路径
     * @return 文件列表名称
     */
    public static ArrayList<String> getDirNames(String path) {
        ArrayList<String> dirNames = new ArrayList<String>();
        File file = new File(path);
        if (!file.exists()) {
            return null;
        }
        File[] tempList = file.listFiles();
        for (int i = 0; i < tempList.length; i++) {
            dirNames.add(tempList[i].getName());
        }
        return dirNames;
    }

    /**
     * 获取路径下的所有后缀为.suffix文件名称
     *
     * @param path   文件夹路径
     * @param suffix 文件后缀名
     * @return 文件名称列表
     */
    public static ArrayList<String> getDirFiles(String path, String suffix) {
        return getDirFiles(new File(path), suffix);
    }

    /**
     * 获取路径下的所有后缀为.suffix文件名称
     *
     * @param file   文件夹路径
     * @param suffix 文件后缀名
     * @return 文件名称列表
     */
    public static ArrayList<String> getDirFiles(File file, String suffix) {
        ArrayList<String> filearrays = new ArrayList<String>();
        checkNew(file);
        String[] files = null;
        if (file != null) {
            files = file.list();
            if (files != null) {
                for (String file1 : files) {
                    int dian = file1.lastIndexOf(".");//.最后出现的位置
                    if (suffix == null) {
                        filearrays.add(file1);
                    } else {
                        if (dian >= 0) {
                            String name = file1.substring(0, dian).trim();
                            String type = file1.substring(dian + 1, file1.length()).trim();
                            if (type != null && type.equals(suffix)) {
                                filearrays.add(name);
                            }
                        }
                    }
                }
            }
        }
        if (filearrays.size() == 0) {
//            System.out.println("路径:" + file.getPath() + " 下不存在." + suffix + "文件");
        }
        return filearrays;
    }


    /**
     * 获取路径下的所有后缀为.suffix文件名称
     *
     * @param file   文件夹
     * @param suffix 文件后缀名
     * @return 文件列表
     */
    public static ArrayList<File> getDirectoryFiles(File file, String suffix) {
        ArrayList<File> filearrays = new ArrayList<File>();
        checkNew(file);
        if (file != null) {
            File[] files = file.listFiles();
            for (File dirFile : files) {
                if (dirFile.isDirectory()) {
                    filearrays.addAll(getDirectoryFiles(dirFile, suffix));
                } else {
                    filearrays.add(dirFile);
                }
            }
        }
        return filearrays;
    }

    /**
     * 获取路径下的所有后缀为.suffix文件名称
     *
     * @param path   文件夹
     * @param suffix 文件后缀名
     * @return 文件列表
     */
    public static ArrayList<File> getDirectoryFiles(String path, String suffix) {
        return getDirectoryFiles(new File(path), suffix);
    }

    /**
     * 路径不存在则新建
     *
     * @param path 路径
     */
    public static void checkNew(File path) {
        if (!path.exists()) {
            path.mkdirs();
        }
    }

    /**
     * 复制文件方法
     *
     * @param inStream  源文件
     * @param outStream 目标路径
     */
    public static void copy(InputStream inStream, OutputStream outStream) {
        try {
            byte data[] = new byte[1024 * 8];//创建搬运工具
            int len = 0;//创建长度
            while ((len = inStream.read(data)) != -1)//循环读取数据
            {
                outStream.write(data, 0, len);
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                inStream.close();//释放资源
                outStream.close();//释放资源
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    /**
     * @param path 文件路径
     * @return 结果
     * @throws IOException 异常
     */
    public static String readJarFileContent(InputStream path) throws IOException {
        BufferedReader in = new BufferedReader(new InputStreamReader(path));
        StringBuffer buffer = new StringBuffer();
        String content = "";
        String line = "";
        while ((line = in.readLine()) != null) {
            content += line + "\r\n";
        }
        return content;
    }

    /**
     * 路径存在则删除
     *
     * @param path 路径
     */
    public static void checkDel(File path) {
        if (path.exists()) {
            path.delete();
        }
    }

    /**
     * 删除文件夹
     *
     * @param dirFile 路径
     */
    public static void deleteDirectory(File dirFile) {
        if (dirFile.isFile()) {
            dirFile.delete();//清理文件
        } else {
            File list[] = dirFile.listFiles();
            if (list != null) {
                for (File f : list) {
                    deleteDirectory(f);
                }
                dirFile.delete();//清理目录
            }
        }
    }

    /**
     * 删除文件夹
     *
     * @param dirFile 路径
     */
    public static void cleanDirectory(File dirFile) {
        File list[] = dirFile.listFiles();
        if (list != null) {
            for (File f : list) {
                deleteDirectory(f);
            }
        }
    }

    /**
     * 判断文件是否存在
     *
     * @param path 路径
     * @return 判断结果
     */
    public static boolean checkPath(File path) {
        if (!path.exists()) return false;
        return true;
    }

    /**
     * 删除文件，可以是文件或文件夹
     *
     * @param path 要删除的文件名
     * @return 删除成功返回true，否则返回false
     */
    public static boolean del(String path) {
        File file = new File(path);
        if (!file.exists()) {
//            System.out.println("删除文件失败:" + path + "不存在！");
            return false;
        } else {
            if (file.isFile())
                return deleteFile(path);
            else
                return deleteDirectory(path, true);
        }
    }

    /**
     * 情况文件夹
     *
     * @param path 要删除的文件名
     * @return 删除成功返回true，否则返回false
     */
    public static boolean clear(String path) {
        File file = new File(path);
        if (!file.exists()) {
//            System.out.println("删除文件失败:" + path + "不存在！");
            return false;
        } else {
            if (file.isFile()) {
                try {
                    writeContent(path, "");
                    return true;
                } catch (IOException e) {
                    e.printStackTrace();
                    return false;
                }
            } else {
                return deleteDirectory(path, false);
            }

        }
    }

    /**
     * 删除单个文件
     *
     * @param path 要删除的文件的文件名
     * @return 单个文件删除成功返回true，否则返回false
     */
    public static boolean deleteFile(String path) {
        File file = new File(path);
        // 如果文件路径所对应的文件存在，并且是一个文件，则直接删除
        if (file.exists() && file.isFile()) {
            if (file.delete()) {
                return true;
            } else {
                return false;
            }
        } else {
            return false;
        }
    }

    /**
     * 判断文件是否存在
     *
     * @param path 文件路径
     * @return 存在true，否则返回false
     */
    public static boolean fileExists(String path) {
        File file = new File(path);
        // 如果文件路径所对应的文件存在，并且是一个文件，则直接删除
        if (file.exists() && file.isFile()) {
            return true;
        } else {
            return false;
        }
    }

    /**
     * 判断文件目录是否为空
     *
     * @param dirPath 目录路径
     * @return 判断结果
     */
    public static boolean isDirEmpty(String dirPath) {
        File dir = new File(dirPath);
        if (dir != null && dir.exists() && dir.isDirectory()) {
            File[] files = dir.listFiles();
            if (files != null && files.length > 0) {
                return false;
            } else {
                return true;
            }
        } else {
            return true;
        }
    }

    /**
     * 对.zip文件进行解压缩
     *
     * @param zipFile 解压缩文件
     * @param descDir 压缩的目标地址
     * @return 文件列表
     * @throws Exception 异常
     */
    @SuppressWarnings("rawtypes")
    public static List<File> unzipFile(String zipFile, String descDir) throws Exception {
        byte[] _byte = new byte[1024];
        List<File> _list = new ArrayList<File>();

        ZipFile _zipFile = new ZipFile(new File(zipFile), "GBK");
        for (Enumeration entries = _zipFile.getEntries(); entries.hasMoreElements(); ) {
            ZipEntry entry = (ZipEntry) entries.nextElement();
            File _file = new File(descDir + File.separator + entry.getName());
            if (entry.isDirectory()) {
                _file.mkdirs();
            } else {
                File _parent = _file.getParentFile();
                if (!_parent.exists()) {
                    _parent.mkdirs();
                }
                InputStream _in = _zipFile.getInputStream(entry);
                OutputStream _out = new FileOutputStream(_file);
                int len = 0;
                while ((len = _in.read(_byte)) > 0) {
                    _out.write(_byte, 0, len);
                }
                _in.close();
                _out.flush();
                _out.close();
                _list.add(_file);
            }
        }
        _zipFile.close();
        return _list;
    }

    /**
     * zip解压
     *
     * @param inputFile   待解压文件名
     * @param destDirPath 解压路径
     * @throws Exception 异常处理
     */
    public static void un7z(String inputFile, String destDirPath) throws Exception {
        File srcFile = new File(inputFile);//获取当前压缩文件
        // 判断源文件是否存在
        if (!srcFile.exists()) {
            throw new Exception(srcFile.getPath() + "所指文件不存在");
        }
        //开始解压
        SevenZFile zIn = new SevenZFile(srcFile);
        SevenZArchiveEntry entry = null;
        File file = null;
        while ((entry = zIn.getNextEntry()) != null) {
            if (!entry.isDirectory()) {
                file = new File(destDirPath, entry.getName());
                if (!file.exists()) {
                    new File(file.getParent()).mkdirs();//创建此文件的上级目录
                }
                OutputStream out = new FileOutputStream(file);
                BufferedOutputStream bos = new BufferedOutputStream(out);
                int len = -1;
                byte[] buf = new byte[1024];
                while ((len = zIn.read(buf)) != -1) {
                    bos.write(buf, 0, len);
                }
                // 关流顺序，先打开的后关闭
                bos.close();
                out.close();
            }
        }
        zIn.close();
    }


    /**
     * 删除目录及目录下的文件
     *
     * @param dir     要删除的目录的文件路径
     * @param delthis 是否删除当前目录
     * @return 目录删除成功返回true，否则返回false
     */
    public static boolean deleteDirectory(String dir, boolean delthis) {
        // 如果dir不以文件分隔符结尾，自动添加文件分隔符
        if (!dir.endsWith(File.separator))
            dir = dir + File.separator;
        File dirFile = new File(dir);
        // 如果dir对应的文件不存在，或者不是一个目录，则退出
        if ((!dirFile.exists()) || (!dirFile.isDirectory())) {
//            System.out.println("删除目录失败：" + dir + "不存在！");
            return false;
        }
        boolean flag = true;
        // 删除文件夹中的所有文件包括子目录
        File[] files = dirFile.listFiles();
        for (int i = 0; i < files.length; i++) {
            // 删除子文件
            if (files[i].isFile()) {
                flag = deleteFile(files[i].getAbsolutePath());
                if (!flag)
                    break;
            }
            // 删除子目录
            else if (files[i].isDirectory()) {
                flag = deleteDirectory(files[i].getAbsolutePath(), true);
                if (!flag)
                    break;
            }
        }
        if (delthis) {
            flag = dirFile.delete();
        }
        return flag;
    }


    /**
     * 新建文件
     *
     * @param file     文件
     * @param filePath 路径
     * @param fileName 名称
     * @throws Exception 处理异常
     */
    public static void uploadFile(byte[] file, String filePath, String fileName) throws Exception {
        File targetFile = new File(filePath);
        if (!targetFile.exists()) {
            targetFile.mkdirs();
        }
        FileOutputStream out = new FileOutputStream(filePath + fileName);
        out.write(file);
        out.flush();
        out.close();
    }
}