package io.github.javpower.vectorex.keynote.analysis;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

public class DictionaryManager {
    private static volatile DictionaryManager instance;
    private final Map<String, Double> wordFrequencies = new HashMap<>();
    private final Set<String> loadedPaths = new HashSet<>();
    private final TrieSegment trie;

    private DictionaryManager() {
        trie = new TrieSegment((char) 0);
        loadMainDictionary();
    }

    public static DictionaryManager getInstance() {
        if (instance == null) {
            synchronized (DictionaryManager.class) {
                if (instance == null) {
                    instance = new DictionaryManager();
                }
            }
        }
        return instance;
    }

    private void loadMainDictionary() {
        // 加载主词典逻辑
        String mainDictPath = "/dict.txt";
        loadDictionary(mainDictPath);
    }

    public void loadUserDictionary(String path) {
        if (loadedPaths.contains(path)) {
            return;
        }
        synchronized (this) {
            if (loadedPaths.contains(path)) {
                return;
            }
            loadDictionary(path);
            loadedPaths.add(path);
        }
    }

    private void loadDictionary(String path) {
        try (BufferedReader reader = new BufferedReader(new InputStreamReader(
                this.getClass().getResourceAsStream(path), Charset.forName("UTF-8")))) {
            String line;
            while ((line = reader.readLine()) != null) {
                String[] tokens = line.split("[\t ]+");
                if (tokens.length < 1) continue;
                String word = tokens[0];
                double freq = tokens.length == 2 ? Double.parseDouble(tokens[1]) : 3.0;
                addWord(word, freq);
            }
        } catch (IOException e) {
            System.err.println("Failed to load dictionary: " + path);
        }
    }

    private void addWord(String word, double freq) {
        word = word.trim().toLowerCase();
        char[] chars = word.toCharArray();
        trie.fillSegment(chars, 0, chars.length, 1);
        wordFrequencies.put(word, Math.log(freq));
    }

    public boolean containsWord(String word) {
        return wordFrequencies.containsKey(word);
    }

    public double getFrequency(String word) {
        return wordFrequencies.getOrDefault(word, Double.NEGATIVE_INFINITY);
    }

    public TrieSegment getTrie() {
        return trie;
    }
}