package io.github.javpower.vectorex.keynote.analysis;

import java.util.Objects;

/**
 * 通用加权项类，表示一个实体及其权重。
 * 该类实现了Comparable接口，以便根据权重对实体进行排序。
 */
public class ScoredEntity<T> implements Comparable<ScoredEntity<T>> {

    // 实体（例如关键词、文档ID等）
    private final T item;

    // 权重（例如TF-IDF值、BM25分数等）
    private final double score;

    /**
     * 构造方法，用于创建一个WeightedItem对象。
     *
     * @param item   实体
     * @param weight 权重，将四舍五入保留四位小数
     */
    public ScoredEntity(T item, double weight) {
        this.item = item;
        this.score = Math.round(weight * 10000.0) / 10000.0; // 保留四位小数
    }

    /**
     * 获取实体。
     *
     * @return 实体
     */
    public T getItem() {
        return item;
    }

    /**
     * 获取权重。
     *
     * @return 权重
     */
    public double getScore() {
        return score;
    }

    /**
     * 比较当前加权项与其他加权项的权重。
     * 用于实现Comparable接口，以便对加权项进行排序。
     *
     * @param other 另一个加权项对象
     * @return 如果当前加权项的权重小于另一个加权项的权重，返回正整数；
     *         如果相等，返回0；否则返回负整数。
     */
    @Override
    public int compareTo(ScoredEntity<T> other) {
        return Double.compare(other.score, this.score); // 降序排列
    }

    /**
     * 计算加权项对象的哈希码。
     *
     * @return 加权项对象的哈希码
     */
    @Override
    public int hashCode() {
        return Objects.hash(item, score);
    }

    /**
     * 判断当前加权项对象是否与另一个对象相等。
     *
     * @param obj 要比较的对象
     * @return 如果两个对象相等，返回true；否则返回false
     */
    @Override
    public boolean equals(Object obj) {
        if (this == obj) return true; // 如果是同一个对象，直接返回true
        if (obj == null || getClass() != obj.getClass()) return false; // 如果对象为null或类型不同，返回false
        ScoredEntity<?> that = (ScoredEntity<?>) obj; // 强制类型转换
        return Double.compare(that.score, this.score) == 0 && Objects.equals(this.item, that.item); // 比较权重和实体
    }

    /**
     * 返回加权项对象的字符串表示形式。
     *
     * @return 加权项对象的字符串表示形式
     */
    @Override
    public String toString() {
        return "ScoredEntity{" +
                "item=" + item +
                ", score=" + score +
                '}';
    }
}