package io.github.javpower.vectorex.keynote.analysis;

import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

public class TrieSegment implements Comparable<TrieSegment> {
    private static final Map<Character, Character> charMap = new HashMap<>();
    private static final int ARRAY_SIZE_LIMIT = 3;

    private final Character nodeChar;
    private Map<Character, TrieSegment> childrenMap;
    private TrieSegment[] childrenArray;
    private int storeSize = 0;
    private int nodeState = 0; // 0: not a word, 1: is a word

    public TrieSegment(Character nodeChar) {
        if (nodeChar == null) {
            throw new IllegalArgumentException("Node character cannot be null");
        }
        this.nodeChar = nodeChar;
    }

    public Character getNodeChar() {
        return nodeChar;
    }

    public boolean hasNextNode() {
        return storeSize > 0;
    }

    public Hit match(char[] charArray, int begin, int length, Hit searchHit) {
        if (searchHit == null) {
            searchHit = new Hit();
            searchHit.setBegin(begin);
        } else {
            searchHit.setUnmatch();
        }
        searchHit.setEnd(begin);

        Character keyChar = charArray[begin];
        TrieSegment segment = findSegment(keyChar);

        if (segment != null) {
            if (length > 1) {
                return segment.match(charArray, begin + 1, length - 1, searchHit);
            } else if (length == 1) {
                if (segment.nodeState == 1) {
                    searchHit.setMatch();
                }
                if (segment.hasNextNode()) {
                    searchHit.setPrefix();
                    searchHit.setMatchedSegment(segment);
                }
            }
        }
        return searchHit;
    }

    private TrieSegment findSegment(Character keyChar) {
        if (childrenArray != null) {
            TrieSegment keySegment = new TrieSegment(keyChar);
            int position = Arrays.binarySearch(childrenArray, 0, storeSize, keySegment);
            if (position >= 0) {
                return childrenArray[position];
            }
        } else if (childrenMap != null) {
            return childrenMap.get(keyChar);
        }
        return null;
    }

    public void fillSegment(char[] charArray, int begin, int length, int state) {
        Character keyChar = charArray[begin];
        Character mappedChar = charMap.computeIfAbsent(keyChar, k -> keyChar);

        TrieSegment segment = findSegment(mappedChar);
        if (segment == null) {
            segment = new TrieSegment(mappedChar);
            addSegment(segment);
        }

        if (length > 1) {
            segment.fillSegment(charArray, begin + 1, length - 1, state);
        } else {
            segment.nodeState = state;
        }
    }

    private void addSegment(TrieSegment segment) {
        if (storeSize <= ARRAY_SIZE_LIMIT) {
            if (childrenArray == null) {
                childrenArray = new TrieSegment[ARRAY_SIZE_LIMIT];
            }
            if (storeSize == ARRAY_SIZE_LIMIT) {
                migrateToMap();
            } else {
                childrenArray[storeSize++] = segment;
                Arrays.sort(childrenArray, 0, storeSize);
            }
        } else {
            if (childrenMap == null) {
                childrenMap = new HashMap<>();
                migrateToMap();
            }
            childrenMap.put(segment.getNodeChar(), segment);
        }
    }

    private void migrateToMap() {
        if (childrenMap == null) {
            childrenMap = new HashMap<>();
        }
        if (childrenArray != null) {
            for (int i = 0; i < storeSize; i++) {
                if (childrenArray[i] != null) {
                    childrenMap.put(childrenArray[i].getNodeChar(), childrenArray[i]);
                }
            }
            childrenArray = null;
        }
    }

    @Override
    public int compareTo(TrieSegment o) {
        return this.nodeChar.compareTo(o.nodeChar);
    }
}