package io.github.javpower.vectorex.keynote.graph.core;

import io.github.javpower.vectorex.keynote.graph.entity.EdgeInfo;
import io.github.javpower.vectorex.keynote.graph.entity.Node;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;

public class AdjacencyCache {
    private final GraphDB graphDB;
    private final Map<String, List<EdgeInfo>> outEdges = new ConcurrentHashMap<>();
    private final Map<String, List<EdgeInfo>> inEdges = new ConcurrentHashMap<>();

    public AdjacencyCache(GraphDB graphDB) {
        this.graphDB = graphDB;
        initializeCache();
    }

    private void initializeCache() {
        graphDB.getAllNodeIds().parallelStream().forEach(nodeId -> {
            updateOutEdges(nodeId);
            updateInEdges(nodeId);
        });
    }

    public void updateCacheForNode(String nodeId) {
        // 清除旧缓存
        outEdges.remove(nodeId);
        inEdges.remove(nodeId);

        // 重新加载数据
        updateOutEdges(nodeId);
        updateInEdges(nodeId);
    }

    private void updateOutEdges(String nodeId) {
        Node node = graphDB.getNode(nodeId);
        if (node != null) {
            List<EdgeInfo> outgoing = node.getOutgoingEdges().stream()
                    .map(rel -> new EdgeInfo(
                            rel.getEndNodeId(),
                            (Double) rel.getProperties().getOrDefault("weight", 1.0)
                    ))
                    .collect(Collectors.toCollection(CopyOnWriteArrayList::new));
            outEdges.put(nodeId, outgoing);
        }
    }

    private void updateInEdges(String nodeId) {
        List<EdgeInfo> incoming = graphDB.getInEdges(nodeId).stream()
                .map(rel -> new EdgeInfo(
                        rel.getStartNodeId(),
                        (Double) rel.getProperties().getOrDefault("weight", 1.0)
                ))
                .collect(Collectors.toCollection(CopyOnWriteArrayList::new));
        inEdges.put(nodeId, incoming);
    }

    public List<EdgeInfo> getOutEdges(String nodeId) {
        return outEdges.getOrDefault(nodeId, Collections.emptyList());
    }

    public List<EdgeInfo> getInEdges(String nodeId) {
        return inEdges.getOrDefault(nodeId, Collections.emptyList());
    }
}