package io.github.javpower.vectorex.keynote.graph.test;

import io.github.javpower.vectorex.keynote.graph.core.GraphDB;
import io.github.javpower.vectorex.keynote.graph.entity.AttributeEdge;
import io.github.javpower.vectorex.keynote.graph.entity.Node;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

public class GraphTest {
    public static void main(String[] args) {
        GraphDB db = new GraphDB("test1.db");

        // 创建测试节点 A-F
        Arrays.asList("A", "B", "C", "D", "E", "F", "G").forEach(id -> {
            Node node = new Node(id);
            node.addLabel("TestNode");
            db.addNode(node);
        });

        // 构建带权重的复杂图结构
        List<AttributeEdge> edges = Arrays.asList(
                createEdge(db, "e1", "A", "B", 3),
                createEdge(db, "e2", "A", "D", 1),
                createEdge(db, "e3", "B", "C", 2),
                createEdge(db, "e4", "D", "C", 4),
                createEdge(db, "e5", "C", "E", 5),
                createEdge(db, "e6", "D", "E", 2),
                createEdge(db, "e7", "E", "F", 1),
                createEdge(db, "e8", "B", "F", 7)
        );

        // 测试用例1：A->F最短路径
        testPath(db, "A", "F", Arrays.asList("A", "D", "E", "F"), 4.0);

        // 测试用例2：A->C的最短路径（两条等权重路径）
        testMultiplePaths(db, "A", "C",
                Arrays.asList(
                        Arrays.asList("A", "B", "C"),
                        Arrays.asList("A", "D", "C")
                ),
                5.0
        );

        // 测试用例3：动态修改权重
        edges.get(5).updateWeight(1.0, db); // 修改D->E权重为1
        testPath(db, "A", "F", Arrays.asList("A", "D", "E", "F"), 3.0);

        // 测试用例4：孤立节点可达性
        testPath(db, "A", "G", Collections.emptyList(), Double.POSITIVE_INFINITY);

        // 测试用例5：环路检测（添加A->A的边）
        AttributeEdge loopEdge = createEdge(db, "e9", "A", "A", 2);
        testPath(db, "A", "A", Collections.singletonList("A"), 0.0);

        // 测试用例6：不连通路径
        testPath(db, "G", "A", Collections.emptyList(), Double.POSITIVE_INFINITY);
    }

    private static AttributeEdge createEdge(GraphDB db, String id, String from, String to, double weight) {
        AttributeEdge edge = new AttributeEdge(id, from, to, weight);
        db.addRelationship(edge);
        return edge;
    }

    private static void testPath(GraphDB db, String start, String end,
                                 List<String> expectedPath, double expectedCost) {
        List<String> path = db.shortestPathWithWeights(start, end);
        double actualCost = calculatePathCost(db, path);

        System.out.println("\n=== 测试路径 " + start + " → " + end + " ===");
        System.out.println("预期路径: " + expectedPath + " (成本: " + expectedCost + ")");
        System.out.println("实际路径: " + path + " (成本: " + actualCost + ")");

        boolean pathValid = path.equals(expectedPath);
        boolean costValid = Math.abs(actualCost - expectedCost) < 0.0001;

        if (pathValid && costValid) {
            System.out.println("√√√ 测试通过");
        } else {
            System.out.println("××× 测试失败！");
        }
    }

    private static void testMultiplePaths(GraphDB db, String start, String end,
                                          List<List<String>> possiblePaths, double expectedCost) {
        List<String> path = db.shortestPathWithWeights(start, end);
        double actualCost = calculatePathCost(db, path);

        System.out.println("\n=== 测试多路径 " + start + " → " + end + " ===");
        System.out.println("可能路径: " + possiblePaths);
        System.out.println("实际路径: " + path + " (成本: " + actualCost + ")");

        boolean pathValid = possiblePaths.contains(path);
        boolean costValid = Math.abs(actualCost - expectedCost) < 0.001;

        if (pathValid && costValid) {
            System.out.println("√√√ 测试通过");
        } else {
            System.out.println("××× 测试失败！");
        }
    }

    private static double calculatePathCost(GraphDB db, List<String> path) {
        if (path.isEmpty()) return Double.POSITIVE_INFINITY;

        double total = 0.0;
        for (int i = 0; i < path.size() - 1; i++) {
            String from = path.get(i);
            String to = path.get(i + 1);
            total += db.getNode(from).getOutgoingEdges().stream()
                    .filter(e -> e.getEndNodeId().equals(to))
                    .findFirst()
                    .map(e -> ((Number) e.getProperties().getOrDefault("weight", 1.0)).doubleValue())
                    .orElse(Double.POSITIVE_INFINITY);
        }
        return total;
    }
}