package io.github.javpower.vectorex.keynote.graph.test;

import io.github.javpower.vectorex.keynote.graph.core.GraphDB;
import io.github.javpower.vectorex.keynote.graph.entity.Node;
import io.github.javpower.vectorex.keynote.graph.entity.Relationship;

import java.util.*;
import java.util.stream.Collectors;

public class SocialNetwork {
    private final GraphDB graphDB;

    public SocialNetwork(String dbPath) {
        this.graphDB = new GraphDB(dbPath);
    }

    public static void main(String[] args) {
        SocialNetwork network = new SocialNetwork("social_network.db");

        // 初始化用户
        List<Node> users = Arrays.asList(
                createUser("Alice", "爱丽丝"),
                createUser("Bob", "鲍勃"),
                createUser("Charlie", "查理"),
                createUser("David", "大卫"),
                createUser("Eve", "伊芙"),
                createUser("Frank", "弗兰克")
        );

        // 添加用户到数据库
        for (Node user : users) {
            network.graphDB.addNode(user);
        }

        // 添加好友关系
        network.addFriendship("Alice", "Bob");
        network.addFriendship("Alice", "Charlie");
        network.addFriendship("Bob", "David");
        network.addFriendship("Charlie", "David");
        network.addFriendship("Charlie", "Eve");
        network.addFriendship("Eve", "Frank");

        // 测试好友推荐
        System.out.println("为爱丽丝推荐的好友：");
        network.recommendFriends("Alice").forEach(System.out::println);
    }

    private static Node createUser(String id, String name) {
        Node user = new Node(id.toLowerCase());
        user.addLabel("User");
        user.setProperty("name", name);
        return user;
    }

    public void addFriendship(String user1, String user2) {
        Node node1 = graphDB.getNode(user1.toLowerCase());
        Node node2 = graphDB.getNode(user2.toLowerCase());
        if (node1 == null || node2 == null) {
            throw new IllegalArgumentException("其中一个用户不存在");
        }

        // 创建双向好友关系
        Relationship rel1 = new Relationship(
                UUID.randomUUID().toString(),
                node1.getId(),
                node2.getId(),
                "FRIEND"
        );
        Relationship rel2 = new Relationship(
                UUID.randomUUID().toString(),
                node2.getId(),
                node1.getId(),
                "FRIEND"
        );

        graphDB.addRelationship(rel1);
        graphDB.addRelationship(rel2);
    }

    public List<String> recommendFriends(String userId) {
        Node user = graphDB.getNode(userId.toLowerCase());
        if (user == null) {
            throw new IllegalArgumentException("用户不存在");
        }

        // 获取当前用户的所有好友
        Set<String> friends = user.getOutgoingEdges().stream().filter(v->v.getType().equals("FRIEND"))
                .map(rel -> rel.getEndNodeId())
                .collect(Collectors.toSet());

        // 获取所有好友的好友
        Map<String, Integer> friendOfFriends = new HashMap<>();
        for (String friend : friends) {
            Node friendNode = graphDB.getNode(friend);
            for (Relationship rel : friendNode.getOutgoingEdges()) {
                String target = rel.getEndNodeId();
                if (!friends.contains(target) && !target.equals(userId)) {
                    friendOfFriends.put(target, friendOfFriends.getOrDefault(target, 0) + 1);
                }
            }
        }

        // 按共同好友数量排序并返回推荐列表
        List<String> recommendations = new ArrayList<>(friendOfFriends.keySet());
        recommendations.sort((a, b) -> friendOfFriends.get(b) - friendOfFriends.get(a));

        // 返回推荐用户的名字
        return recommendations.stream()
                .map(id -> (String) graphDB.getNode(id).getProperty("name"))
                .collect(Collectors.toList());
    }
}