package io.github.javpower.vectorex.keynote.index.scalar;

import io.github.javpower.vectorex.keynote.core.DbData;
import io.github.javpower.vectorex.keynote.query.ConditionBuilder;

import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Predicate;
import java.util.stream.Collectors;

public class ScalarIndexManager {

    // id -> metadata 数据
    private final Map<String, DbData> metadataIndex = new ConcurrentHashMap<>();

    // field -> value -> Set<id>（倒排索引）
    private final Map<String, Map<Object, Set<String>>> invertedIndexMap = new ConcurrentHashMap<>();

    // field -> TreeMap（范围索引）
    private final Map<String, TreeMap<Comparable<?>, Set<String>>> rangeIndexMap = new ConcurrentHashMap<>();

    // 添加
    public void index(DbData data) {
        String id = data.getId();
        Map<String, Object> metadata = data.getMetadata();
        metadataIndex.put(id, data);

        for (Map.Entry<String, Object> entry : metadata.entrySet()) {
            String field = entry.getKey();
            Object value = entry.getValue();
            if(!(value instanceof List)){
                updateInvertedIndex(field, value, id);
            }
            // 如果值是 Comparable 类型，更新范围索引
            if (value instanceof Comparable) {
                updateRangeIndex(field, (Comparable<?>) value, id);
            }
        }
    }

    // 删除文档
    public void remove(String id) {
        DbData metadata = metadataIndex.remove(id);
        if (metadata == null) return;

        for (Map.Entry<String, Object> entry : metadata.getMetadata().entrySet()) {
            String field = entry.getKey();
            Object value = entry.getValue();
            if(!(value instanceof List)){
                // 从倒排索引中删除
                removeFromInvertedIndex(field, value, id);
            }
            // 如果值是 Comparable 类型，从范围索引中删除
            if (value instanceof Comparable) {
                removeFromRangeIndex(field, (Comparable<?>) value, id);
            }
        }
    }

    // 查询
    public List<String> search(ConditionBuilder conditionBuilder) {
        Predicate<Map<String, Object>> predicate = conditionBuilder.build();
        Set<String> resultSet = null;
        // 应用倒排索引条件
        for (Map.Entry<String, Object> entry : conditionBuilder.getIndexConditions().entrySet()) {
            String field = entry.getKey();
            Object value = entry.getValue();
            // 从倒排索引中查询
            Set<String> ids = queryInvertedIndex(field, value);
            if (ids.isEmpty()) return Collections.emptyList();
            resultSet = (resultSet == null) ? new HashSet<>(ids) : intersect(resultSet, ids);
        }
        // 应用范围条件
        for (RangeCondition rangeCondition : conditionBuilder.getRangeConditions()) {
            Set<String> ids = queryRangeIndex(rangeCondition);
            if (ids.isEmpty()) return Collections.emptyList();
            resultSet = (resultSet == null) ? new HashSet<>(ids) : intersect(resultSet, ids);
        }
        // 如果没有索引条件，直接遍历 metadataIndex
        if (resultSet==null){
            return metadataIndex.entrySet().stream()
                    .filter(entry -> predicate.test(entry.getValue().getMetadata()))
                    .map(Map.Entry::getKey)
                    .collect(Collectors.toList());
        }
        // 根据结果过滤
        return resultSet.stream()
                .filter(id -> {
                    DbData metadata = metadataIndex.get(id);
                    return metadata != null && predicate.test(metadata.getMetadata());
                })
                .collect(Collectors.toList());
    }

    // 更新范围索引
    private void updateRangeIndex(String field, Comparable<?> value, String id) {
        TreeMap<Comparable<?>, Set<String>> rangeIndex = rangeIndexMap.computeIfAbsent(field, k -> new TreeMap<>());
        Set<String> ids = rangeIndex.computeIfAbsent(value, k -> new HashSet<>());
        ids.add(id);
    }

    // 从范围索引中删除
    private void removeFromRangeIndex(String field, Comparable<?> value, String id) {
        TreeMap<Comparable<?>, Set<String>> rangeIndex = rangeIndexMap.get(field);
        if (rangeIndex != null) {
            Set<String> ids = rangeIndex.get(value);
            if (ids != null) {
                ids.remove(id);
                if (ids.isEmpty()) {
                    rangeIndex.remove(value);
                }
            }
        }
    }

    // 查询范围索引
    private Set<String> queryRangeIndex(RangeCondition rangeCondition) {
        Set<String> result = new HashSet<>();
        for (Map.Entry<String, TreeMap<Comparable<?>, Set<String>>> entry : rangeIndexMap.entrySet()) {
            TreeMap<Comparable<?>, Set<String>> rangeIndex = entry.getValue();
            NavigableMap<Comparable<?>, Set<String>> subMap = rangeIndex.subMap(
                    rangeCondition.getFromKey(), rangeCondition.isIncludeFrom(),
                    rangeCondition.getToKey(), rangeCondition.isIncludeTo()
            );
            for (Set<String> ids : subMap.values()) {
                result.addAll(ids);
            }
        }
        return result;
    }

    // 更新倒排索引
    private void updateInvertedIndex(String field, Object value, String id) {
        Map<Object, Set<String>> invertedIndex = invertedIndexMap.computeIfAbsent(field, k -> new HashMap<>());
        Set<String> ids = invertedIndex.computeIfAbsent(value, k -> new HashSet<>());
        ids.add(id);
    }

    // 从倒排索引中删除
    private void removeFromInvertedIndex(String field, Object value, String id) {
        Map<Object, Set<String>> invertedIndex = invertedIndexMap.get(field);
        if (invertedIndex != null) {
            Set<String> ids = invertedIndex.get(value);
            if (ids != null) {
                ids.remove(id);
                if (ids.isEmpty()) {
                    invertedIndex.remove(value);
                }
            }
        }
    }

    // 查询倒排索引
    private Set<String> queryInvertedIndex(String field, Object value) {
        Map<Object, Set<String>> invertedIndex = invertedIndexMap.get(field);
        if (invertedIndex == null) return new HashSet<>();
        return invertedIndex.get(value);
    }

    // 求两个集合的交集
    private Set<String> intersect(Set<String> set1, Set<String> set2) {
        Set<String> result = new HashSet<>(set1);
        result.retainAll(set2);
        return result;
    }
    public DbData getDbDataById(String id){
        return metadataIndex.get(id);
    }
}