package io.github.javpower.vectorex.keynote.query;

import io.github.javpower.vectorex.keynote.index.scalar.RangeCondition;
import io.github.javpower.vectorex.keynote.model.ConditionFiled;

import java.util.*;
import java.util.function.Predicate;

public class ConditionBuilder {

    private final List<Predicate<Map<String, Object>>> conditions = new ArrayList<>(); // 条件列表
    private final List<RangeCondition> rangeConditions = new ArrayList<>(); // 范围条件
    private final Map<String, Object> indexConditions = new HashMap<>(); // 倒排索引条件
    private final List<ConditionFiled> operations = new ArrayList<>(); // 操作记录

    // 等于
    public ConditionBuilder eq(String field, Object value) {
        addCondition(field, value, (fieldValue, val) -> Objects.equals(fieldValue, val), "eq");
        indexConditions.put(field, value); // 添加到倒排索引条件
        return this;
    }

    // 不等于
    public ConditionBuilder ne(String field, Object value) {
        addCondition(field, value, (fieldValue, val) -> !Objects.equals(fieldValue, val), "ne");
        return this;
    }

    // 大于
    public ConditionBuilder gt(String field, Comparable<?> value) {
        rangeConditions.add(new RangeCondition(value, null, false, false)); // (value, +∞)
        operations.add(new ConditionFiled("gt", field, value));
        return this;
    }

    // 大于等于
    public ConditionBuilder ge(String field, Comparable<?> value) {
        rangeConditions.add(new RangeCondition(value, null, true, false)); // [value, +∞)
        operations.add(new ConditionFiled("ge", field, value));
        return this;
    }

    // 小于
    public ConditionBuilder lt(String field, Comparable<?> value) {
        rangeConditions.add(new RangeCondition(null, value, false, false)); // (-∞, value)
        operations.add(new ConditionFiled("lt", field, value));
        return this;
    }

    // 小于等于
    public ConditionBuilder le(String field, Comparable<?> value) {
        rangeConditions.add(new RangeCondition(null, value, false, true)); // (-∞, value]
        operations.add(new ConditionFiled("le", field, value));
        return this;
    }

    // 范围查询
    public ConditionBuilder between(String field, Comparable<?> start, Comparable<?> end) {
        rangeConditions.add(new RangeCondition(start, end, true, true)); // [start, end]
        operations.add(new ConditionFiled("between", field, Arrays.asList(start, end)));
        return this;
    }

    // 包含
    public ConditionBuilder in(String field, List<?> values) {
        conditions.add(entity -> {
            Object fieldValue = entity.get(field);
            return values.contains(fieldValue);
        });
        operations.add(new ConditionFiled("in", field, values));
        return this;
    }

    // 模糊查询
    public ConditionBuilder like(String field, String value) {
        conditions.add(entity -> {
            Object fieldValue = entity.get(field);
            return fieldValue != null && fieldValue.toString().contains(value);
        });
        operations.add(new ConditionFiled("like", field, value));
        return this;
    }

    // 与操作
    public ConditionBuilder and(ConditionBuilder other) {
        conditions.addAll(other.conditions);
        rangeConditions.addAll(other.rangeConditions);
        indexConditions.putAll(other.indexConditions);
        operations.add(new ConditionFiled("and", other));
        return this;
    }

    // 或操作
    public ConditionBuilder or(ConditionBuilder other) {
        conditions.add(entity -> other.conditions.stream().anyMatch(condition -> condition.test(entity)));
        rangeConditions.addAll(other.rangeConditions);
        operations.add(new ConditionFiled("or", other));
        return this;
    }

    // 构建最终的 Predicate
    public Predicate<Map<String, Object>> build() {
        return entity -> conditions.stream().allMatch(condition -> condition.test(entity));
    }

    // 获取范围条件
    public List<RangeCondition> getRangeConditions() {
        return rangeConditions;
    }

    // 获取倒排索引条件
    public Map<String, Object> getIndexConditions() {
        return indexConditions;
    }

    // 获取操作记录
    public List<ConditionFiled> getOperations() {
        return operations;
    }

    // 添加条件
    private <T> void addCondition(String field, T value, BiPredicate<Object, T> condition, String operation) {
        conditions.add(entity -> {
            Object fieldValue = entity.get(field);
            return condition.test(fieldValue, value);
        });
        operations.add(new ConditionFiled(operation, field, value));
    }

    // 条件接口
    @FunctionalInterface
    private interface BiPredicate<T, U> {
        boolean test(T t, U u);
    }
}