/*
 * Decompiled with CFR 0.152.
 */
package lphy.evolution.io;

import java.awt.Color;
import java.io.BufferedReader;
import java.io.EOFException;
import java.io.IOException;
import java.io.Reader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.TreeMap;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import jebl.evolution.io.ImportException;
import jebl.evolution.io.ImportHelper;
import jebl.evolution.io.NexusImporter;
import jebl.evolution.sequences.BasicSequence;
import jebl.evolution.sequences.Sequence;
import jebl.evolution.sequences.SequenceType;
import jebl.evolution.sequences.State;
import jebl.util.Attributable;
import lphy.evolution.Taxa;
import lphy.evolution.Taxon;
import lphy.evolution.alignment.Alignment;
import lphy.evolution.alignment.ContinuousCharacterData;
import lphy.evolution.datatype.Continuous;
import lphy.evolution.datatype.DataType;
import lphy.evolution.datatype.SequenceTypeFactory;
import lphy.evolution.io.MetaDataAlignment;
import lphy.evolution.traits.CharSetBlock;
import lphy.util.LoggerUtils;

public class NexusParser {
    protected final ImportHelper helper;
    protected NexusBlock nextBlock = null;
    protected String nextBlockName = null;
    protected int taxonCount = 0;
    protected int siteCount = 0;
    protected SequenceType sequenceType = null;
    protected String gapCharacters = "-";
    protected String matchCharacters = ".";
    protected String missingCharacters = "?";
    protected boolean isInterleaved = false;
    protected ContinuousCharacterData continuousCharacterData;

    public NexusParser(String fileName) {
        Reader reader = this.getReader(fileName);
        this.helper = new ImportHelper(reader);
        this.helper.setExpectedInputLength(0L);
        this.helper.setCommentDelimiters('[', ']', '\u0000', '!', '&');
    }

    protected Reader getReader(String fileName) {
        BufferedReader reader = null;
        try {
            if (!(fileName.endsWith("nex") || fileName.endsWith("nexus") || fileName.endsWith("nxs"))) {
                throw new IOException("Invalid Nexus file name !  fileName = " + fileName);
            }
            Path nexFile = Paths.get(fileName, new String[0]);
            if (!nexFile.toFile().exists() || nexFile.toFile().isDirectory()) {
                throw new IOException("Cannot find Nexus file ! " + nexFile + ", user.dir = " + System.getProperty("user.dir"));
            }
            reader = Files.newBufferedReader(nexFile);
        }
        catch (IOException e) {
            LoggerUtils.logStackTrace(e);
        }
        return reader;
    }

    public static void main(String[] args) {
        try {
            String fileName = args[0];
            System.out.println("Loading " + fileName);
            NexusParser importer = new NexusParser(fileName);
            if (fileName.equals("Dengue4.nex")) {
                MetaDataAlignment nexusAlignment = importer.importNexus("forward");
                System.out.println(nexusAlignment.toJSON());
            } else if (fileName.equals("primate.nex")) {
                MetaDataAlignment nexusAlignment = importer.importNexus(null);
                Alignment coding = nexusAlignment.charset("coding");
                System.out.println("coding : " + coding.toJSON());
                Alignment noncoding = nexusAlignment.charset("noncoding");
                System.out.println("noncoding : " + noncoding.toJSON());
            } else if (fileName.equals("haemulidae_trophic_traits.nex")) {
                importer.importNexus(null);
                System.out.println(importer.continuousCharacterData.toJSON());
            }
        }
        catch (Exception e) {
            e.printStackTrace();
        }
    }

    public MetaDataAlignment importNexus(String ageDirectionStr) throws IOException, ImportException {
        boolean done = false;
        MetaDataAlignment nexusData = null;
        List<jebl.evolution.taxa.Taxon> taxonList = null;
        while (!done) {
            try {
                NexusBlock block = this.findNextBlock();
                if (block == NexusBlock.TAXA) {
                    taxonList = this.readTaxaBlock();
                    continue;
                }
                if (block == NexusBlock.CHARACTERS) {
                    if (taxonList == null) {
                        throw new NexusImporter.MissingBlockException("TAXA block is missing");
                    }
                    nexusData = this.readCharactersBlock(taxonList);
                    continue;
                }
                if (block == NexusBlock.DATA) {
                    nexusData = this.readDataBlock(taxonList);
                    continue;
                }
                if (block == NexusBlock.ASSUMPTIONS) {
                    this.readAssumptionsBlock(nexusData);
                    continue;
                }
                if (block != NexusBlock.CALIBRATION) continue;
                this.readCalibrationBlock(nexusData, ageDirectionStr);
            }
            catch (EOFException ex) {
                done = true;
            }
        }
        if (DataType.isSame(this.sequenceType, Continuous.getInstance())) {
            if (this.continuousCharacterData == null) {
                throw new NexusImporter.MissingBlockException("Fail to load continuous data in MATRIX");
            }
        } else {
            if (nexusData == null) {
                throw new NexusImporter.MissingBlockException("DATA or CHARACTERS block is missing");
            }
            if (nexusData instanceof MetaDataAlignment) {
                LoggerUtils.log.info("Load " + nexusData.toString());
            }
        }
        return nexusData;
    }

    protected MetaDataAlignment readCharactersBlock(List<jebl.evolution.taxa.Taxon> taxonList) throws ImportException, IOException {
        this.siteCount = 0;
        this.sequenceType = null;
        this.readDataBlockHeader("MATRIX", NexusBlock.CHARACTERS);
        List<Sequence> sequences = this.readSequenceData(taxonList);
        MetaDataAlignment nexusData = this.createNexusAlignment(sequences);
        this.findEndBlock();
        return nexusData;
    }

    protected MetaDataAlignment readDataBlock(List<jebl.evolution.taxa.Taxon> taxonList) throws ImportException, IOException {
        this.taxonCount = 0;
        this.siteCount = 0;
        this.sequenceType = null;
        this.readDataBlockHeader("MATRIX", NexusBlock.DATA);
        MetaDataAlignment nexusData = null;
        if (DataType.isSame(this.sequenceType, Continuous.getInstance())) {
            LoggerUtils.log.info("Loading continuous character data ... ");
            this.continuousCharacterData = this.readContinuousCharacterData();
        } else {
            List<Sequence> sequences = this.readSequenceData(taxonList);
            nexusData = this.createNexusAlignment(sequences);
        }
        this.findEndBlock();
        return nexusData;
    }

    private MetaDataAlignment createNexusAlignment(List<Sequence> sequences) {
        if (this.sequenceType == null) {
            throw new IllegalArgumentException("Fail to find data type before parsing sequences !");
        }
        if (this.siteCount < 1) {
            throw new IllegalArgumentException("NCHAR < 1 ! " + this.siteCount);
        }
        int seqSize = sequences.size();
        Taxon[] taxons = new Taxon[seqSize];
        for (int t = 0; t < seqSize; ++t) {
            Sequence sequence = sequences.get(t);
            jebl.evolution.taxa.Taxon jeblTaxon = sequence.getTaxon();
            if (jeblTaxon == null) {
                throw new IllegalArgumentException("Cannot find taxon in sequence ! " + t);
            }
            taxons[t] = new Taxon(jeblTaxon.getName());
        }
        MetaDataAlignment nexusData = new MetaDataAlignment(Taxa.createTaxa(taxons), this.siteCount, this.sequenceType);
        for (int t = 0; t < seqSize; ++t) {
            Sequence sequence = sequences.get(t);
            for (int s = 0; s < sequence.getLength(); ++s) {
                State state = sequence.getState(s);
                int stateNum = state.getIndex();
                nexusData.setState(t, s, stateNum);
            }
        }
        return nexusData;
    }

    private void readDataBlockHeader(String tokenToLookFor, NexusBlock block) throws ImportException, IOException {
        String token;
        boolean foundDimensions = false;
        boolean foundTitle = false;
        boolean foundFormat = false;
        do {
            if ((token = this.helper.readToken()).equalsIgnoreCase("TITLE")) {
                if (foundTitle) {
                    throw new ImportException.DuplicateFieldException("TITLE");
                }
                foundTitle = true;
                continue;
            }
            if (token.equalsIgnoreCase("DIMENSIONS")) {
                if (foundDimensions) {
                    throw new ImportException.DuplicateFieldException("DIMENSIONS");
                }
                boolean nchar = block == NexusBlock.TAXA;
                boolean ntax = block == NexusBlock.CHARACTERS;
                do {
                    String token2 = this.helper.readToken("=;");
                    if (this.helper.getLastDelimiter() != 61) {
                        throw new ImportException.BadFormatException("Unknown subcommand, '" + token2 + "', or missing '=' in DIMENSIONS command");
                    }
                    if (token2.equalsIgnoreCase("NTAX")) {
                        if (block == NexusBlock.CHARACTERS) {
                            throw new ImportException.BadFormatException("NTAX subcommand in CHARACTERS block");
                        }
                        this.taxonCount = this.helper.readInteger(";");
                        ntax = true;
                        continue;
                    }
                    if (token2.equalsIgnoreCase("NCHAR")) {
                        if (block == NexusBlock.TAXA) {
                            throw new ImportException.BadFormatException("NCHAR subcommand in TAXA block");
                        }
                        this.siteCount = this.helper.readInteger(";");
                        nchar = true;
                        continue;
                    }
                    throw new ImportException.BadFormatException("Unknown subcommand, '" + token2 + "', in DIMENSIONS command");
                } while (this.helper.getLastDelimiter() != 59);
                if (!ntax) {
                    throw new ImportException.BadFormatException("NTAX subcommand missing from DIMENSIONS command");
                }
                if (!nchar) {
                    throw new ImportException.BadFormatException("NCHAR subcommand missing from DIMENSIONS command");
                }
                foundDimensions = true;
                continue;
            }
            if (!token.equalsIgnoreCase("FORMAT")) continue;
            if (foundFormat) {
                throw new ImportException.DuplicateFieldException("FORMAT");
            }
            this.sequenceType = null;
            do {
                String token2;
                if ((token2 = this.helper.readToken("=;")).equalsIgnoreCase("GAP")) {
                    if (this.helper.getLastDelimiter() != 61) {
                        throw new ImportException.BadFormatException("Expecting '=' after GAP subcommand in FORMAT command");
                    }
                    this.gapCharacters = this.helper.readToken(";");
                    continue;
                }
                if (token2.equalsIgnoreCase("MISSING")) {
                    if (this.helper.getLastDelimiter() != 61) {
                        throw new ImportException.BadFormatException("Expecting '=' after MISSING subcommand in FORMAT command");
                    }
                    this.missingCharacters = this.helper.readToken(";");
                    continue;
                }
                if (token2.equalsIgnoreCase("MATCHCHAR")) {
                    if (this.helper.getLastDelimiter() != 61) {
                        throw new ImportException.BadFormatException("Expecting '=' after MATCHCHAR subcommand in FORMAT command");
                    }
                    this.matchCharacters = this.helper.readToken(";");
                    continue;
                }
                if (token2.equalsIgnoreCase("DATATYPE")) {
                    if (this.helper.getLastDelimiter() != 61) {
                        throw new ImportException.BadFormatException("Expecting '=' after DATATYPE subcommand in FORMAT command");
                    }
                    String token3 = this.helper.readToken(";");
                    try {
                        this.sequenceType = SequenceTypeFactory.INSTANCE.getDataType(token3);
                    }
                    catch (UnsupportedOperationException e) {
                        throw new ImportException.UnparsableDataException(e.getMessage());
                    }
                } else {
                    if (!token2.equalsIgnoreCase("INTERLEAVE")) continue;
                    this.isInterleaved = true;
                }
            } while (this.helper.getLastDelimiter() != 59);
            foundFormat = true;
        } while (!token.equalsIgnoreCase(tokenToLookFor));
        if (!foundDimensions) {
            throw new ImportException.MissingFieldException("DIMENSIONS");
        }
        if (block != NexusBlock.TAXA && this.sequenceType == null) {
            throw new ImportException.MissingFieldException("DATATYPE. Only Nucleotide or Protein sequences are supported.");
        }
    }

    protected List<Sequence> readSequenceData(List<jebl.evolution.taxa.Taxon> taxonList) throws ImportException, IOException {
        boolean sequencherStyle = false;
        String firstSequence = null;
        ArrayList<Sequence> sequences = new ArrayList<Sequence>();
        if (this.isInterleaved) {
            ArrayList<StringBuilder> sequencesData = new ArrayList<StringBuilder>(this.taxonCount);
            ArrayList<jebl.evolution.taxa.Taxon> taxons = new ArrayList<jebl.evolution.taxa.Taxon>();
            List<jebl.evolution.taxa.Taxon> taxList = taxonList != null ? taxonList : taxons;
            int[] charsRead = new int[this.taxonCount];
            for (int i = 0; i < this.taxonCount; ++i) {
                sequencesData.add(new StringBuilder());
                charsRead[i] = 0;
            }
            boolean firstLoop = true;
            int readCount = 0;
            while (readCount < this.siteCount * this.taxonCount) {
                for (int i = 0; i < this.taxonCount; ++i) {
                    int sequenceIndex;
                    String token = this.helper.readToken();
                    jebl.evolution.taxa.Taxon taxon = jebl.evolution.taxa.Taxon.getTaxon((String)token);
                    if (firstLoop) {
                        if (taxonList != null) {
                            sequenceIndex = taxonList.indexOf(taxon);
                        } else {
                            sequenceIndex = taxons.size();
                            taxons.add(taxon);
                        }
                    } else {
                        sequenceIndex = taxList.indexOf(taxon);
                    }
                    if (sequenceIndex < 0) {
                        throw new ImportException.UnknownTaxonException("Unexpected taxon:" + token + " (expecting " + taxList.get(i).getName() + ")");
                    }
                    StringBuffer buffer = new StringBuffer();
                    this.helper.readSequenceLine(buffer, this.sequenceType, ";", this.gapCharacters, this.missingCharacters, this.matchCharacters, firstSequence);
                    String seqString = buffer.toString();
                    try {
                        if (firstLoop && Integer.parseInt(taxon.toString()) == this.taxonCount && Integer.parseInt(seqString) == this.siteCount) {
                            --i;
                            taxons.remove(taxon);
                            sequencherStyle = true;
                            continue;
                        }
                    }
                    catch (NumberFormatException numberFormatException) {
                        // empty catch block
                    }
                    readCount += seqString.length();
                    int n = sequenceIndex;
                    charsRead[n] = charsRead[n] + seqString.length();
                    ((StringBuilder)sequencesData.get(sequenceIndex)).append(seqString);
                    if (i == 0) {
                        firstSequence = seqString;
                    }
                    if (this.helper.getLastDelimiter() != 59) continue;
                    if (i < this.taxonCount - 1) {
                        throw new ImportException.TooFewTaxaException();
                    }
                    for (int k = 0; k < this.taxonCount; ++k) {
                        if (charsRead[k] == this.siteCount) continue;
                        throw new ImportException.ShortSequenceException(taxList.get(k).getName() + " has length " + charsRead[k] + ", expecting " + this.siteCount);
                    }
                }
                firstLoop = false;
            }
            if (!sequencherStyle && this.helper.getLastDelimiter() != 59) {
                throw new ImportException.BadFormatException("Expecting ';' after sequences data");
            }
            for (int k = 0; k < this.taxonCount; ++k) {
                BasicSequence sequence = new BasicSequence(this.sequenceType, taxList.get(k), (CharSequence)sequencesData.get(k));
                sequences.add((Sequence)sequence);
            }
        } else {
            for (int i = 0; i < this.taxonCount; ++i) {
                String token = this.helper.readToken();
                jebl.evolution.taxa.Taxon taxon = jebl.evolution.taxa.Taxon.getTaxon((String)token);
                if (taxonList != null && !taxonList.contains(taxon)) {
                    StringBuilder message = new StringBuilder("Expected: ").append(token).append("\nActual taxa:\n");
                    for (jebl.evolution.taxa.Taxon taxon1 : taxonList) {
                        message.append(taxon1).append("\n");
                    }
                    throw new ImportException.UnknownTaxonException(message.toString());
                }
                StringBuilder buffer = new StringBuilder();
                this.helper.readSequence(buffer, this.sequenceType, ";", this.siteCount, this.gapCharacters, this.missingCharacters, this.matchCharacters, firstSequence, true);
                String seqString = buffer.toString();
                if (seqString.length() != this.siteCount) {
                    throw new ImportException.ShortSequenceException(taxon.getName() + " has length " + seqString.length() + ", expecting " + this.siteCount);
                }
                if (i == 0) {
                    firstSequence = seqString;
                }
                if (this.helper.getLastDelimiter() == 59 && i < this.taxonCount - 1) {
                    throw new ImportException.TooFewTaxaException();
                }
                BasicSequence sequence = new BasicSequence(this.sequenceType, taxon, (CharSequence)seqString);
                sequences.add((Sequence)sequence);
            }
            if (this.helper.getLastDelimiter() != 59) {
                throw new ImportException.BadFormatException("Expecting ';' after sequences data");
            }
        }
        return sequences;
    }

    protected void readCalibrationBlock(MetaDataAlignment nexusData, String ageDirectionStr) throws ImportException, IOException {
        String token;
        do {
            if ((token = this.helper.readToken(";")).equalsIgnoreCase("OPTIONS")) {
                String token2 = this.helper.readToken("=");
                if (!token2.equalsIgnoreCase("SCALE")) continue;
                String scale = this.helper.readToken(";");
                if (scale.toLowerCase().endsWith("s")) {
                    scale = scale.substring(0, scale.length() - 1);
                }
                nexusData.setChronoUnit(switch (scale) {
                    case "year" -> ChronoUnit.YEARS;
                    default -> throw new UnsupportedOperationException("Unsupported scale = " + scale);
                });
                continue;
            }
            if (!token.equalsIgnoreCase("TIPCALIBRATION")) continue;
            if (nexusData.getChronoUnit() == null) {
                throw new ImportException("Cannot find SCALE unit, e.g. year");
            }
            LinkedHashMap<String, String> ageMap = new LinkedHashMap<String, String>();
            do {
                int lastDelimiter;
                String date = null;
                String taxonNm = null;
                do {
                    String token2 = this.helper.readToken(":=,;");
                    if (this.helper.getLastDelimiter() != 61) {
                        if (this.helper.getLastDelimiter() == 58) {
                            date = token2;
                        } else {
                            taxonNm = token2;
                        }
                    }
                    lastDelimiter = this.helper.getLastDelimiter();
                    if (date != null && taxonNm != null) {
                        ageMap.put(taxonNm, date);
                        continue;
                    }
                    if (lastDelimiter != 44 && lastDelimiter != 59) continue;
                    throw new ImportException();
                } while (lastDelimiter != 44 && lastDelimiter != 59);
            } while (this.helper.getLastDelimiter() != 59);
            if (ageMap.size() < 1) {
                throw new ImportException("Cannot parse TIPCALIBRATION !");
            }
            if (ageMap.size() != this.taxonCount) {
                System.err.println("Warning: " + ageMap.size() + " tips have dates, but taxon count = " + this.taxonCount);
            }
            nexusData.assignAges(ageMap, ageDirectionStr);
        } while (this.isNotEnd(token));
    }

    protected void readAssumptionsBlock(MetaDataAlignment nexusData) throws ImportException, IOException {
        String token;
        TreeMap<String, List<CharSetBlock>> charsetMap = new TreeMap<String, List<CharSetBlock>>();
        do {
            if (!(token = this.helper.readToken(";")).equalsIgnoreCase("CHARSET")) continue;
            String charset = this.helper.readToken("=");
            ArrayList<CharSetBlock> charSetBlocks = new ArrayList<CharSetBlock>();
            do {
                String oneBlock = this.helper.readToken(";");
                try {
                    CharSetBlock charSetBlock = CharSetBlock.Utils.parseCharSet(oneBlock);
                    charSetBlocks.add(charSetBlock);
                }
                catch (IllegalArgumentException e) {
                    throw new ImportException("Charset " + charset + " : " + e.getMessage());
                }
            } while (this.helper.getLastDelimiter() != 59);
            charsetMap.put(charset, charSetBlocks);
        } while (this.isNotEnd(token));
        nexusData.setCharsetMap(charsetMap);
    }

    private ContinuousCharacterData readContinuousCharacterData() throws ImportException, IOException {
        assert (this.taxonCount > 0 && this.siteCount > 0);
        Double[][] continuousData = new Double[this.taxonCount][this.siteCount];
        Taxon[] taxa = new Taxon[this.taxonCount];
        if (this.isInterleaved) {
            throw new UnsupportedOperationException("in dev");
        }
        for (int i = 0; i < this.taxonCount; ++i) {
            String token = this.helper.readToken();
            taxa[i] = new Taxon(token);
            for (int j = 0; j < this.siteCount; ++j) {
                token = this.helper.readToken();
                try {
                    continuousData[i][j] = Double.parseDouble(token);
                    continue;
                }
                catch (NumberFormatException ex) {
                    if (j < this.siteCount - 1) {
                        throw new ImportException.ShortSequenceException(taxa[i].getName() + " has " + j + " traits, expecting " + this.siteCount);
                    }
                    throw new ImportException.BadFormatException("Double value is expected for continuous data at taxon " + i + " trait " + j);
                }
            }
            if (this.helper.getLastDelimiter() != 59 || i >= this.taxonCount - 1) continue;
            throw new ImportException.TooFewTaxaException(Integer.toString(i + 1));
        }
        String token = this.helper.readToken(";");
        if (this.helper.getLastDelimiter() != 59) {
            throw new ImportException.BadFormatException("Expecting ';' after continuous data\n" + this.helper.getLastDelimiter());
        }
        return new ContinuousCharacterData(new Taxa.Simple(taxa), continuousData);
    }

    public NexusBlock findNextBlock() throws IOException {
        this.findToken("BEGIN", true);
        this.nextBlockName = this.helper.readToken(";").toUpperCase();
        return this.findBlockName(this.nextBlockName);
    }

    protected NexusBlock findBlockName(String blockName) {
        try {
            this.nextBlock = NexusBlock.valueOf(blockName);
        }
        catch (IllegalArgumentException e) {
            this.nextBlock = null;
        }
        if (this.nextBlock == null) {
            this.nextBlock = NexusBlock.UNKNOWN;
        }
        return this.nextBlock;
    }

    public void findEndBlock() throws IOException {
        try {
            String token;
            while (!(token = this.helper.readToken(";")).equalsIgnoreCase("END") && !token.equalsIgnoreCase("ENDBLOCK")) {
            }
        }
        catch (EOFException eOFException) {
            // empty catch block
        }
        this.nextBlock = NexusBlock.UNKNOWN;
    }

    protected boolean isNotEnd(String token) {
        return !token.equalsIgnoreCase("END") && !token.equalsIgnoreCase("ENDBLOCK");
    }

    private void findToken(String query, boolean ignoreCase) throws IOException {
        boolean found = false;
        do {
            String token = this.helper.readToken();
            if ((!ignoreCase || !token.equalsIgnoreCase(query)) && !token.equals(query)) continue;
            found = true;
        } while (!found);
    }

    static void parseAndClearMetaComments(Attributable item, ImportHelper importHelper) throws ImportException.BadFormatException {
        for (String meta : importHelper.getMetaComments()) {
            NexusParser.parseMetaCommentPairs(meta, item);
        }
        importHelper.clearLastMetaComment();
    }

    static void parseMetaCommentPairs(String meta, Attributable item) throws ImportException.BadFormatException {
        Pattern pattern = Pattern.compile("(\"[^\"]*\"+|[^,=\\s]+)\\s*(=\\s*(\\{(\\{[^\\}]+\\},?)+\\}|\\{[^\\}]+\\}|\"[^\"]*\"+|[^,]+))?");
        Matcher matcher = pattern.matcher(meta);
        while (matcher.find()) {
            String label = matcher.group(1);
            if (label.charAt(0) == '\"') {
                label = label.substring(1, label.length() - 1);
            }
            if (label == null || label.trim().length() == 0) {
                throw new ImportException.BadFormatException("Badly formatted attribute: '" + matcher.group() + "'");
            }
            String value = matcher.group(2);
            if (value != null && value.trim().length() > 0) {
                item.setAttribute(label, NexusParser.parseValue(value.substring(1)));
                continue;
            }
            item.setAttribute(label, (Object)Boolean.TRUE);
        }
    }

    static Object parseValue(String value) {
        if ((value = value.trim()).startsWith("{")) {
            String[] elements;
            if ((value = value.substring(1, value.length() - 1)).startsWith("{")) {
                value = value.replaceAll("\\},\\{", "}@,@{");
                elements = value.split("@,@");
            } else {
                elements = value.split(",");
            }
            Object[] values = new Object[elements.length];
            for (int i = 0; i < elements.length; ++i) {
                values[i] = NexusParser.parseValue(elements[i]);
            }
            return values;
        }
        if (value.startsWith("#")) {
            String colourValue = value.substring(1);
            if (colourValue.startsWith("-")) {
                try {
                    return Color.decode(colourValue);
                }
                catch (NumberFormatException numberFormatException) {
                }
            } else {
                return Color.decode("0x" + colourValue);
            }
        }
        if (value.startsWith("\"") && value.endsWith("\"")) {
            return value.subSequence(1, value.length() - 1);
        }
        if (value.equalsIgnoreCase("TRUE") || value.equalsIgnoreCase("FALSE")) {
            return Boolean.valueOf(value);
        }
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException numberFormatException) {
            try {
                return Double.parseDouble(value);
            }
            catch (NumberFormatException numberFormatException2) {
                return value;
            }
        }
    }

    private List<jebl.evolution.taxa.Taxon> readTaxaBlock() throws ImportException, IOException {
        this.taxonCount = 0;
        this.readDataBlockHeader("TAXLABELS", NexusBlock.TAXA);
        if (this.taxonCount == 0) {
            throw new ImportException.MissingFieldException("NTAXA");
        }
        ArrayList<jebl.evolution.taxa.Taxon> taxa = new ArrayList<jebl.evolution.taxa.Taxon>();
        do {
            String name;
            if ((name = this.helper.readToken(";")).equals("")) {
                throw new ImportException.UnknownTaxonException("Expected nonempty taxon name, got empty string");
            }
            jebl.evolution.taxa.Taxon taxon = jebl.evolution.taxa.Taxon.getTaxon((String)name);
            taxa.add(taxon);
            NexusParser.parseAndClearMetaComments((Attributable)taxon, this.helper);
        } while (this.helper.getLastDelimiter() != 59);
        if (taxa.size() != this.taxonCount) {
            throw new ImportException.BadFormatException("Number of taxa doesn't match NTAXA field");
        }
        this.findEndBlock();
        return taxa;
    }

    public static enum NexusBlock {
        UNKNOWN,
        TAXA,
        CHARACTERS,
        DATA,
        ASSUMPTIONS,
        CALIBRATION,
        UNALIGNED,
        DISTANCES,
        TREES;

    }
}

