package com.admob.ads.banner

import android.os.Bundle
import android.view.ViewGroup
import androidx.core.view.isVisible
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import com.admob.AdFormat
import com.admob.AdType
import com.admob.Constant
import com.admob.SAdCallback
import com.admob.addLoadingView
import com.admob.ads.AdsSDK
import com.admob.ads.AdsSDK.isEnableBanner
import com.admob.ads.AdsSDK.isPremium
import com.admob.anchoredAdaptiveBannerSize
import com.admob.getComponentActivityOnTop
import com.admob.getPaidTrackingBundle
import com.admob.inlineAdaptiveBannerSize
import com.admob.isEnable
import com.admob.isNetworkAvailable
import com.google.ads.mediation.admob.AdMobAdapter
import com.google.android.gms.ads.AdListener
import com.google.android.gms.ads.AdRequest
import com.google.android.gms.ads.AdSize
import com.google.android.gms.ads.AdView
import com.google.android.gms.ads.LoadAdError
import com.google.firebase.crashlytics.FirebaseCrashlytics
import timber.log.Timber

object AdmobBanner {

    private const val TAG = "AdmobBanner"

    private val banners = mutableMapOf<String, AdView?>()

    /**
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showInlineAdaptive(
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        null,
        adContainer,
        adUnitId,
        BannerAdSize.BannerInlineAdaptive,
        forceRefresh,
        callback,
    )

    /**
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showAnchoredAdaptive(
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        null,
        adContainer,
        adUnitId,
        BannerAdSize.BannerAnchoredAdaptive,
        forceRefresh,
        callback,
    )


    /**
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun show300x250(
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        null,
        adContainer,
        adUnitId,
        BannerAdSize.Banner300x250,
        forceRefresh,
        callback,
    )

    /**
     * Each position show be Unique AdUnitID
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param showOnBottom: Show on Top or Bottom
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showCollapsible(
        lifecycle: Lifecycle? = null,
        adContainer: ViewGroup,
        adUnitId: String,
        showOnBottom: Boolean = true,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        lifecycle,
        adContainer,
        adUnitId,
        if (showOnBottom) BannerAdSize.BannerCollapsibleBottom else BannerAdSize.BannerCollapsibleTop,
        forceRefresh,
        callback,
    )

    private fun show(
        lifecycle: Lifecycle? = null,
        adContainer: ViewGroup,
        space: String,
        bannerType: BannerAdSize,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) {
        val adChild = AdsSDK.getAdChild(space) ?: return

        if (!isEnableBanner || isPremium || (adChild.adsType != AdFormat.Banner) || !AdsSDK.app.isNetworkAvailable() || !adChild.isEnable()) {
            adContainer.removeAllViews()
            callback?.onDisable()
            adContainer.isVisible = false
            return
        }

        if ((bannerType == BannerAdSize.BannerCollapsibleTop || bannerType == BannerAdSize.BannerCollapsibleBottom) && forceRefresh) {
            destroyAdBySpace(space)
        }


        val adSize = getAdSize(bannerType)
        addLoadingLayout(adContainer, adSize)

        if (!adContainer.context.isNetworkAvailable()) {
            return
        }

        val adView = banners[space]

        if (adView == null || forceRefresh) {
//            Log.e(TAG, "show: adView = null-${adView == null} and forceRefresh: $forceRefresh ")
            Timber.d("$TAG show: adView = null-${adView == null} and forceRefresh: $forceRefresh ")

            val context =
                if (bannerType == BannerAdSize.BannerCollapsibleBottom || bannerType == BannerAdSize.BannerCollapsibleTop)
                    AdsSDK.getComponentActivityOnTop() ?: adContainer.context
                else
                    AdsSDK.app

            AdView(context).let {
                val id = if (AdsSDK.isDebugging) {
                    if (bannerType == BannerAdSize.BannerCollapsibleBottom || bannerType == BannerAdSize.BannerCollapsibleTop) {
                        Constant.ID_ADMOB_BANNER_COLLAPSE_TEST
                    } else {
                        Constant.ID_ADMOB_BANNER_TEST
                    }
                } else {
                    adChild.adsId
                }

                it.adUnitId = id
                it.setAdSize(adSize)
                it.setAdCallback(it, space, bannerType, callback) {
                    addExistBanner(lifecycle, adContainer, it)
                }
                it.loadAd(getAdRequest(bannerType))

                AdsSDK.adCallback.onAdStartLoading(adChild.adsId, AdType.Banner)
                callback?.onAdStartLoading(adChild.adsId, AdType.Banner)
            }
        }

        if (adView != null) {
//            Log.e(TAG, "show: adView != null")
            Timber.d("$TAG show: adView != null")
            addExistBanner(lifecycle, adContainer, adView)
            adView.setAdCallback(adView, space, bannerType, callback) {
                addExistBanner(lifecycle, adContainer, adView)
            }
            return
        }

    }


    private fun addExistBanner(
        lifecycle: Lifecycle?,
        adContainer: ViewGroup,
        bannerView: AdView
    ) {
        if (lifecycle?.currentState != Lifecycle.State.DESTROYED) {
            adContainer.removeAllViews()
            if (bannerView.parent is ViewGroup && bannerView.parent != null) {
                (bannerView.parent as ViewGroup).removeAllViews()
            }
            adContainer.addView(bannerView)

            lifecycle?.addObserver(object : DefaultLifecycleObserver {

                override fun onStop(owner: LifecycleOwner) {
                    super.onStop(owner)
//                    Log.e("DucLH----", "onStop")
                    Timber.d("$TAG onStop")
                }

                override fun onDestroy(owner: LifecycleOwner) {
                    super.onDestroy(owner)
                    Timber.d("$TAG onDestroy")
                    bannerView.destroy()
//                    Log.e("DucLH----", "onDestroy")
                    lifecycle.removeObserver(this)
                    adContainer.removeAllViews()
                    bannerView.removeAllViews()
                }
            })
        }
    }

    private fun addLoadingLayout(adContainer: ViewGroup, adSize: AdSize) {
        val lp = adContainer.layoutParams
        lp.width = adSize.getWidthInPixels(adContainer.context)
        lp.height = adSize.getHeightInPixels(adContainer.context)
        adContainer.layoutParams = lp
        adContainer.requestLayout()
        adContainer.addLoadingView()
    }

    private fun getAdSize(bannerAdSize: BannerAdSize): AdSize {
        return when (bannerAdSize) {
            BannerAdSize.BannerInlineAdaptive -> inlineAdaptiveBannerSize
            BannerAdSize.BannerAnchoredAdaptive -> anchoredAdaptiveBannerSize
            BannerAdSize.BannerCollapsibleTop -> anchoredAdaptiveBannerSize
            BannerAdSize.BannerCollapsibleBottom -> anchoredAdaptiveBannerSize
            BannerAdSize.Banner300x250 -> AdSize.MEDIUM_RECTANGLE
        }
    }

    private fun getAdRequest(bannerAdSize: BannerAdSize): AdRequest {
        val adRequestBuilder = AdRequest.Builder()
        val extras = Bundle()

        if (bannerAdSize == BannerAdSize.BannerCollapsibleTop) {
            extras.putString("collapsible", "top")
            adRequestBuilder.addNetworkExtrasBundle(AdMobAdapter::class.java, extras)
        }

        if (bannerAdSize == BannerAdSize.BannerCollapsibleBottom) {
            extras.putString("collapsible", "bottom")
            adRequestBuilder.addNetworkExtrasBundle(AdMobAdapter::class.java, extras)
        }

        return adRequestBuilder.build()
    }

    private fun AdView.setAdCallback(
        adView: AdView,
        space: String,
        bannerType: BannerAdSize,
        sAdCallback: SAdCallback?,
        onAdLoaded: () -> Unit
    ) {
        adListener = object : AdListener() {
            override fun onAdClicked() {
                AdsSDK.adCallback.onAdClicked(adUnitId, AdType.Banner)
                sAdCallback?.onAdClicked(adUnitId, AdType.Banner)
            }

            override fun onAdClosed() {
                AdsSDK.adCallback.onAdClosed(adUnitId, AdType.Banner)
                sAdCallback?.onAdClosed(adUnitId, AdType.Banner)
            }

            override fun onAdFailedToLoad(var1: LoadAdError) {
                banners[adView.adUnitId] = null
                AdsSDK.adCallback.onAdFailedToLoad(adUnitId, AdType.Banner, var1)
                sAdCallback?.onAdFailedToLoad(adUnitId, AdType.Banner, var1)
                runCatching { Throwable(var1.message) }
            }

            override fun onAdImpression() {
                AdsSDK.adCallback.onAdImpression(adUnitId, AdType.Banner)
                sAdCallback?.onAdImpression(adUnitId, AdType.Banner)
            }

            override fun onAdLoaded() {
                AdsSDK.adCallback.onAdLoaded(adUnitId, AdType.Banner)
                sAdCallback?.onAdLoaded(adUnitId, AdType.Banner)
                adView.setOnPaidEventListener { adValue ->
                    val bundle =
                        getPaidTrackingBundle(adValue, adUnitId, "Banner", adView.responseInfo)
                    AdsSDK.adCallback.onPaidValueListener(bundle)
                    sAdCallback?.onPaidValueListener(bundle)
                }
                banners[space]?.let {
                    if (bannerType == BannerAdSize.BannerCollapsibleTop || bannerType == BannerAdSize.BannerCollapsibleBottom) {
//                        it.destroy()
                    }
                }
                banners[space] = adView

                onAdLoaded.invoke()
            }
        }
    }


    fun setEnableBanner(isEnable: Boolean) {
        if (!isEnable) {
            try {
                banners.forEach { (_, adView) ->
                    val viewGroup = adView?.parent as? ViewGroup
                    adView?.destroy()
                    viewGroup?.removeAllViews()
                    viewGroup?.isVisible = false
                }
            } catch (e: Exception) {
                e.printStackTrace()
                FirebaseCrashlytics.getInstance().recordException(e)
            }
        }
    }

    fun getAdsView(space: String): AdView? = banners[space]

    fun destroyAdBySpace(space: String) {
        kotlin.runCatching {
            banners[space]?.let { adView ->
                Timber.d("$TAG destroyAdBySpace: $space")
                adView.destroy()
                (adView.parent as? ViewGroup)?.removeView(adView)
                banners[space] = null
            }
        }.onFailure {
            FirebaseCrashlytics.getInstance().recordException(it)
        }
    }
}