package com.admob.ads.rewarded

import android.os.CountDownTimer
import androidx.appcompat.app.AppCompatActivity
import com.admob.AdFormat
import com.admob.AdType
import com.admob.SAdCallback
import com.admob.ads.AdsSDK
import com.admob.getPaidTrackingBundle
import com.admob.isEnable
import com.admob.isNetworkAvailable
import com.admob.ui.dialogs.DialogShowLoadingAds
import com.admob.waitActivityResumed
import com.google.android.gms.ads.AdError
import com.google.android.gms.ads.FullScreenContentCallback
import com.google.android.gms.ads.LoadAdError
import com.google.android.gms.ads.rewarded.RewardedAd
import com.google.android.gms.ads.rewarded.RewardedAdLoadCallback
import timber.log.Timber


object ADRewarded {


    private var timer: CountDownTimer? = null

    /**
     * @param activity: Show on this activity
     * @param callBack
     * @param onUserEarnedReward
     * @param onFailureUserNotEarn
     */
    fun show(
        activity: AppCompatActivity,
        space: String,
        timeout: Long = 10_000L,
        isShowDefaultLoadingDialog: Boolean = true,
        callBack: SAdCallback? = null,
        onFailureUserNotEarn: () -> Unit = {},
        onUserEarnedReward: () -> Unit
    ) {

        val adChild = AdsSDK.getAdChild(space) ?: return
        val adUnitId = adChild.adsId

        if (!AdsSDK.isEnableRewarded) {
            onUserEarnedReward.invoke()
            return
        }

        if (!AdsSDK.app.isNetworkAvailable()) {
            onFailureUserNotEarn.invoke()
            return
        }

        if (AdsSDK.isPremium || (adChild.adsType != AdFormat.Reward) || !adChild.isEnable()
        ) {
            onUserEarnedReward.invoke()
            return
        }


        var dialog: DialogShowLoadingAds? = null

        if (isShowDefaultLoadingDialog) {
            dialog = DialogShowLoadingAds(activity).apply { show() }
        }


        AdsSDK.adCallback.onAdStartLoading(adUnitId, AdType.Rewarded)
        callBack?.onAdStartLoading(adUnitId, AdType.Native)

        var loadedReward: RewardedAd? = null

        RewardedAd.load(
            AdsSDK.app,
            adUnitId,
            AdsSDK.defaultAdRequest(),
            object : RewardedAdLoadCallback() {
                override fun onAdFailedToLoad(error: LoadAdError) {
//                    Log.e("ThoNH-1", "onAdFailedToLoad")
                    Timber.d("inter: onAdFailedToLoad: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdFailedToLoad(error)
                    AdsSDK.adCallback.onAdFailedToLoad(adUnitId, AdType.Rewarded, error)
                    callBack?.onAdFailedToLoad(adUnitId, AdType.Rewarded, error)
                    onFailureUserNotEarn.invoke()
                    dialog?.dismiss()
                    timer?.cancel()
                }

                override fun onAdLoaded(rewardedAd: RewardedAd) {
//                    Log.e("ThoNH-1", "onAdLoaded")
                    Timber.d("inter: onAdLoaded: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdLoaded(rewardedAd)
                    AdsSDK.adCallback.onAdLoaded(adUnitId, AdType.Rewarded)
                    callBack?.onAdLoaded(adUnitId, AdType.Rewarded)

                    rewardedAd.setOnPaidEventListener { adValue ->
                        val bundle = getPaidTrackingBundle(
                            adValue,
                            adUnitId,
                            "Rewarded",
                            rewardedAd.responseInfo
                        )
                        AdsSDK.adCallback.onPaidValueListener(bundle)
                        callBack?.onPaidValueListener(bundle)
                    }

                    rewardedAd.fullScreenContentCallback = object : FullScreenContentCallback() {
                        override fun onAdClicked() {
//                            Log.e("ThoNH-1", "onAdClicked")
                            Timber.d("inter: onAdClicked: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdClicked()
                            AdsSDK.adCallback.onAdClicked(adUnitId, AdType.Rewarded)
                            callBack?.onAdClicked(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdDismissedFullScreenContent() {
                            Timber.d("inter: onAdDismissedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
//                            Log.e("ThoNH-1", "onAdDismissedFullScreenContent")
                            Timber.d("inter: onAdDismissedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdDismissedFullScreenContent()
                            AdsSDK.adCallback.onAdDismissedFullScreenContent(
                                adUnitId,
                                AdType.Rewarded
                            )
                            callBack?.onAdDismissedFullScreenContent(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdFailedToShowFullScreenContent(error: AdError) {
                            Timber.d("inter: onAdFailedToShowFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
//                            Log.e("ThoNH-1", "onAdFailedToShowFullScreenContent")
                            super.onAdFailedToShowFullScreenContent(error)
                            AdsSDK.adCallback.onAdFailedToShowFullScreenContent(
                                adUnitId,
                                error.message,
                                AdType.Rewarded
                            )
                            callBack?.onAdFailedToShowFullScreenContent(
                                adUnitId,
                                error.message,
                                AdType.Rewarded
                            )
                            onFailureUserNotEarn.invoke()
                            dialog?.dismiss()
                            timer?.cancel()
                        }

                        override fun onAdImpression() {
                            Timber.d("inter: onAdImpression: $adUnitId ${AdType.Rewarded.name}")
//                            Log.e("ThoNH-1", "onAdImpression")
                            super.onAdImpression()
                            AdsSDK.adCallback.onAdImpression(adUnitId, AdType.Rewarded)
                            callBack?.onAdImpression(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdShowedFullScreenContent() {
//                            Log.e("ThoNH-1", "onAdShowedFullScreenContent")
                            Timber.d("inter: onAdShowedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdShowedFullScreenContent()
                            AdsSDK.adCallback.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                            callBack?.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                        }
                    }

                    loadedReward = rewardedAd
                }
            }
        )


        timer?.cancel()
        timer = object : CountDownTimer(timeout, 500) {
            override fun onTick(millisUntilFinished: Long) {

                if (!AdsSDK.isEnableRewarded) {
                    timer?.cancel()
                    onUserEarnedReward.invoke()
                    return
                }

                loadedReward?.let { reward ->
                    timer?.cancel()
                    activity.waitActivityResumed {
                        dialog?.dismiss()
                        reward.show(activity) { _ ->
//                            Log.e("ThoNH-1", "onUserEarnedReward")
                            Timber.d("inter: onUserEarnedReward: $adUnitId ${AdType.Rewarded.name}")
                            onUserEarnedReward.invoke()
                        }
                    }

                }
            }

            override fun onFinish() {
                timer?.cancel()
                onFailureUserNotEarn.invoke()
                dialog?.dismiss()
            }
        }.start()
    }

}