package com.admob.tracking

import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.launch

object SGTracking {

    // A shared flow for publishing events
    private val eventFlow = MutableSharedFlow<TrackingEvent>()

    /**
     * Function for subscribers to collect events
     * @param collector: The lambda function to collect events
     * Usage:
     * collectEvents { event ->
     *    // Handle the event
     *    println(event)
     *    // Do something with the event
     * }
     */
    suspend fun collectEvents(collector: suspend (TrackingEvent) -> Unit) {
        eventFlow.collect { event -> collector(event) }
    }

    /**
        * Function to log events (previously 'publish')
        * @param nameEvent: The name of the event
        * @param params: The key-value pairs to be logged
        * Usage:
        * logEvent("event_name") {
        *    add("key1", value1)
        *   add("key2", value2)
        * }
     */
    fun logEvent(nameEvent: String, params: EventParameterBuilder.() -> Unit) {
        CoroutineScope(Dispatchers.Main).launch {
            val data = EventParameterBuilder().apply(params).build()
            eventFlow.emit(TrackingEvent(nameEvent, data))
        }
    }
}