package com.admob.ads.banner

import android.app.Activity
import android.view.ViewGroup
import androidx.core.os.bundleOf
import androidx.core.view.isVisible
import androidx.lifecycle.DefaultLifecycleObserver
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleOwner
import com.admob.AdFormat
import com.admob.AdType
import com.admob.Constant
import com.admob.SAdCallback
import com.admob.addLoadingView
import com.admob.ads.AdsSDK
import com.admob.ads.AdsSDK.isEnableBanner
import com.admob.ads.AdsSDK.isPremium
import com.admob.anchoredAdaptiveBannerSize
import com.admob.getActivityOnTop
import com.admob.inlineAdaptiveBannerSize
import com.admob.isEnable
import com.admob.isNetworkAvailable
import com.google.android.libraries.ads.mobile.sdk.banner.AdSize
import com.google.android.libraries.ads.mobile.sdk.banner.BannerAd
import com.google.android.libraries.ads.mobile.sdk.banner.BannerAdEventCallback
import com.google.android.libraries.ads.mobile.sdk.banner.BannerAdRefreshCallback
import com.google.android.libraries.ads.mobile.sdk.banner.BannerAdRequest
import com.google.android.libraries.ads.mobile.sdk.common.AdLoadCallback
import com.google.android.libraries.ads.mobile.sdk.common.LoadAdError
import com.google.firebase.crashlytics.FirebaseCrashlytics
import timber.log.Timber

object AdmobBanner {

    private const val TAG = "AdmobBanner"

    private val banners = mutableMapOf<String, BannerAd?>()

    /**
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showInlineAdaptive(
        activity: Activity,
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        activity,
        null,
        adContainer,
        adUnitId,
        BannerAdSize.BannerInlineAdaptive,
        forceRefresh,
        callback,
    )

    /**
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showAnchoredAdaptive(
        activity: Activity,
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        activity,
        null,
        adContainer,
        adUnitId,
        BannerAdSize.BannerAnchoredAdaptive,
        forceRefresh,
        callback,
    )


    /**
     * @deprecated
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun show300x250(
        activity: Activity,
        adContainer: ViewGroup,
        adUnitId: String,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        activity,
        null,
        adContainer,
        adUnitId,
        BannerAdSize.Banner300x250,
        forceRefresh,
        callback,
    )

    /**
     * Each position show be Unique AdUnitID
     * @param adContainer: ViewGroup contain this Ad
     * @param adUnitId AdId
     * @param showOnBottom: Show on Top or Bottom
     * @param forceRefresh always load new ad then fill to ViewGroup
     * @param callback callback
     */
    fun showCollapsible(
        activity: Activity,
        lifecycle: Lifecycle? = null,
        adContainer: ViewGroup,
        adUnitId: String,
        showOnBottom: Boolean = true,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) = show(
        activity,
        lifecycle,
        adContainer,
        adUnitId,
        if (showOnBottom) BannerAdSize.BannerCollapsibleBottom else BannerAdSize.BannerCollapsibleTop,
        forceRefresh,
        callback,
    )

    private fun show(
        activity: Activity,
        lifecycle: Lifecycle? = null,
        adContainer: ViewGroup,
        space: String,
        bannerType: BannerAdSize,
        forceRefresh: Boolean = false,
        callback: SAdCallback? = null
    ) {
        val adChild = AdsSDK.getAdChild(space) ?: return

        if (!isEnableBanner || isPremium || (adChild.adsType != AdFormat.Banner) || !AdsSDK.app.isNetworkAvailable() || !adChild.isEnable()) {
            adContainer.removeAllViews()
            callback?.onDisable()
            adContainer.isVisible = false
            return
        }

        if ((bannerType == BannerAdSize.BannerCollapsibleTop || bannerType == BannerAdSize.BannerCollapsibleBottom) && forceRefresh) {
            destroyAdBySpace(space)
        }

        val adSize = getAdSize(bannerType)
        addLoadingLayout(adContainer, adSize)

        if (!adContainer.context.isNetworkAvailable()) {
            return
        }

        val bannerAd = banners[space]

        if (bannerAd == null || forceRefresh) {
            Timber.d("$TAG show: bannerAd = null-${bannerAd == null} and forceRefresh: $forceRefresh ")

            val id = if (AdsSDK.isDebugging) {
                if (bannerType == BannerAdSize.BannerCollapsibleBottom || bannerType == BannerAdSize.BannerCollapsibleTop) {
                    Constant.ID_ADMOB_BANNER_COLLAPSE_TEST
                } else {
                    Constant.ID_ADMOB_BANNER_TEST
                }
            } else {
                adChild.adsId
            }

            val adRequest = BannerAdRequest.Builder(id, adSize)
                .apply {
                    // Add collapsible extras if needed
                    if (bannerType == BannerAdSize.BannerCollapsibleTop) {
                        setGoogleExtrasBundle(bundleOf("collapsible" to "top"))
                    } else if (bannerType == BannerAdSize.BannerCollapsibleBottom) {
                        setGoogleExtrasBundle(bundleOf("collapsible" to "bottom"))
                    }
                }
                .build()

            BannerAd.load(
                adRequest,
                object : AdLoadCallback<BannerAd> {
                    override fun onAdLoaded(ad: BannerAd) {
                        banners[space]?.let {
                            // Destroy previous collapsible ad if exists
                            when {
                                bannerType != BannerAdSize.BannerCollapsibleTop && bannerType != BannerAdSize.BannerCollapsibleBottom -> {
                                }
                            }
                        }
                        banners[space] = ad
                        
                        // Set event callbacks
                        ad.adEventCallback = object : BannerAdEventCallback {
                            override fun onAdClicked() {
                                AdsSDK.adCallback.onAdClicked(id, AdType.Banner)
                                callback?.onAdClicked(id, AdType.Banner)
                            }

//                            override fun onAdClosed() {
//                                AdsSDK.adCallback.onAdClosed(id, AdType.Banner)
//                                callback?.onAdClosed(id, AdType.Banner)
//                            }

                            override fun onAdImpression() {
                                AdsSDK.adCallback.onAdImpression(id, AdType.Banner)
                                callback?.onAdImpression(id, AdType.Banner)
                            }
                        }

                                                 // Set refresh callback for automatic refreshes
                         ad.bannerAdRefreshCallback = object : BannerAdRefreshCallback {
                             override fun onAdRefreshed() {
                                 Timber.d("$TAG onAdRefreshed: $space")
                                 // Banner refreshed automatically, view is already updated
                                 AdsSDK.adCallback.onAdLoaded(id, AdType.Banner)
                                 callback?.onAdLoaded(id, AdType.Banner)
                             }

                             override fun onAdFailedToRefresh(adError: LoadAdError) {
                                 Timber.d("$TAG onAdFailedToRefresh: $space - ${adError.message}")
                                 AdsSDK.adCallback.onAdFailedToLoad(id, AdType.Banner, adError)
                                 callback?.onAdFailedToLoad(id, AdType.Banner, adError)
                             }
                         }



                        // Set paid event listener
//                        ad.setOnPaidEventListener { adValue ->
//                            val bundle = getPaidTrackingBundle(adValue, id, "Banner", ad.responseInfo)
//                            AdsSDK.adCallback.onPaidValueListener(bundle)
//                            callback?.onPaidValueListener(bundle)
//                        }


                        AdsSDK.adCallback.onAdLoaded(id, AdType.Banner)
                        callback?.onAdLoaded(id, AdType.Banner)

                                                 // Add banner view to container on UI thread
                         // Next Gen SDK callbacks run on background thread, must dispatch to UI thread
//                         val activity = if (bannerType == BannerAdSize.BannerCollapsibleBottom || bannerType == BannerAdSize.BannerCollapsibleTop)
//                             AdsSDK.getComponentActivityOnTop()
//                         else
//                             AdsSDK.getComponentActivityOnTop()

                        activity.runOnUiThread {
                            addExistBanner(activity, lifecycle, adContainer, ad)
                        }
                    }

                    override fun onAdFailedToLoad(adError: LoadAdError) {
                        banners[space] = null
                        AdsSDK.adCallback.onAdFailedToLoad(id, AdType.Banner, adError)
                        callback?.onAdFailedToLoad(id, AdType.Banner, adError)
                        runCatching { Throwable(adError.message) }
                    }
                }
            )

            AdsSDK.adCallback.onAdStartLoading(adChild.adsId, AdType.Banner)
            callback?.onAdStartLoading(adChild.adsId, AdType.Banner)
        }

        if (bannerAd != null) {
            Timber.d("$TAG show: bannerAd != null")
            addExistBanner(activity,lifecycle, adContainer, bannerAd)
            return
        }
    }

    private fun addExistBanner(
        activity: Activity,
        lifecycle: Lifecycle?,
        adContainer: ViewGroup,
        bannerAd: BannerAd
    ) {
        if (lifecycle?.currentState != Lifecycle.State.DESTROYED) {
            adContainer.removeAllViews()
            
            val bannerView = bannerAd.getView(activity)
            if (bannerView.parent is ViewGroup && bannerView.parent != null) {
                (bannerView.parent as ViewGroup).removeView(bannerView)
            }
            adContainer.addView(bannerView)

            lifecycle?.addObserver(object : DefaultLifecycleObserver {
                override fun onStop(owner: LifecycleOwner) {
                    super.onStop(owner)
                    Timber.d("$TAG onStop")
                }

                override fun onDestroy(owner: LifecycleOwner) {
                    super.onDestroy(owner)
                    Timber.d("$TAG onDestroy")
                    bannerAd.destroy()
                    lifecycle.removeObserver(this)
                    adContainer.removeAllViews()
                }
            })
        }
    }

    private fun addLoadingLayout(adContainer: ViewGroup, adSize: AdSize) {
        // Give the banner container a placeholder height to avoid sudden layout shifts when the ad loads
        val lp = adContainer.layoutParams
        lp.width = adSize.getWidthInPixels(adContainer.context)
        lp.height = adSize.getHeightInPixels(adContainer.context)
        adContainer.layoutParams = lp
        adContainer.requestLayout()
        adContainer.addLoadingView()
    }

    private fun getAdSize(bannerAdSize: BannerAdSize): AdSize {
        return when (bannerAdSize) {
            BannerAdSize.BannerInlineAdaptive -> inlineAdaptiveBannerSize
            BannerAdSize.BannerAnchoredAdaptive -> anchoredAdaptiveBannerSize
            BannerAdSize.BannerCollapsibleTop -> anchoredAdaptiveBannerSize
            BannerAdSize.BannerCollapsibleBottom -> anchoredAdaptiveBannerSize
            BannerAdSize.Banner300x250 -> AdSize.MEDIUM_RECTANGLE
        }
    }

    fun setEnableBanner(isEnable: Boolean) {
        if (!isEnable) {
            try {
                val activity = AdsSDK.getActivityOnTop()
                banners.forEach { (_, bannerAd) ->
                    bannerAd?.let { ad ->
                        val bannerView = ad.getView(activity!!)
                        val viewGroup = bannerView.parent as? ViewGroup
                        ad.destroy()
                        viewGroup?.removeAllViews()
                        viewGroup?.isVisible = false
                    }
                }
            } catch (e: Exception) {
                e.printStackTrace()
                FirebaseCrashlytics.getInstance().recordException(e)
            }
        }
    }

    fun getAdsView(space: String): BannerAd? = banners[space]

    fun destroyAdBySpace(space: String) {
        kotlin.runCatching {
            val activity = AdsSDK.getActivityOnTop()
            banners[space]?.let { bannerAd ->
                Timber.d("$TAG destroyAdBySpace: $space")
                val bannerView = bannerAd.getView(activity!!)
                bannerAd.destroy()
                (bannerView.parent as? ViewGroup)?.removeView(bannerView)
                banners[space] = null
            }
        }.onFailure {
            FirebaseCrashlytics.getInstance().recordException(it)
        }
    }
}