package com.admob.ads.open

import com.admob.AdFormat
import com.admob.AdType
import com.admob.AdsChild
import com.admob.Constant
import com.admob.SAdCallback
import com.admob.ads.AdsSDK
import com.admob.getActivityOnTop
import com.admob.isEnable
import com.admob.isNetworkAvailable
import com.admob.logAdImpression
import com.google.android.libraries.ads.mobile.sdk.appopen.AppOpenAd
import com.google.android.libraries.ads.mobile.sdk.appopen.AppOpenAdEventCallback
import com.google.android.libraries.ads.mobile.sdk.common.AdLoadCallback
import com.google.android.libraries.ads.mobile.sdk.common.AdRequest
import com.google.android.libraries.ads.mobile.sdk.common.FullScreenContentError
import com.google.android.libraries.ads.mobile.sdk.common.LoadAdError

object AdmobOpen {

    private const val TAG = "AdmobOpenAds"

    internal fun load(
        space: String,
        callback: SAdCallback? = null,
        onAdLoadFailure: () -> Unit = {},
        onAdLoaded: (appOpenAd: AppOpenAd) -> Unit = {},
    ) {
        val adChild = AdsSDK.getAdChild(space)
        if (adChild == null) {
            onAdLoadFailure.invoke()
            return
        }

        if (!AdsSDK.app.isNetworkAvailable() || AdsSDK.isPremium || (adChild.adsType != AdFormat.Open) || !adChild.isEnable()) {
            onAdLoadFailure.invoke()
            return
        }



        AdsSDK.adCallback.onAdStartLoading(adChild.adsId, AdType.OpenApp)
        callback?.onAdStartLoading(adChild.adsId, AdType.OpenApp)
        val adsId = if (AdsSDK.isDebugging) Constant.ID_ADMOB_OPEN_APP_TEST else adChild.adsId
        AppOpenAd.load(
            AdRequest.Builder(adsId).build(),
            object : AdLoadCallback<AppOpenAd> {
                override fun onAdFailedToLoad(adError: LoadAdError) {
                    AdsSDK.adCallback.onAdFailedToLoad(adsId, AdType.OpenApp, adError)
                    callback?.onAdFailedToLoad(adsId, AdType.OpenApp, adError)
                    onAdLoadFailure.invoke()
                }

                override fun onAdLoaded(ad: AppOpenAd) {
                    super.onAdLoaded(ad)
                    AdsSDK.adCallback.onAdLoaded(adsId, AdType.OpenApp)
                    callback?.onAdLoaded(adsId, AdType.OpenApp)
                    onAdLoaded.invoke(ad)
//                    ad.setOnPaidEventListener { adValue ->
//                        val bundle = getPaidTrackingBundle(adValue, adChild.adsId, "OpenApp", ad.responseInfo)
//                        AdsSDK.adCallback.onPaidValueListener(bundle)
//                        callback?.onPaidValueListener(bundle)
//                    }
                }
            }
        )
    }

    internal fun show(
        space: String,
        appOpenAd: AppOpenAd,
        callback: SAdCallback? = null
    ) {
        val adChild = AdsSDK.getAdChild(space) ?: AdsChild().apply {
            adsId = "Unknown"
            adsType = AdFormat.Open
        }
        if (!AdsSDK.isEnableOpenAds) {
            return
        }

        val activity = AdsSDK.getActivityOnTop() ?: return

        appOpenAd.adEventCallback = object : AppOpenAdEventCallback {
            override fun onAdClicked() {
                AdsSDK.adCallback.onAdClicked(adChild.adsId, AdType.OpenApp)
                callback?.onAdClicked(adChild.adsId, AdType.OpenApp)
            }

            override fun onAdDismissedFullScreenContent() {
                AdsSDK.adCallback.onAdDismissedFullScreenContent(adChild.adsId, AdType.OpenApp)
                callback?.onAdDismissedFullScreenContent(adChild.adsId, AdType.OpenApp)
            }

            override fun onAdFailedToShowFullScreenContent(fullScreenContentError: FullScreenContentError) {
                AdsSDK.adCallback.onAdFailedToShowFullScreenContent(
                    fullScreenContentError.message,
                    adChild.adsId,
                    AdType.OpenApp
                )
                callback?.onAdFailedToShowFullScreenContent(
                    adChild.adsId,
                    fullScreenContentError.message,
                    AdType.OpenApp
                )
            }

            override fun onAdImpression() {
                AdsSDK.adCallback.onAdImpression(adChild.adsId, AdType.OpenApp)
                callback?.onAdImpression(adChild.adsId, AdType.OpenApp)
                logAdImpression(TAG)
            }

            override fun onAdShowedFullScreenContent() {
                AdsSDK.adCallback.onAdShowedFullScreenContent(adChild.adsId, AdType.OpenApp)
                callback?.onAdShowedFullScreenContent(adChild.adsId, AdType.OpenApp)
            }
        }

        appOpenAd.show(activity)
    }
}