package com.admob.ads.open

import android.os.Bundle
import android.os.CountDownTimer
import com.admob.AdFormat
import com.admob.AdType
import com.admob.AdsChild
import com.admob.Constant
import com.admob.SAdCallback
import com.admob.ads.AdsSDK
import com.admob.ads.open.AdmobOpenResume.adUnitId
import com.admob.ads.open.AdmobOpenResume.spaceName
import com.admob.app_update.UpdateHelper
import com.admob.getComponentActivityOnTop
import com.admob.onNextActionWhenResume
import com.admob.waitActivityResumed
import com.google.android.libraries.ads.mobile.sdk.appopen.AppOpenAd
import com.google.android.libraries.ads.mobile.sdk.common.LoadAdError
import com.google.firebase.analytics.ktx.analytics
import com.google.firebase.ktx.Firebase

object AdmobOpenSplash {

    private var timer: CountDownTimer? = null

    /**
     * @param space: space
     * @param timeout: timeout to wait ad show
     * @param onAdLoaded: callback when adLoaded => for update UI or something
     * @param nextAction: callback for your work, general handle nextActivity or nextFragment
     */
    fun show(
        space: String,
        timeout: Long,
        onAdLoaded: () -> Unit = {},
        onPaidValueListener: (Bundle) -> Unit,
        nextAction: () -> Unit
    ) {

        val adChild = AdsSDK.getAdChild(space) ?: AdsChild().apply {
            adsId = "Unknown"
            adsType = AdFormat.Open
        }

        adUnitId = if (AdsSDK.isDebugging) Constant.ID_ADMOB_OPEN_APP_TEST else adChild.adsId
        spaceName = space

        var isNextActionExecuted = false

        fun checkUpdateAndNext(){
            UpdateHelper.checkUpdateAfterShowAd(nextAction)
        }
        fun callNextAction() {
            if (isNextActionExecuted) {
                return
            }
            isNextActionExecuted = true
            onNextActionWhenResume(::checkUpdateAndNext)
        }

        if (!AdsSDK.isEnableOpenAds) {
            onAdLoaded.invoke()
            nextAction.invoke()
            return
        }

        val callback = object : SAdCallback {

            override fun onAdLoaded(adUnit: String, adType: AdType) {
                super.onAdLoaded(adUnit, adType)
                onAdLoaded.invoke()
            }

            override fun onAdFailedToLoad(adUnit: String, adType: AdType, error: LoadAdError) {
                super.onAdFailedToLoad(adUnit, adType, error)
                timer?.cancel()
                callNextAction()
            }

            override fun onAdFailedToShowFullScreenContent(error : String,adUnit: String, adType: AdType) {
                super.onAdFailedToShowFullScreenContent(error, adUnit, adType)
                timer?.cancel()
                callNextAction()
            }

            override fun onAdDismissedFullScreenContent(adUnit: String, adType: AdType) {
                super.onAdDismissedFullScreenContent(adUnit, adType)
                timer?.cancel()
                callNextAction()
            }

            override fun onPaidValueListener(bundle: Bundle) {
                super.onPaidValueListener(bundle)
                onPaidValueListener.invoke(bundle)
            }
        }

        var adOpenAd: AppOpenAd? = null

        AdmobOpen.load(
            adUnitId,
            callback,
            onAdLoaded = { adOpenAd = it }
        )

        timer?.cancel()
        timer = object : CountDownTimer(timeout, 1000) {
            override fun onTick(millisUntilFinished: Long) {

                if (!AdsSDK.isEnableOpenAds) {
                    timer?.cancel()
                    onAdLoaded.invoke()
                    nextAction.invoke()
                    return
                }

                adOpenAd?.let { appOpenAd ->
                    timer?.cancel()
                    onNextActionWhenResume {
                        AdsSDK.getComponentActivityOnTop()?.waitActivityResumed {
                            AdmobOpen.show(space, appOpenAd, callback)
                        }
                    }
                }
            }

            override fun onFinish() {
                timer?.cancel()
                callNextAction()
            }
        }.start()
    }

    /**
     * @param spaceH: space for High
     * @param spaceM: space for Medium
     * @param spaceL: space for Low
     * @param timeout: timeout to wait ad show
     * @param onAdLoaded: callback when adLoaded => for update UI or something
     * @param nextAction: callback for your work, general handle nextActivity or nextFragment
     */
    fun showWithFloor(
        spaceH: String,
        spaceM: String,
        spaceL: String,
        timeout: Long,
        onAdLoaded: () -> Unit = {},
        onPaidValueListener: (Bundle) -> Unit,
        nextAction: () -> Unit
    ) {

        Firebase.analytics.logEvent("Splash_Ads_Floor", null)

        var isNextActionExecuted = false

        var currentSpaceLoading = spaceH

        var adOpenAd: AppOpenAd? = null

        fun callNextAction() {
            if (isNextActionExecuted) {
                return
            }
            isNextActionExecuted = true
            onNextActionWhenResume(nextAction)
        }

        if (!AdsSDK.isEnableOpenAds) {
            onAdLoaded.invoke()
            nextAction.invoke()
            return
        }

        val callback = object : SAdCallback {

            override fun onAdLoaded(adUnit: String, adType: AdType) {
                super.onAdLoaded(adUnit, adType)
                onAdLoaded.invoke()
            }

            override fun onAdFailedToLoad(adUnit: String, adType: AdType, error: LoadAdError) {
                super.onAdFailedToLoad(adUnit, adType, error)


                currentSpaceLoading = when (currentSpaceLoading) {
                    spaceH -> {
                        Firebase.analytics.logEvent("Splash_Ads_Floor_Medium", null)
                        spaceM
                    }

                    spaceM -> {
                        Firebase.analytics.logEvent("Splash_Ads_Floor_Low", null)
                        spaceL
                    }

                    else -> {
                        ""
                    }
                }

                if (currentSpaceLoading.isEmpty()) {
                    timer?.cancel()
                    callNextAction()
                } else {
                    AdmobOpen.load(
                        currentSpaceLoading,
                        this,
                        onAdLoaded = { adOpenAd = it }
                    )
                }
            }

            override fun onAdFailedToShowFullScreenContent(error : String,adUnit: String, adType: AdType) {
                super.onAdFailedToShowFullScreenContent(error, adUnit, adType)
                timer?.cancel()
                callNextAction()
            }

            override fun onAdDismissedFullScreenContent(adUnit: String, adType: AdType) {
                super.onAdDismissedFullScreenContent(adUnit, adType)
                timer?.cancel()
                callNextAction()
            }

            override fun onPaidValueListener(bundle: Bundle) {
                super.onPaidValueListener(bundle)
                onPaidValueListener.invoke(bundle)
            }
        }

        Firebase.analytics.logEvent("Splash_Ads_Floor_High", null)

        AdmobOpen.load(
            currentSpaceLoading,
            callback,
            onAdLoaded = { adOpenAd = it }
        )

        timer?.cancel()
        timer = object : CountDownTimer(timeout, 1000) {
            override fun onTick(millisUntilFinished: Long) {

                if (!AdsSDK.isEnableOpenAds) {
                    timer?.cancel()
                    onAdLoaded.invoke()
                    nextAction.invoke()
                    return
                }

                adOpenAd?.let { appOpenAd ->
                    timer?.cancel()
                    onNextActionWhenResume {
                        AdsSDK.getComponentActivityOnTop()?.waitActivityResumed {
                            AdmobOpen.show(currentSpaceLoading,appOpenAd, callback)
                        }
                    }
                }
            }

            override fun onFinish() {
                timer?.cancel()
                callNextAction()
            }
        }.start()
    }
}