package com.admob.ads.rewarded

import android.os.CountDownTimer
import androidx.activity.ComponentActivity
import com.admob.AdFormat
import com.admob.AdType
import com.admob.SAdCallback
import com.admob.ads.AdsSDK
import com.admob.isEnable
import com.admob.isNetworkAvailable
import com.admob.ui.dialogs.DialogShowLoadingAds
import com.admob.waitActivityResumed
import com.google.android.libraries.ads.mobile.sdk.common.AdLoadCallback
import com.google.android.libraries.ads.mobile.sdk.common.FullScreenContentError
import com.google.android.libraries.ads.mobile.sdk.common.LoadAdError
import com.google.android.libraries.ads.mobile.sdk.rewarded.RewardedAd
import com.google.android.libraries.ads.mobile.sdk.rewarded.RewardedAdEventCallback
import com.google.android.libraries.ads.mobile.sdk.rewardedinterstitial.RewardedInterstitialAd
import com.google.android.libraries.ads.mobile.sdk.rewardedinterstitial.RewardedInterstitialAdEventCallback
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import timber.log.Timber


object ADRewarded {


    private var timer: CountDownTimer? = null

    /**
     * @param activity: Show on this activity
     * @param callBack
     * @param onUserEarnedReward
     * @param onFailureUserNotEarn
     */
    fun show(
        activity: ComponentActivity,
        space: String,
        timeout: Long = 10_000L,
        isShowDefaultLoadingDialog: Boolean = true,
        callBack: SAdCallback? = null,
        onFailureUserNotEarn: () -> Unit = {},
        onUserEarnedReward: () -> Unit
    ) {

        val adChild = AdsSDK.getAdChild(space) ?: return
        val adUnitId = adChild.adsId

        if (!AdsSDK.isEnableRewarded) {
            CoroutineScope(Dispatchers.Main).launch {
                onUserEarnedReward.invoke()
            }

            return
        }

        if (!AdsSDK.app.isNetworkAvailable()) {
            CoroutineScope(Dispatchers.Main).launch {
                onFailureUserNotEarn.invoke()
            }
            return
        }

        if (AdsSDK.isPremium || (adChild.adsType != AdFormat.Reward) || !adChild.isEnable()
        ) {
            CoroutineScope(Dispatchers.Main).launch {
                onUserEarnedReward.invoke()
            }
            return
        }


        var dialog: DialogShowLoadingAds? = null

        if (isShowDefaultLoadingDialog) {
            dialog = DialogShowLoadingAds(activity).apply { show() }
        }


        AdsSDK.adCallback.onAdStartLoading(adUnitId, AdType.Rewarded)
        callBack?.onAdStartLoading(adUnitId, AdType.Native)

        var loadedReward: RewardedAd? = null

        RewardedAd.load(
            AdsSDK.getAdRequest(adUnitId),
            object : AdLoadCallback<RewardedAd> {
                override fun onAdFailedToLoad(adError: LoadAdError) {
                    Timber.d("Rewarded: onAdFailedToLoad: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdFailedToLoad(adError)
                    AdsSDK.adCallback.onAdFailedToLoad(adUnitId, AdType.Rewarded, adError)
                    callBack?.onAdFailedToLoad(adUnitId, AdType.Rewarded, adError)
                    onFailureUserNotEarn.invoke()
                    dialog?.dismiss()
                    timer?.cancel()
                }

                override fun onAdLoaded(ad: RewardedAd) {
                    Timber.d("Rewarded: onAdLoaded: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdLoaded(ad)
                    AdsSDK.adCallback.onAdLoaded(adUnitId, AdType.Rewarded)
                    callBack?.onAdLoaded(adUnitId, AdType.Rewarded)

//                    ad.setOnPaidEventListener { adValue ->
//                        val bundle = getPaidTrackingBundle(
//                            adValue,
//                            adUnitId,
//                            "Rewarded",
//                            ad.responseInfo
//                        )
//                        AdsSDK.adCallback.onPaidValueListener(bundle)
//                        callBack?.onPaidValueListener(bundle)
//                    }

                    ad.adEventCallback = object : RewardedAdEventCallback {
                        override fun onAdClicked() {
                            Timber.d("Rewarded: onAdClicked: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdClicked()
                            AdsSDK.adCallback.onAdClicked(adUnitId, AdType.Rewarded)
                            callBack?.onAdClicked(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdDismissedFullScreenContent() {
                            Timber.d("Rewarded: onAdDismissedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdDismissedFullScreenContent()
                            AdsSDK.adCallback.onAdDismissedFullScreenContent(
                                adUnitId,
                                AdType.Rewarded
                            )
                            callBack?.onAdDismissedFullScreenContent(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdFailedToShowFullScreenContent(fullScreenContentError: FullScreenContentError) {
                            Timber.d("Rewarded: onAdFailedToShowFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdFailedToShowFullScreenContent(fullScreenContentError)
                            AdsSDK.adCallback.onAdFailedToShowFullScreenContent(
                                fullScreenContentError.message,
                                adUnitId,
                                AdType.Rewarded
                            )
                            callBack?.onAdFailedToShowFullScreenContent(
                                fullScreenContentError.message,
                                adUnitId,
                                AdType.Rewarded
                            )
                            onFailureUserNotEarn.invoke()
                            dialog?.dismiss()
                            timer?.cancel()
                        }

                        override fun onAdImpression() {
                            Timber.d("Rewarded: onAdImpression: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdImpression()
                            AdsSDK.adCallback.onAdImpression(adUnitId, AdType.Rewarded)
                            callBack?.onAdImpression(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdShowedFullScreenContent() {
                            Timber.d("Rewarded: onAdShowedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdShowedFullScreenContent()
                            AdsSDK.adCallback.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                            callBack?.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                        }
                    }

                    loadedReward = ad
                }
            }
        )


        timer?.cancel()
        timer = object : CountDownTimer(timeout, 500) {
            override fun onTick(millisUntilFinished: Long) {

                if (!AdsSDK.isEnableRewarded) {
                    timer?.cancel()
                    CoroutineScope(Dispatchers.Main).launch {
                        onUserEarnedReward.invoke()
                    }
                    return
                }

                loadedReward?.let { reward ->
                    timer?.cancel()
                    activity.waitActivityResumed {
                        dialog?.dismiss()
                        reward.show(activity) { _ ->
                            Timber.d("Rewarded: onUserEarnedReward: $adUnitId ${AdType.Rewarded.name}")
                            CoroutineScope(Dispatchers.Main).launch {
                                onUserEarnedReward.invoke()
                            }
                        }
                    }

                }
            }

            override fun onFinish() {
                timer?.cancel()
                onFailureUserNotEarn.invoke()
                dialog?.dismiss()
            }
        }.start()
    }


    /**
     * @param activity: Show on this activity
     * @param callBack
     * @param onUserEarnedReward
     * @param onFailureUserNotEarn
     */
    fun showRewardInterstitial(
        activity: ComponentActivity,
        space: String,
        timeout: Long = 10_000L,
        isShowDefaultLoadingDialog: Boolean = true,
        callBack: SAdCallback? = null,
        onFailureUserNotEarn: () -> Unit = {},
        onUserEarnedReward: () -> Unit
    ) {

        val adChild = AdsSDK.getAdChild(space) ?: return
        val adUnitId = adChild.adsId

        if (!AdsSDK.isEnableRewarded) {
            CoroutineScope(Dispatchers.Main).launch {
                onUserEarnedReward.invoke()
            }
            return
        }

        if (!AdsSDK.app.isNetworkAvailable()) {
            onFailureUserNotEarn.invoke()
            return
        }

        if (AdsSDK.isPremium || (adChild.adsType != AdFormat.Reward) || !adChild.isEnable()
        ) {
            CoroutineScope(Dispatchers.Main).launch {
                onUserEarnedReward.invoke()
            }
            return
        }


        var dialog: DialogShowLoadingAds? = null

        if (isShowDefaultLoadingDialog) {
            dialog = DialogShowLoadingAds(activity).apply { show() }
        }


        AdsSDK.adCallback.onAdStartLoading(adUnitId, AdType.Rewarded)
        callBack?.onAdStartLoading(adUnitId, AdType.Native)

        var loadedReward: RewardedInterstitialAd? = null

        RewardedInterstitialAd.load(
            AdsSDK.getAdRequest(adUnitId),
            object : AdLoadCallback<RewardedInterstitialAd> {
                override fun onAdFailedToLoad(adError: LoadAdError) {
                    Timber.d("Rewarded: onAdFailedToLoad: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdFailedToLoad(adError)
                    AdsSDK.adCallback.onAdFailedToLoad(adUnitId, AdType.Rewarded, adError)
                    callBack?.onAdFailedToLoad(adUnitId, AdType.Rewarded, adError)
                    onFailureUserNotEarn.invoke()
                    dialog?.dismiss()
                    timer?.cancel()
                }

                override fun onAdLoaded(ad: RewardedInterstitialAd) {
                    Timber.d("Rewarded: onAdLoaded: $adUnitId ${AdType.Rewarded.name}")
                    super.onAdLoaded(ad)
                    AdsSDK.adCallback.onAdLoaded(adUnitId, AdType.Rewarded)
                    callBack?.onAdLoaded(adUnitId, AdType.Rewarded)

//                    ad.setOnPaidEventListener { adValue ->
//                        val bundle = getPaidTrackingBundle(
//                            adValue,
//                            adUnitId,
//                            "Rewarded",
//                            ad.responseInfo
//                        )
//                        AdsSDK.adCallback.onPaidValueListener(bundle)
//                        callBack?.onPaidValueListener(bundle)
//                    }

                    ad.adEventCallback = object : RewardedInterstitialAdEventCallback {
                        override fun onAdClicked() {
                            Timber.d("Rewarded: onAdClicked: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdClicked()
                            AdsSDK.adCallback.onAdClicked(adUnitId, AdType.Rewarded)
                            callBack?.onAdClicked(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdDismissedFullScreenContent() {
                            Timber.d("Rewarded: onAdDismissedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdDismissedFullScreenContent()
                            AdsSDK.adCallback.onAdDismissedFullScreenContent(
                                adUnitId,
                                AdType.Rewarded
                            )
                            callBack?.onAdDismissedFullScreenContent(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdFailedToShowFullScreenContent(fullScreenContentError: FullScreenContentError) {
                            Timber.d("Rewarded: onAdFailedToShowFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdFailedToShowFullScreenContent(fullScreenContentError)
                            AdsSDK.adCallback.onAdFailedToShowFullScreenContent(
                                fullScreenContentError.message,
                                adUnitId,
                                AdType.Rewarded
                            )
                            callBack?.onAdFailedToShowFullScreenContent(
                                fullScreenContentError.message,
                                adUnitId,
                                AdType.Rewarded
                            )
                            onFailureUserNotEarn.invoke()
                            dialog?.dismiss()
                            timer?.cancel()
                        }

                        override fun onAdImpression() {
                            Timber.d("Rewarded: onAdImpression: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdImpression()
                            AdsSDK.adCallback.onAdImpression(adUnitId, AdType.Rewarded)
                            callBack?.onAdImpression(adUnitId, AdType.Rewarded)
                        }

                        override fun onAdShowedFullScreenContent() {
                            Timber.d("Rewarded: onAdShowedFullScreenContent: $adUnitId ${AdType.Rewarded.name}")
                            super.onAdShowedFullScreenContent()
                            AdsSDK.adCallback.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                            callBack?.onAdShowedFullScreenContent(adUnitId, AdType.Rewarded)
                        }
                    }

                    loadedReward = ad
                }
            }
        )


        timer?.cancel()
        timer = object : CountDownTimer(timeout, 500) {
            override fun onTick(millisUntilFinished: Long) {

                if (!AdsSDK.isEnableRewarded) {
                    timer?.cancel()
                    CoroutineScope(Dispatchers.Main).launch {
                        onUserEarnedReward.invoke()
                    }
                    return
                }

                loadedReward?.let { reward ->
                    timer?.cancel()
                    activity.waitActivityResumed {
                        dialog?.dismiss()
                        reward.show(activity) { _ ->
                            Timber.d("Rewarded: onUserEarnedReward: $adUnitId ${AdType.Rewarded.name}")
                            CoroutineScope(Dispatchers.Main).launch {
                                onUserEarnedReward.invoke()
                            }
                        }
                    }

                }
            }

            override fun onFinish() {
                timer?.cancel()
                onFailureUserNotEarn.invoke()
                dialog?.dismiss()
            }
        }.start()
    }

}