package io.pica.cmp

import android.app.Activity
import android.util.Log
import com.google.android.ump.ConsentDebugSettings
import com.google.android.ump.ConsentForm
import com.google.android.ump.ConsentInformation
import com.google.android.ump.ConsentRequestParameters
import com.google.android.ump.UserMessagingPlatform
import io.pica.ads.AdsController

class CMPController(private val activity: Activity) {

    fun isGDPR(): Boolean {
        return GDPRUtils.isGDPR(activity)
    }

    fun showCMP(
        isTesting: Boolean,
        titleDialog: String? = null,
        listTestDeviceIds: List<String> = listOf(),
        desDialog: String? = null
    ) {
        val dialogLoadingForm = DialogLoadingForm(activity, titleDialog, desDialog)
        dialogLoadingForm.show()

        if (UserMessagingPlatform.getConsentInformation(activity).isConsentFormAvailable) {
            UserMessagingPlatform.loadConsentForm(
                activity,
                { consentForm ->
                    showCMPDialog(dialogLoadingForm, consentForm)
                },
                {
                    dismissDialog(dialogLoadingForm)
                }
            )
        } else {
            showCMP(false, listTestDeviceIds, object : CMPCallback {
                override fun onShowAd() {
                    dialogLoadingForm.dismiss()
                }

                override fun onChangeScreen() {
                    dialogLoadingForm.dismiss()
                }
            })
        }
    }

    private fun showCMPDialog(dialog: DialogLoadingForm, consentForm: ConsentForm) {
        dialog.dismiss()
        consentForm.show(activity) {
            dismissDialog(dialog)
        }
    }

    private fun dismissDialog(dialog: DialogLoadingForm) {
        dialog.dismiss()
    }

    fun showCMP(isTesting: Boolean, listTestDeviceIds: List<String> = listOf(), cmpCallback: CMPCallback) {
        Log.d("CMPController", "CMPController showCMP: ${GDPRUtils.isGDPR2(activity)}")
        if (GDPRUtils.isGDPR2(activity) == 0) {
            cmpCallback.onShowAd()
            AdsController.getInstance().cmpComplete()
            return
        }
        Log.d("CMPController", "CMPController showCMP: 2")
        val debugSettings = if (isTesting) {
            Log.d("CMPController", "CMPController showCMP: 3")
            ConsentDebugSettings.Builder(activity)
                .setDebugGeography(ConsentDebugSettings.DebugGeography.DEBUG_GEOGRAPHY_EEA)
                .addTestDeviceHashedId("F0F58405E2206F77FC8B7A0D27FE4358")
                .apply {
                    listTestDeviceIds.forEach {
                        addTestDeviceHashedId(it)
                    }
                }
                .build()
        } else {
            null
        }

        val consentRequestParameters = ConsentRequestParameters.Builder()
            .setTagForUnderAgeOfConsent(false)
            .setConsentDebugSettings(debugSettings)
            .build()

        val consentInformation = UserMessagingPlatform.getConsentInformation(activity)

        consentInformation.requestConsentInfoUpdate(activity, consentRequestParameters,
            {
                showCMPConsentInfoUpdateSuccess(consentInformation, cmpCallback)
            },
            {
                cmpCallback.onChangeScreen()
                AdsController.getInstance().cmpComplete()
            })
    }

    private fun showCMPConsentInfoUpdateSuccess(
        consentInformation: ConsentInformation,
        cmpCallback: CMPCallback
    ) {
        Log.d("CMPController", "a: ")
        val canShowAds = GDPRUtils.canShowAds(activity)
        consentInformation.canRequestAds()

        if (canShowAds) {
            Log.d("CMPController", "c: ")
            GDPRUtils.setUserConsent(true)
            cmpCallback.onShowAd()
            AdsController.getInstance().cmpComplete()
        } else {
            Log.d("CMPController", "d: ")
            UserMessagingPlatform.loadConsentForm(activity,
                { consentForm ->
                    Log.d("CMPController", "CMPController showCMPConsentInfoUpdateSuccess: 1")
                    Log.d(
                        "CMPController",
                        "CMPController showCMPConsentInfoUpdateSuccess: ${activity.javaClass.name}"
                    )
                    consentForm.show(activity) {
                        Log.d("CMPController", "CMPController showCMPConsentInfoUpdateSuccess: 3")
                        Log.d(
                            "CMPController",
                            "CMPController showCMPConsentInfoUpdateSuccess: error: ${it?.message} 2"
                        )

                        if (GDPRUtils.canShowAds(activity)) {
                            Log.d(
                                "CMPController",
                                "CMPController showCMPConsentInfoUpdateSuccess: 4"
                            )
                            GDPRUtils.setUserConsent(true)
                            cmpCallback.onShowAd()
                            AdsController.getInstance().cmpComplete()
                        } else {
                            Log.d(
                                "CMPController",
                                "CMPController showCMPConsentInfoUpdateSuccess: 5"
                            )
                            GDPRUtils.setUserConsent(false)
                            cmpCallback.onChangeScreen()
                            AdsController.getInstance().cmpComplete()
                        }
                    }
                },
                { formError ->
                    Log.d("CMPController", "e: " + formError.message)
                    cmpCallback.onShowAd()
                    AdsController.getInstance().cmpComplete()
                })
        }
    }
}
