package io.tools.billinghelper

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.net.Uri
import com.android.billingclient.api.AcknowledgePurchaseParams
import com.android.billingclient.api.BillingClient
import com.android.billingclient.api.BillingClientStateListener
import com.android.billingclient.api.BillingFlowParams
import com.android.billingclient.api.BillingResult
import com.android.billingclient.api.ConsumeParams
import com.android.billingclient.api.ProductDetails
import com.android.billingclient.api.Purchase
import com.android.billingclient.api.PurchasesUpdatedListener
import com.android.billingclient.api.QueryProductDetailsParams
import com.android.billingclient.api.QueryPurchasesParams
import io.tools.billinghelper.model.ErrorType
import io.tools.billinghelper.model.ProductPriceInfo
import kotlinx.coroutines.CompletableDeferred
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers.IO
import kotlinx.coroutines.Dispatchers.Main
import kotlinx.coroutines.awaitAll
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext
import timber.log.Timber

class BillingManager(private val context: Context) {

    private val TAG = "BillingManager"

    companion object {

        private var isClientReady = false
        private var billingClient: BillingClient? = null
        private var billingEventListener: BillingEventListener? = null
        private var billingClientListener: BillingClientListener? = null
        private var purchasesUpdatedListener: PurchasesUpdatedListener? = null

        private val subKeys by lazy { mutableListOf<String>() }
        private val inAppKeys by lazy { mutableListOf<String>() }
        private val consumeAbleKeys by lazy { mutableListOf<String>() }
        private val allProducts by lazy { mutableListOf<ProductDetails>() }
        private val purchasedSubsProductList by lazy { mutableListOf<Purchase>() }
        private val purchasedInAppProductList by lazy { mutableListOf<Purchase>() }

        private var enableLog = false
    }

    /**
     * Set the listener for billing events
     */
    fun initialize() {
        if (billingClient == null) {
            isClientReady = false
            logBilling("Setup new billing client")
            purchasesUpdatedListener = PurchasesUpdatedListener { billingResult, purchases ->
                when (billingResult.responseCode) {
                    BillingClient.BillingResponseCode.OK -> {
                        purchases?.let {
                            for (purchase in it) {
                                logBilling("purchases --> $purchase")
                                CoroutineScope(IO).launch {
                                    handlePurchase(purchase)
                                }
                            }
                            billingEventListener?.onProductsPurchased(purchasedSubsProductList)
                        }
                    }

                    BillingClient.BillingResponseCode.USER_CANCELED -> {
                        logBilling("User pressed back or canceled a dialog." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.USER_CANCELED)
                    }

                    BillingClient.BillingResponseCode.SERVICE_UNAVAILABLE -> {
                        logBilling("Network connection is down." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.SERVICE_UNAVAILABLE)

                    }

                    BillingClient.BillingResponseCode.BILLING_UNAVAILABLE -> {
                        logBilling("Billing API version is not supported for the type requested." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.BILLING_UNAVAILABLE)

                    }

                    BillingClient.BillingResponseCode.ITEM_UNAVAILABLE -> {
                        logBilling("Requested product is not available for purchase." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.ITEM_UNAVAILABLE)

                    }

                    BillingClient.BillingResponseCode.DEVELOPER_ERROR -> {
                        logBilling("Invalid arguments provided to the API." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.DEVELOPER_ERROR)

                    }

                    BillingClient.BillingResponseCode.ERROR -> {
                        logBilling("Fatal error during the API action." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.ERROR)
                    }

                    BillingClient.BillingResponseCode.ITEM_ALREADY_OWNED -> {
                        logBilling("Failure to purchase since item is already owned." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.ITEM_ALREADY_OWNED)
                    }

                    BillingClient.BillingResponseCode.ITEM_NOT_OWNED -> {
                        logBilling("Failure to consume since item is not owned." + " Response code: " + billingResult.responseCode)
                        billingEventListener?.onBillingError(ErrorType.ITEM_NOT_OWNED)
                    }

                    BillingClient.BillingResponseCode.SERVICE_DISCONNECTED, BillingClient.BillingResponseCode.SERVICE_TIMEOUT -> {
                        logBilling("Initialization error: service disconnected/timeout. Trying to reconnect...")
                        billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
                    }

                    else -> {
                        logBilling("Initialization error: ")
                        billingEventListener?.onBillingError(ErrorType.ERROR)
                    }
                }
            }
            billingClient = BillingClient.newBuilder(context)
                .setListener(purchasesUpdatedListener!!)
                .enablePendingPurchases().build()
            startConnection()
        } else {
            billingClientListener?.onClientAllReadyConnected()
            logBilling("Client already connected")
        }
    }

    /**
     * Set the listener for billing events
     */
    private fun startConnection() {

        logBilling("Connect start with Google Play")
        billingClient?.startConnection(object : BillingClientStateListener {
            override fun onBillingSetupFinished(billingResult: BillingResult) {
                if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                    logBilling("Connected to Google Play")
                    isClientReady = true

                    CoroutineScope(Main).launch {
                        // Define CompletableDeferred for each async task
                        val subsDeferred = CompletableDeferred<Unit>()
                        val inAppDeferred = CompletableDeferred<Unit>()
                        val purchasesDeferred = CompletableDeferred<Unit>()

                        // Fetch subscriptions
                        withContext(IO) {
                            if (subKeys.isNotEmpty()) {
                                fetchAvailableAllSubsProducts(subKeys, subsDeferred)
                            } else {
                                subsDeferred.complete(Unit)
                            }
                        }

                        // Fetch in-app products
                        withContext(IO) {
                            if (inAppKeys.isNotEmpty()) {
                                fetchAvailableAllInAppProducts(inAppKeys, inAppDeferred)
                            } else {
                                inAppDeferred.complete(Unit)
                            }
                        }

                        // Fetch active purchases
                        withContext(IO) {
                            fetchActivePurchases(purchasesDeferred)
                        }

                        // Await all CompletableDeferred to complete
                        awaitAll(subsDeferred, inAppDeferred, purchasesDeferred)

                        // Notify the listener on the Main thread
                        withContext(Main) {
                            logBilling("Billing client is ready")
                            billingClientListener?.onClientReady()
                        }
                    }

                } else if (billingResult.responseCode == BillingClient.BillingResponseCode.BILLING_UNAVAILABLE) {
                    billingClientListener?.onPurchasesUpdated()
                }
            }

            override fun onBillingServiceDisconnected() {
                logBilling("Fail to connect with Google Play")
                isClientReady = false

                // callback with Main thread because billing throw it in IO thread
                CoroutineScope(Main).launch {
                    billingClientListener?.onClientInitError()
                }
            }
        })
    }

    /**
     * Set the listener for billing events
     */
    private fun fetchAvailableAllSubsProducts(
        productListKeys: MutableList<String>,
        subsDeferred: CompletableDeferred<Unit>
    ) {
        // Early return if billing client is null
        val client = billingClient ?: run {
            logBilling("Billing client null while fetching All Subscription Products")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
            subsDeferred.complete(Unit)
            return
        }

        // Create a list of QueryProductDetailsParams.Product from the productListKeys
        val productList = productListKeys.map {
            logBilling("Subscription key: $it")
            QueryProductDetailsParams.Product.newBuilder()
                .setProductId(it)
                .setProductType(BillingClient.ProductType.SUBS)
                .build()
        }

        // Build the QueryProductDetailsParams
        val queryProductDetailsParams = QueryProductDetailsParams.newBuilder()
            .setProductList(productList)
            .build()

        // Query product details asynchronously
        client.queryProductDetailsAsync(queryProductDetailsParams) { billingResult, productDetailsList ->

            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {

                productDetailsList.forEach { productDetails ->
                    logBilling("Subscription product details: $productDetails")
                    allProducts.add(productDetails)
                }
            } else {
                logBilling("Failed to retrieve SUBS prices: ${billingResult.debugMessage}")
            }

            subsDeferred.complete(Unit)

        }
    }

    /**
     * Subscribe product: Subscribe to a Subscription, Subscribe to a offer
     * @param activity: Activity
     * @param productId: String
     * @param offerId: String
     */
    fun subscribe(activity: Activity, productId: String, offerId: String = "") {
        if (billingClient != null) {
            val productInfo = getProductDetail(productId, offerId, BillingClient.ProductType.SUBS)
            if (productInfo != null) {
                val productDetailsParamsList = ArrayList<BillingFlowParams.ProductDetailsParams>()
                if (productInfo.productType == BillingClient.ProductType.SUBS && productInfo.subscriptionOfferDetails != null) {
                    val offerToken =
                        getOfferToken(productInfo.subscriptionOfferDetails, productId, offerId)
                    if (offerToken.trim { it <= ' ' } != "") {
                        productDetailsParamsList.add(
                            BillingFlowParams.ProductDetailsParams.newBuilder()
                                .setProductDetails(productInfo).setOfferToken(offerToken).build()
                        )
                    } else {
                        billingEventListener?.onBillingError(ErrorType.OFFER_NOT_EXIST)
                        logBilling("The offer id: $productId doesn't seem to exist on Play Console")
                        return
                    }
                } else {
                    productDetailsParamsList.add(
                        BillingFlowParams.ProductDetailsParams.newBuilder()
                            .setProductDetails(productInfo).build()
                    )
                }
                val billingFlowParams = BillingFlowParams.newBuilder()
                    .setProductDetailsParamsList(productDetailsParamsList).build()
                billingClient!!.launchBillingFlow(activity, billingFlowParams)
            } else {
                billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
                logBilling("Billing client can not launch billing flow because product details are missing")
            }
        } else {
            logBilling("Billing client null while purchases")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
        }
    }

    /**
     * Upgrade or Downgrade Subscription
     * @param activity: Activity
     * @param updateProductId: String
     * @param updateOfferId: String
     * @param oldProductID: String
     * @param policy: Int
     *
     * Policy values:
     * 1. DEFERRED
     *    Replacement takes effect when the old plan expires, and the new price will be charged at the same time.
     * 2. IMMEDIATE_AND_CHARGE_FULL_PRICE
     *    Replacement takes effect immediately, and the user is charged full price of new plan and is given a full billing cycle of subscription, plus remaining prorated time from the old plan.
     * 3. IMMEDIATE_AND_CHARGE_PRORATED_PRICE
     *    Replacement takes effect immediately, and the billing cycle remains the same.
     * 4. IMMEDIATE_WITHOUT_PRORATION
     *    Replacement takes effect immediately, and the new price will be charged on next recurrence time.
     * 5. IMMEDIATE_WITH_TIME_PRORATION
     *    Replacement takes effect immediately, and the remaining time will be prorated and credited to the user.
     * 6. UNKNOWN_SUBSCRIPTION_UPGRADE_DOWNGRADE_POLICY
     *
     * Example:
     * FunSolBillingHelper(this).upgradeOrDowngradeSubscription(this, "New Base Plan ID", "New Offer Id (If offer)", "Old Base Plan ID", BillingFlowParams.ProrationMode.IMMEDIATE_AND_CHARGE_FULL_PRICE)
     */
    private fun upgradeOrDowngradeSubscription(
        activity: Activity,
        updateProductId: String,
        updateOfferId: String,
        oldProductID: String,
        policy: Int
    ) {

        if (billingClient != null) {
            val productInfo =
                getProductDetail(updateProductId, updateOfferId, BillingClient.ProductType.SUBS)
            if (productInfo != null) {
                val oldToken = getOldPurchaseToken(oldProductID)
                if (oldToken.trim().isNotEmpty()) {
                    val productDetailsParamsList =
                        ArrayList<BillingFlowParams.ProductDetailsParams>()
                    if (productInfo.productType == BillingClient.ProductType.SUBS && productInfo.subscriptionOfferDetails != null) {
                        val offerToken = getOfferToken(
                            productInfo.subscriptionOfferDetails, updateProductId, updateOfferId
                        )
                        if (offerToken.trim { it <= ' ' } != "") {
                            productDetailsParamsList.add(
                                BillingFlowParams.ProductDetailsParams.newBuilder()
                                    .setProductDetails(productInfo).setOfferToken(offerToken)
                                    .build()
                            )
                        } else {
                            billingEventListener?.onBillingError(ErrorType.OFFER_NOT_EXIST)
                            logBilling("The offer id: $updateProductId doesn't seem to exist on Play Console")
                            return
                        }
                    } else {
                        productDetailsParamsList.add(
                            BillingFlowParams.ProductDetailsParams.newBuilder()
                                .setProductDetails(productInfo).build()
                        )
                    }
                    val billingFlowParams = BillingFlowParams.newBuilder()
                        .setProductDetailsParamsList(productDetailsParamsList)
                        .setSubscriptionUpdateParams(
                            BillingFlowParams.SubscriptionUpdateParams.newBuilder()
                                .setOldPurchaseToken(oldToken)
                                .setSubscriptionReplacementMode(policy)
                                .build()
                        ).build()
                    billingClient!!.launchBillingFlow(activity, billingFlowParams)
                } else {
                    logBilling("old purchase token not found")
                    billingEventListener?.onBillingError(ErrorType.OLD_PURCHASE_TOKEN_NOT_FOUND)

                }
            } else {
                logBilling("Billing client can not launch billing flow because product details are missing while update")
                billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
            }
        } else {
            logBilling("Billing client null while Update subs")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
        }
    }

    /**
     * get old purchase token
     * @param basePlanKey: String
     */
    private fun getOldPurchaseToken(basePlanKey: String): String {
        // Find the product that matches the subscription and base plan key
        val matchingProduct = allProducts.firstOrNull { product ->
            product.productType == BillingClient.ProductType.SUBS && product.subscriptionOfferDetails?.any { it.basePlanId == basePlanKey } == true
        }

        // If a matching product is found, find the corresponding purchase token
        matchingProduct?.let { product ->
            val matchingPurchase = purchasedSubsProductList.firstOrNull { purchase ->
                purchase.products.firstOrNull() == product.productId
            }
            return matchingPurchase?.purchaseToken ?: ""
        }

        // Return empty string if no matching product or purchase is found
        return ""
    }

    /**
     * get Offer token
     * @param offerList: List<ProductDetails.SubscriptionOfferDetails>?
     * @param productId: String
     * @param offerId: String
     */
    private fun getOfferToken(
        offerList: List<ProductDetails.SubscriptionOfferDetails>?,
        productId: String,
        offerId: String
    ): String {
        for (product in offerList!!) {
            if (product.offerId != null && product.offerId == offerId && product.basePlanId == productId) {
                return product.offerToken
            } else if (offerId.trim { it <= ' ' } == "" && product.basePlanId == productId && product.offerId == null) {
                // case when no offer in base plan
                return product.offerToken
            }
        }
        logBilling("No Offer find")
        return ""
    }

    /**
     * Set subscription keys
     * @param keysList: MutableList<String>
     */
    fun setSubKeys(keysList: MutableList<String>): BillingManager {
        subKeys.addAll(keysList)
        return this
    }

    /**
     * For check if any subscription is subscribe
     * @return Boolean
     */
    fun isSubsPremiumUser(): Boolean {
        return purchasedSubsProductList.isNotEmpty()
    }

    /**
     * For check if any specific subscription is subscribe (by Base Plan ID)
     * @param basePlanKey: String
     */
    fun isSubsPremiumUserByBasePlanKey(basePlanKey: String): Boolean {
        val isPremiumUser = allProducts.any { product ->
            product.productType == BillingClient.ProductType.SUBS &&
                    product.subscriptionOfferDetails?.any { it.basePlanId == basePlanKey } == true &&
                    purchasedSubsProductList.any { it.products.firstOrNull() == product.productId }
        }

        if (!isPremiumUser) {
            billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
        }

        return isPremiumUser
    }

    /**
     * For check if any specific subscription is subscribe (by Subscription ID)
     * @param subId: String
     * @return Boolean
     */
    fun isSubsPremiumUserBySubIDKey(subId: String): Boolean {
        return purchasedSubsProductList.any { it.products.first() == subId }
    }


    /**
     * Check subscription support
     * @return Boolean
     */
    fun areSubscriptionsSupported(): Boolean {
        return if (billingClient != null) {
            val responseCode =
                billingClient!!.isFeatureSupported(BillingClient.FeatureType.SUBSCRIPTIONS)
            responseCode.responseCode == BillingClient.BillingResponseCode.OK
        } else {
            logBilling("billing client null while check subscription support ")
            billingEventListener?.onBillingError(ErrorType.BILLING_UNAVAILABLE)

            false
        }
    }

    /**
     * Cancel Subscription
     * @param activity: Activity
     * @param SubId: String
     */
    fun unsubscribe(activity: Activity, SubId: String) {
        try {
            val subscriptionUrl =
                "http://play.google.com/store/account/subscriptions?package=" + activity.packageName + "&sku=" + SubId
            val intent = Intent()
            intent.action = Intent.ACTION_VIEW
            intent.data = Uri.parse(subscriptionUrl)
            activity.startActivity(intent)
            activity.finish()
        } catch (e: Exception) {
            logBilling("Handling subscription cancellation: error while trying to unsubscribe")
            e.printStackTrace()
        }
    }

    //////////////////////////////////////////////////// In-App /////////////////////////////////////////////////////////////

    /**
     * Buy In-App Product
     * Subscribe to a Subscription
     * @param activity: Activity
     * @param productId: String
     * @param isPersonalizedOffer: Boolean
     */
    fun buyInApp(activity: Activity, productId: String, isPersonalizedOffer: Boolean = false) {
        val client = billingClient ?: run {
            logBilling("Error: Billing client is null.")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
            return
        }

        val productInfo = getProductDetail(productId, "", BillingClient.ProductType.INAPP)
        if (productInfo != null) {
            val productDetailsParamsList = listOf(
                BillingFlowParams.ProductDetailsParams.newBuilder()
                    .setProductDetails(productInfo)
                    .build()
            )
            val billingFlowParams = BillingFlowParams.newBuilder()
                .setProductDetailsParamsList(productDetailsParamsList)
                .setIsOfferPersonalized(isPersonalizedOffer)
                .build()

            client.launchBillingFlow(activity, billingFlowParams)
            logBilling("Initiating purchase for IN-APP product: $productId")
        } else {
            logBilling("Error: IN-APP product details missing for product ID: $productId")
            billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
        }
    }

//    private fun fetchActiveInAppPurchasesHistory() {
//        val params = QueryPurchaseHistoryParams.newBuilder().setProductType(BillingClient.ProductType.INAPP)
//        billingClient?.queryPurchaseHistoryAsync(params.build()) { billingResult, purchases ->
//            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
//                logFunsolBilling("In-APP History item already buy founded list size ${purchases?.size}")
//                if (purchases?.isEmpty() == true) {
//                    billingClientListener?.onPurchasesListReady(purchasedProductList)
//                }
//                purchases?.forEach {
//                    logFunsolBilling("In-APP History already buy products : ${it.products.first()}")
//                    purchasedInAppProductList.add(it)
//                    billingClientListener?.onPurchasesListReady(purchasedProductList)
//                }
//            } else {
//                logFunsolBilling("No In-APP History item already buy")
//            }
//
//        }
//    }

    private fun fetchAvailableAllInAppProducts(
        productListKeys: MutableList<String>,
        inAppDeferred: CompletableDeferred<Unit>
    ) {
        // Early return if billing client is null
        val client = billingClient ?: run {
            logBilling("Billing client null while fetching All In-App Products")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
            inAppDeferred.complete(Unit)
            return
        }

        val productList = productListKeys.map {
            logBilling("In-App key: $it")
            QueryProductDetailsParams.Product.newBuilder()
                .setProductId(it)
                .setProductType(BillingClient.ProductType.INAPP)
                .build()
        }

        val queryProductDetailsParams = QueryProductDetailsParams.newBuilder()
            .setProductList(productList)
            .build()

        client.queryProductDetailsAsync(queryProductDetailsParams) { billingResult, productDetailsList ->
            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                productDetailsList.forEach { productDetails ->
                    logBilling("In-app product details: $productDetails")
                    allProducts.add(productDetails)
                }
            } else {
                logBilling("Failed to retrieve In-APP prices: ${billingResult.debugMessage}")
            }
            inAppDeferred.complete(Unit)
        }
    }

    fun isInAppPremiumUser(): Boolean {
        return purchasedInAppProductList.isNotEmpty()
    }

    fun isInAppPremiumUserByInAppKey(inAppKey: String): Boolean {
        return purchasedInAppProductList.any { purchase ->
            purchase.products.any { it == inAppKey }
        }
    }

    fun setInAppKeys(keysList: MutableList<String>): BillingManager {
        inAppKeys.addAll(keysList)
        return this
    }

    fun setConsumableKeys(keysList: MutableList<String>): BillingManager {
        consumeAbleKeys.addAll(keysList)
        return this
    }

    ///////////////////////////////////////////////// Common ////////////////////////////////////////////////////////////

    fun getAllProductPrices(): MutableList<ProductPriceInfo> {
        val priceList = mutableListOf<ProductPriceInfo>()

        // Place try catch because billing internal class throw null pointer some time on ProductType
        try {
            allProducts.forEach {

                if (it.productType == BillingClient.ProductType.INAPP) {
                    val productPrice = ProductPriceInfo()
                    productPrice.title = it.title
                    productPrice.type = it.productType
                    productPrice.subsKey = it.productId
                    productPrice.productBasePlanKey = ""
                    productPrice.productOfferKey = ""
                    productPrice.price = it.oneTimePurchaseOfferDetails?.formattedPrice.toString()
                    productPrice.priceMicro =
                        it.oneTimePurchaseOfferDetails?.priceAmountMicros ?: 0L
                    productPrice.currencyCode =
                        it.oneTimePurchaseOfferDetails?.priceCurrencyCode.toString()
                    productPrice.duration = "lifeTime"
                    priceList.add(productPrice)
                } else {
                    it.subscriptionOfferDetails?.forEach { subIt ->
                        val productPrice = ProductPriceInfo()
                        productPrice.title = it.title
                        productPrice.type = it.productType
                        productPrice.subsKey = it.productId
                        productPrice.productBasePlanKey = subIt.basePlanId
                        productPrice.productOfferKey = subIt.offerId.toString()
                        productPrice.price =
                            subIt.pricingPhases.pricingPhaseList.first().formattedPrice
                        productPrice.priceMicro =
                            subIt.pricingPhases.pricingPhaseList.first().priceAmountMicros ?: 0L
                        productPrice.currencyCode =
                            subIt.pricingPhases.pricingPhaseList.first().priceCurrencyCode.toString()
                        productPrice.duration =
                            subIt.pricingPhases.pricingPhaseList.first().billingPeriod
                        priceList.add(productPrice)
                    }

                }
            }
        } catch (e: java.lang.Exception) {
            return mutableListOf()
        } catch (e: Exception) {
            return mutableListOf()
        }

        return priceList
    }

    fun getProductPriceByKey(basePlanKey: String, offerKey: String): ProductPriceInfo? {
        // Place try catch because billing internal class throw null pointer some time on ProductType
        try {
            allProducts.forEach {
                if (it.productType == BillingClient.ProductType.SUBS) {
                    it.subscriptionOfferDetails?.forEach { subIt ->
                        if (offerKey.trim().isNotEmpty()) {
                            if (subIt.basePlanId == basePlanKey && subIt.offerId == offerKey) {
                                val productPrice = ProductPriceInfo()
                                productPrice.title = it.title
                                productPrice.type = it.productType
                                productPrice.subsKey = it.productId
                                productPrice.productBasePlanKey = subIt.basePlanId
                                productPrice.productOfferKey = subIt.offerId.toString()
                                productPrice.price =
                                    subIt.pricingPhases.pricingPhaseList.first().formattedPrice
                                productPrice.priceMicro =
                                    subIt.pricingPhases.pricingPhaseList.first().priceAmountMicros
                                        ?: 0L
                                productPrice.currencyCode =
                                    subIt.pricingPhases.pricingPhaseList.first().priceCurrencyCode.toString()
                                productPrice.duration =
                                    subIt.pricingPhases.pricingPhaseList.first().billingPeriod
                                return productPrice
                            }
                        } else {
                            if (subIt.basePlanId == basePlanKey && subIt.offerId == null) {
                                val productPrice = ProductPriceInfo()
                                productPrice.title = it.title
                                productPrice.type = it.productType
                                productPrice.subsKey = it.productId
                                productPrice.productBasePlanKey = subIt.basePlanId
                                productPrice.productOfferKey = subIt.offerId.toString()
                                productPrice.price =
                                    subIt.pricingPhases.pricingPhaseList.first().formattedPrice
                                productPrice.priceMicro =
                                    subIt.pricingPhases.pricingPhaseList.first().priceAmountMicros
                                        ?: 0L
                                productPrice.currencyCode =
                                    subIt.pricingPhases.pricingPhaseList.first().priceCurrencyCode.toString()
                                productPrice.duration =
                                    subIt.pricingPhases.pricingPhaseList.first().billingPeriod
                                return productPrice
                            }
                        }
                    }
                }

            }
        } catch (e: java.lang.Exception) {
            ///leave blank because below code auto handle this
        } catch (e: Exception) {
            ///leave blank because below code auto handle this
        }
        logBilling("SUBS Product Price not found because product is missing")
        billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
        return null
    }

    fun getProductPriceByKey(productKey: String): ProductPriceInfo? {
        // Place try catch because billing internal class throw null pointer some time on ProductType
        try {
            allProducts.forEach {
                if (it.productType == BillingClient.ProductType.INAPP) {
                    if (it.productId == productKey) {
                        val productPrice = ProductPriceInfo()
                        productPrice.title = it.title
                        productPrice.type = it.productType
                        productPrice.subsKey = it.productId
                        productPrice.productBasePlanKey = ""
                        productPrice.productOfferKey = ""
                        productPrice.price =
                            it.oneTimePurchaseOfferDetails?.formattedPrice.toString()

                        productPrice.priceMicro =
                            it.oneTimePurchaseOfferDetails?.priceAmountMicros ?: 0L
                        productPrice.currencyCode =
                            it.oneTimePurchaseOfferDetails?.priceCurrencyCode.toString()
                        productPrice.duration = "lifeTime"
                        return productPrice
                    }
                }

            }
        } catch (e: java.lang.Exception) {
            ///leave blank because below code auto handle this
        } catch (e: Exception) {
            ///leave blank because below code auto handle this
        }
        logBilling("IN-APP Product Price not found because product is missing")
        billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
        return null
    }

    private fun handlePurchase(purchase: Purchase) {
        // Ensure billingClient is not null
        val billingClient = billingClient ?: run {
            logBilling("Billing client is null while handling purchases")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
            return
        }

        // Get the product type of the purchase
        val productType = getProductType(purchase.products.first())

        // Handle non-purchased states
        if (purchase.purchaseState != Purchase.PurchaseState.PURCHASED) {
            logBilling("No item purchased: ${purchase.packageName}")
            if (purchase.purchaseState == Purchase.PurchaseState.PENDING) {
                logBilling("Purchase is pending, cannot acknowledge until purchased")
                billingEventListener?.onBillingError(ErrorType.ACKNOWLEDGE_WARNING)
            }
            return
        }

        // Handle purchase acknowledgment
        if (!purchase.isAcknowledged) {
            acknowledgePurchase(billingClient, purchase, productType)
        } else {
            logBilling("Item already acknowledged")
            purchasedSubsProductList.add(purchase)
            billingClientListener?.onPurchasesUpdated()
        }

        // Handle consumable purchases
        if (consumeAbleKeys.contains(purchase.products.first())) {
            consumePurchase(billingClient, purchase)
        } else {
            logBilling("This purchase is not consumable")
        }
    }

    // Helper function to acknowledge a purchase
    private fun acknowledgePurchase(
        billingClient: BillingClient,
        purchase: Purchase,
        productType: String
    ) {
        val acknowledgePurchaseParams = AcknowledgePurchaseParams.newBuilder()
            .setPurchaseToken(purchase.purchaseToken)
            .build()

        billingClient.acknowledgePurchase(acknowledgePurchaseParams) {
            if (it.responseCode == BillingClient.BillingResponseCode.OK) {
                logBilling("$productType item acknowledged")
                // Add purchase to the appropriate list
                if (productType.trim().isNotEmpty()) {
                    if (productType == BillingClient.ProductType.INAPP) {
                        purchasedInAppProductList.add(purchase)
                    } else {
                        purchasedSubsProductList.add(purchase)
                    }
                    billingClientListener?.onPurchasesUpdated()
                } else {
                    logBilling("Product type not found while handling purchase")
                }
                billingEventListener?.onPurchaseAcknowledged(purchase)
            } else {
                logBilling("Acknowledge error: ${it.debugMessage} (code: ${it.responseCode})")
                billingEventListener?.onBillingError(ErrorType.ACKNOWLEDGE_ERROR)
            }
        }
    }

    // Helper function to consume a purchase
    private fun consumePurchase(billingClient: BillingClient, purchase: Purchase) {
        val consumeParams =
            ConsumeParams.newBuilder().setPurchaseToken(purchase.purchaseToken).build()
        billingClient.consumeAsync(consumeParams) { result, _ ->
            if (result.responseCode == BillingClient.BillingResponseCode.OK) {
                logBilling("Purchase consumed")
                billingEventListener?.onPurchaseConsumed(purchase)
            } else {
                logBilling("Failed to consume purchase: ${result.debugMessage} (code: ${result.responseCode})")
                billingEventListener?.onBillingError(ErrorType.CONSUME_ERROR)
            }
        }
    }

    fun fetchActivePurchases(purchasesDeferred: CompletableDeferred<Unit> = CompletableDeferred()) {
        fetchAndUpdateActivePurchases(purchasesDeferred)
//        fetchActiveInAppPurchasesHistory()
    }

    private fun fetchAndUpdateActivePurchases(purchasesDeferred: CompletableDeferred<Unit>) {
        val billingClient = billingClient
        if (billingClient == null) {
            logBilling("Billing client is null while fetching active purchases")
            billingEventListener?.onBillingError(ErrorType.SERVICE_DISCONNECTED)
            purchasesDeferred.complete(Unit)
            return
        }

        val scope = CoroutineScope(IO)

        fun handleBillingResult(
            billingResult: BillingResult,
            purchases: List<Purchase>,
            productType: String,
            purchasesDeferred: CompletableDeferred<Unit>
        ) {
            if (billingResult.responseCode == BillingClient.BillingResponseCode.OK) {
                val activePurchases =
                    purchases.filter { it.purchaseState == Purchase.PurchaseState.PURCHASED }
                logBilling("$productType purchases found: ${activePurchases.size}")

                if (activePurchases.isEmpty()) {
                    billingClientListener?.onPurchasesUpdated()
                    purchasesDeferred.complete(Unit)
                    return
                }

                scope.launch {
                    activePurchases.forEach { purchase ->
                        logBilling("$productType purchase: ${purchase.products.first()}")
                        handlePurchase(purchase)
                    }
                    purchasesDeferred.complete(Unit)
                }
            } else {
                logBilling("No $productType purchases found")
            }
        }

        billingClient.queryPurchasesAsync(
            QueryPurchasesParams.newBuilder().setProductType(BillingClient.ProductType.SUBS).build()
        ) { billingResult, purchases ->
            handleBillingResult(billingResult, purchases, "SUBS", purchasesDeferred)
        }

        billingClient.queryPurchasesAsync(
            QueryPurchasesParams.newBuilder().setProductType(BillingClient.ProductType.INAPP)
                .build()
        ) { billingResult, purchases ->
            handleBillingResult(billingResult, purchases, "IN-APP", purchasesDeferred)
        }
    }

    fun getProductDetail(
        productKey: String,
        offerKey: String = "",
        productType: String
    ): ProductDetails? {

        val offerKeyNormalized = offerKey.trim().takeIf { it.isNotEmpty() } ?: "null"

        if (allProducts.isEmpty()) {
            billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
            return null
        }

        val product = allProducts.find { product ->
            when (productType) {
                BillingClient.ProductType.INAPP -> {
                    if (product.productId == productKey) {
                        logBilling("In App product detail: title: ${product.title} price: ${product.oneTimePurchaseOfferDetails?.formattedPrice}")
                        true
                    } else {
                        false
                    }
                }

                BillingClient.ProductType.SUBS -> {
                    product.subscriptionOfferDetails?.any { subDetails ->
                        val isMatchingBasePlan = subDetails.basePlanId.equals(productKey, true)
                        val isMatchingOfferId =
                            subDetails.offerId.toString().equals(offerKeyNormalized, true)
                        if (isMatchingBasePlan && isMatchingOfferId) {
                            logBilling("Subscription product detail: basePlanId: ${subDetails.basePlanId} offerId: ${subDetails.offerId}")
                        }
                        isMatchingBasePlan && isMatchingOfferId
                    } ?: false
                }

                else -> false
            }
        }

        if (product == null) {
            billingEventListener?.onBillingError(ErrorType.PRODUCT_NOT_EXIST)
        }

        return product
    }

    private fun getProductType(productKey: String): String {
        allProducts.forEach { productDetail ->
            if (productDetail.productType == BillingClient.ProductType.INAPP) {
                if (productDetail.productId == productKey) {
                    return productDetail.productType
                }
            } else {
                productDetail.subscriptionOfferDetails?.forEach {
                    if (it.basePlanId == productKey) {
                        return productDetail.productType
                    }
                }
            }
        }
        return ""
    }

    fun isClientReady(): Boolean {
        return isClientReady
    }

    fun enableLogging(isEnableLog: Boolean = true): BillingManager {
        enableLog = isEnableLog
        return this
    }

    private fun logBilling(message: String) {
        if (enableLog) {
            Timber.tag(TAG).d(message)
        }
    }

    fun release() {
        if (billingClient != null && billingClient!!.isReady) {
            logBilling("BillingHelper instance release: ending connection...")
            billingClient?.endConnection()
        }
    }

    fun setBillingEventListener(billingEventListeners: BillingEventListener?): BillingManager {
        billingEventListener = billingEventListeners
        return this
    }

    fun setBillingClientListener(billingClientListeners: BillingClientListener?): BillingManager {
        billingClientListener = billingClientListeners
        return this
    }
}