/*
 * Author: Mikhail Fedotov
 * Github: https://github.com/KStateMachine
 * Copyright (c) 2024.
 * All rights reserved.
 */

package ru.nsk.kstatemachine.event

import ru.nsk.kstatemachine.state.DataState
import ru.nsk.kstatemachine.state.FinalDataState
import ru.nsk.kstatemachine.state.IState
import ru.nsk.kstatemachine.statemachine.StateMachine
import ru.nsk.kstatemachine.statemachine.undo

/**
 * Base interface for events which may trigger transitions of [StateMachine]
 * Events are expected to be immutable subjects by design.
 */
interface Event

/**
 * Event holding some data
 */
interface DataEvent<out D : Any> : Event {
    val data: D
}

/**
 * User may call [StateMachine.processEvent] with [UndoEvent] as alternative to calling [StateMachine.undo]
 */
object UndoEvent : Event

/**
 * Marker interface for all events that are generated by the library itself
 */
sealed interface GeneratedEvent : Event

/**
 * Special event generated by the library when a state is finished.
 * Transitions use special event matcher by default to match only related events.
 * If [FinishedEvent] is generated by [FinalDataState] entry, [data] field of event will receive data
 * from this state using [DataExtractor]
 */
class FinishedEvent internal constructor(val state: IState, val data: Any? = null) : GeneratedEvent

/**
 * Initial event which is processed on state machine start
 */
sealed interface StartEvent : GeneratedEvent {
    val startState: IState
}

/**
 * [startStates] must contain at least one state. If there are multiple states they must be sub-children of a parallel
 * state.
 */
internal class StartEventImpl(val startStates: Set<IState>) : StartEvent {
    override val startState = startStates.first()
}

internal class StartDataEventImpl<D : Any>(
    override val startState: DataState<D>,
    override val data: D
) : StartEvent, DataEvent<D>

internal object StopEvent : GeneratedEvent

/**
 * This event is processed even if the [StateMachine] is already stopped
 */
internal class DestroyEvent(val stop: Boolean) : GeneratedEvent

/**
 * System event which is used by the library to wrap original event and argument,
 * so user may access them, when this event is processed.
 * Currently only [UndoEvent] is transformed to this event.
 * @param event original event
 * @param argument original argument
 */
class WrappedEvent(val event: Event, val argument: Any?) : GeneratedEvent

/**
 * Special kind of event, which is not processed by a stateMachine itself but used to
 * represent different kinds of [GeneratedEvent] in serialized form for event recording feature.
 */
data class SerializableGeneratedEvent(val eventType: EventType) : GeneratedEvent {
    sealed interface EventType {
        object Start: EventType
        object Stop: EventType
        class Destroy(val stop: Boolean) : EventType
    }
}