/*
 * Copyright (c) 2020.
 *
 * This file is part of xmlutil.
 *
 * This file is licenced to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You should have received a copy of the license with the source distribution.
 * Alternatively, you may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package nl.adaptivity.xmlutil.serialization

import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.KSerializer
import kotlinx.serialization.SerializationException
import kotlinx.serialization.builtins.nullable
import kotlinx.serialization.descriptors.*
import kotlinx.serialization.modules.SerializersModule
import nl.adaptivity.xmlutil.*
import nl.adaptivity.xmlutil.core.impl.multiplatform.assert
import nl.adaptivity.xmlutil.serialization.XmlSerializationPolicy.DeclaredNameInfo
import nl.adaptivity.xmlutil.serialization.XmlSerializationPolicy.XmlEncodeDefault
import nl.adaptivity.xmlutil.serialization.impl.XmlQNameSerializer
import nl.adaptivity.xmlutil.serialization.structure.*


/**
 *
 */
public interface XmlSerializationPolicy {

    public val defaultPrimitiveOutputKind: OutputKind get() = OutputKind.Attribute
    public val defaultObjectOutputKind: OutputKind get() = OutputKind.Element

    public val isStrictNames: Boolean get() = false
    public val isStrictBoolean: Boolean get() = false

    @ExperimentalXmlUtilApi
    public val verifyElementOrder: Boolean get() = false

    @OptIn(ExperimentalSerializationApi::class)
    @ExperimentalXmlUtilApi
    public fun defaultOutputKind(serialKind: SerialKind): OutputKind =
        when (serialKind) {
            SerialKind.ENUM,
            StructureKind.OBJECT -> defaultObjectOutputKind

            is PrimitiveKind -> defaultPrimitiveOutputKind
            PolymorphicKind.OPEN -> OutputKind.Element
            else -> OutputKind.Element
        }

    public fun invalidOutputKind(message: String): Unit = ignoredSerialInfo(message)

    public fun ignoredSerialInfo(message: String)

    public fun effectiveName(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo,
        outputKind: OutputKind,
        useName: DeclaredNameInfo = tagParent.elementUseNameInfo
    ): QName

    public fun isListEluded(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): Boolean

    public fun isTransparentPolymorphic(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): Boolean

    public fun polymorphicDiscriminatorName(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): QName?

    @Suppress("DEPRECATION")
    public fun serialTypeNameToQName(
        typeNameInfo: DeclaredNameInfo,
        parentNamespace: Namespace
    ): QName =
        serialNameToQName(typeNameInfo.serialName, parentNamespace)

    @Suppress("DEPRECATION")
    public fun serialUseNameToQName(
        useNameInfo: DeclaredNameInfo,
        parentNamespace: Namespace
    ): QName =
        serialNameToQName(useNameInfo.serialName, parentNamespace)

    @Deprecated("It is recommended to override serialTypeNameToQName and serialUseNameToQName instead")
    public fun serialNameToQName(
        serialName: String,
        parentNamespace: Namespace
    ): QName

    public data class DeclaredNameInfo(
        val serialName: String,
        val annotatedName: QName?,
        val isDefaultNamespace: Boolean/* = false*/
    ) {
        internal constructor(serialName: String) : this(serialName, null, false)

        init {
            check(!(isDefaultNamespace && annotatedName == null)) { "Default namespace requires there to be an annotated name" }
        }
    }

    public data class ActualNameInfo(
        val serialName: String,
        val annotatedName: QName
    )

    @Deprecated("Don't use or implement this, use the 3 parameter version")
    public fun effectiveOutputKind(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): OutputKind

    public fun effectiveOutputKind(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo,
        canBeAttribute: Boolean
    ): OutputKind {
        @Suppress("DEPRECATION")
        val base = effectiveOutputKind(serializerParent, tagParent)

        if (!canBeAttribute && base == OutputKind.Attribute) {
            return handleAttributeOrderConflict(
                serializerParent,
                tagParent,
                base
            )
        }
        return base
    }

    public fun overrideSerializerOrNull(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): KSerializer<*>? {
        return null
    }

    @ExperimentalXmlUtilApi
    @Suppress("DirectUseOfResultType", "DEPRECATION")
    public fun handleUnknownContentRecovering(
        input: XmlReader,
        inputKind: InputKind,
        descriptor: XmlDescriptor,
        name: QName?,
        candidates: Collection<Any>
    ): List<XML.ParsedData<*>> {
        handleUnknownContent(input, inputKind, name, candidates)
        return emptyList()
    }

    public fun onElementRepeated(parentDescriptor: XmlDescriptor, childIndex: Int) {}

    @Deprecated("Use the recoverable version that allows returning a value")
    public fun handleUnknownContent(
        input: XmlReader,
        inputKind: InputKind,
        name: QName?,
        candidates: Collection<Any>
    )

    public fun handleAttributeOrderConflict(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo,
        outputKind: OutputKind
    ): OutputKind {
        throw SerializationException("Node ${serializerParent.elementUseNameInfo.serialName} wants to be an attribute but cannot due to ordering constraints")
    }

    public fun shouldEncodeElementDefault(elementDescriptor: XmlDescriptor?): Boolean

    /**
     * Allow modifying the ordering of children.
     */
    public fun initialChildReorderMap(
        parentDescriptor: SerialDescriptor
    ): Collection<XmlOrderConstraint>? = null

    public fun updateReorderMap(
        original: Collection<XmlOrderConstraint>,
        children: List<XmlDescriptor>
    ): Collection<XmlOrderConstraint> = original

    @OptIn(ExperimentalSerializationApi::class)
    public fun enumEncoding(enumDescriptor: SerialDescriptor, index: Int): String {
        return enumDescriptor.getElementName(index)
    }

    @ExperimentalXmlUtilApi
    public fun preserveSpace(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): Boolean {
        return true
    }

    /** Determine the name of map keys for a given map type */
    public fun mapKeyName(serializerParent: SafeParentInfo): DeclaredNameInfo =
        DeclaredNameInfo("key") // minimal default for implementations.

    /**
     * Determine the name of the values for a given map type
     */
    public fun mapValueName(serializerParent: SafeParentInfo, isListEluded: Boolean): DeclaredNameInfo =
        DeclaredNameInfo("value") // minimal default for implementations.

    /**
     * Determine the name to use for the map element (only used when a map entry is wrapped)
     */
    public fun mapEntryName(serializerParent: SafeParentInfo, isListEluded: Boolean): QName =
        QName(serializerParent.namespace.namespaceURI, "entry") // minimal default for implementations.

    /**
     * Determine whether the key attribute should be collapsed into the value tag rather
     * than the value being nested in a container for the element.
     */
    public fun isMapValueCollapsed(mapParent: SafeParentInfo, valueDescriptor: XmlDescriptor): Boolean = false

    /**
     * Determine namespace prefixes to make sure are set upon the tag.
     */
    @ExperimentalXmlUtilApi
    public fun elementNamespaceDecls(serializerParent: SafeParentInfo): List<Namespace> = emptyList()

    /**
     * Determine the delimiters to use to separate attributes. When writing will always write the
     * first element.
     */
    @ExperimentalXmlUtilApi
    public fun attributeListDelimiters(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): Array<String> =
        arrayOf(" ", "\n", "\t", "\r")

    /**
     * Determine the delimiters to use to separate primitive/textual list elements when use inside an element.
     * When writing will always write the first element.
     */
    @ExperimentalXmlUtilApi
    public fun textListDelimiters(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): Array<String> =
        attributeListDelimiters(serializerParent, tagParent)

    public enum class XmlEncodeDefault {
        ALWAYS, ANNOTATED, NEVER
    }

    public companion object {

        /**
         * Helper function that allows more flexibility on null namespace use. If either the found
         * name has the null namespace, or the candidate has null namespace, this will map (for the
         * correct child).
         */
        @ExperimentalXmlUtilApi
        public fun recoverNullNamespaceUse(
            inputKind: InputKind,
            descriptor: XmlDescriptor,
            name: QName?
        ): List<XML.ParsedData<*>>? {
            if (name != null) {
                if (name.namespaceURI == "") {
                    for (idx in 0 until descriptor.elementsCount) {
                        val candidate = descriptor.getElementDescriptor(idx)
                        if (inputKind.mapsTo(candidate.effectiveOutputKind) &&
                            candidate.tagName.localPart == name.getLocalPart()
                        ) {
                            return listOf(XML.ParsedData(idx, Unit, true))
                        }
                    }
                } else {
                    for (idx in 0 until descriptor.elementsCount) {
                        val candidate = descriptor.getElementDescriptor(idx)
                        if (inputKind.mapsTo(candidate.effectiveOutputKind) &&
                            candidate.tagName.isEquivalent(QName(name.localPart))
                        ) {
                            return listOf(XML.ParsedData(idx, Unit, true))
                        }
                    }
                }
            }
            return null
        }

    }

}

public fun XmlSerializationPolicy.typeQName(xmlDescriptor: XmlDescriptor): QName {
    return xmlDescriptor.typeDescriptor.typeQname
        ?: serialTypeNameToQName(xmlDescriptor.typeDescriptor.typeNameInfo, xmlDescriptor.tagParent.namespace)
}

/**
 * Default implementation of a serialization policy that provides a behaviour that attempts to create an XML format
 * that resembles what would be created manually in line with XML's design.
 *
 * @property pedantic Enable some stricter behaviour
 * @property autoPolymorphic Should polymorphic information be retrieved using [SerializersModule] configuration. This replaces
 *  *                     [XmlPolyChildren], but changes serialization where that annotation is not applied. This option will
 *  *                     become the default in the future although XmlPolyChildren will retain precedence (when present)
 * @property encodeDefault Determine whether defaults need to be encoded
 * @property unknownChildHandler A function that is called when an unknown child is found. By default an exception is thrown
 *  *                     but the function can silently ignore it as well.
 * @property typeDiscriminatorName When set, use a type discriminator property
 * @property throwOnRepeatedElement When a single-value elemement is repeated in the content, will this throw an
 *   exception or only retain the final value
 * @property verifyElementOrder Verify that element children are in the order required by order annotations (and
 *   fail if not correct). Note that attribute order in XML is arbitrary and not meaningful.
 * @property isStrictAttributeNames Process attribute name reading strictly according to the XML standard, or a
 *   name handling that is a bit more lenient
 * @property isStrictBoolean Parse boolean data according to the requirements of XML, rather than the (very lenient)
 *   toBoolean function from the Kotlin standard library.
 */
public open class DefaultXmlSerializationPolicy
private constructor(
    public val pedantic: Boolean,
    public val autoPolymorphic: Boolean = false,
    public val encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
    public val unknownChildHandler: UnknownChildHandler = XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER,
    public val typeDiscriminatorName: QName? = null,
    public val throwOnRepeatedElement: Boolean = false,
    public override val verifyElementOrder: Boolean = false,
    public override val isStrictNames: Boolean,
    public override val isStrictBoolean: Boolean = false,
) : XmlSerializationPolicy {

    @Deprecated("Use builder")
    @ExperimentalXmlUtilApi
    public constructor(
        pedantic: Boolean,
        autoPolymorphic: Boolean = false,
        encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
        unknownChildHandler: UnknownChildHandler = XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER,
        typeDiscriminatorName: QName? = null,
        throwOnRepeatedElement: Boolean = false,
        verifyElementOrder: Boolean = false,
    ) : this(pedantic, autoPolymorphic, encodeDefault, unknownChildHandler, typeDiscriminatorName, throwOnRepeatedElement, verifyElementOrder, false)

    @Suppress("DEPRECATION")
    @Deprecated("Use builder")
    @ExperimentalXmlUtilApi
    public constructor(
        pedantic: Boolean,
        typeDiscriminatorName: QName,
        encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
        unknownChildHandler: UnknownChildHandler = XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER,
        throwOnRepeatedElement: Boolean = false,
        verifyElementOrder: Boolean = false,
    ) : this(pedantic, false, encodeDefault, unknownChildHandler, typeDiscriminatorName, throwOnRepeatedElement, verifyElementOrder)

    /**
     * Stable constructor that doesn't use experimental api.
     */
    @Suppress("DEPRECATION")
    @Deprecated("Use builder")
    @OptIn(ExperimentalXmlUtilApi::class)
    public constructor(
        pedantic: Boolean,
        autoPolymorphic: Boolean = false,
        encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
    ) : this(pedantic, autoPolymorphic, encodeDefault, XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER)

    @Suppress("DEPRECATION")
    @Deprecated("Use the unknownChildHandler version that allows for recovery")
    @ExperimentalXmlUtilApi
    public constructor(
        pedantic: Boolean,
        autoPolymorphic: Boolean = false,
        encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
        unknownChildHandler: NonRecoveryUnknownChildHandler
    ) : this(
        pedantic,
        autoPolymorphic,
        encodeDefault,
        UnknownChildHandler { input, inputKind, _, name, candidates ->
            unknownChildHandler(input, inputKind, name, candidates); emptyList()
        }
    )

    @Suppress("DEPRECATION")
    @Deprecated("Use the primary constructor that takes the recoverable handler")
    @ExperimentalXmlUtilApi
    public constructor(
        pedantic: Boolean,
        autoPolymorphic: Boolean = false,
        unknownChildHandler: NonRecoveryUnknownChildHandler
    ) : this(pedantic, autoPolymorphic, XmlEncodeDefault.ANNOTATED, unknownChildHandler)

    @OptIn(ExperimentalXmlUtilApi::class)
    public constructor(original: XmlSerializationPolicy?) : this(
        pedantic = (original as? DefaultXmlSerializationPolicy)?.pedantic ?: false,
        autoPolymorphic = (original as? DefaultXmlSerializationPolicy)?.autoPolymorphic ?: false,
        encodeDefault = (original as? DefaultXmlSerializationPolicy)?.encodeDefault ?: XmlEncodeDefault.ANNOTATED,
        unknownChildHandler = original?.let { orig -> // If there is an original, get from it
            (orig as? DefaultXmlSerializationPolicy)?.unknownChildHandler // take the existing one if present
                ?: UnknownChildHandler { input, inputKind, descriptor, name, candidates ->
                    orig.handleUnknownContentRecovering(input, inputKind, descriptor, name, candidates)
                }
        } ?: XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER, // otherwise the default
        typeDiscriminatorName = (original as? DefaultXmlSerializationPolicy)?.typeDiscriminatorName,
        throwOnRepeatedElement = (original as? DefaultXmlSerializationPolicy)?.throwOnRepeatedElement ?: false,
        verifyElementOrder = original?.verifyElementOrder ?: false,
        isStrictNames = original?.isStrictNames ?: false,
        isStrictBoolean = original?.isStrictBoolean ?: false,
    )

    @OptIn(ExperimentalXmlUtilApi::class)
    protected constructor(builder: Builder) : this(
        pedantic = builder.pedantic,
        autoPolymorphic = builder.autoPolymorphic,
        encodeDefault = builder.encodeDefault,
        unknownChildHandler = builder.unknownChildHandler,
        typeDiscriminatorName = builder.typeDiscriminatorName,
        throwOnRepeatedElement = builder.throwOnRepeatedElement,
        verifyElementOrder = builder.verifyElementOrder,
        isStrictNames = builder.isStrictAttributeNames,
        isStrictBoolean = builder.isStrictBoolean,
    )

    public constructor(config: Builder.() -> Unit) : this(Builder().apply(config))

    override fun polymorphicDiscriminatorName(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): QName? {
        return typeDiscriminatorName
    }

    override fun isListEluded(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): Boolean {
        val useAnnotations = tagParent.elementUseAnnotations
        val isMixed = useAnnotations.firstOrNull<XmlValue>()?.value == true
        if (isMixed) return true

        val reqChildrenName =
            useAnnotations.firstOrNull<XmlChildrenName>()?.toQName()
        return reqChildrenName == null
    }

    override fun isTransparentPolymorphic(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): Boolean {
        val xmlPolyChildren =
            tagParent.elementUseAnnotations.firstOrNull<XmlPolyChildren>()
        return autoPolymorphic || xmlPolyChildren != null
    }

    @Deprecated("Don't use or implement this, use the 3 parameter version")
    override fun effectiveOutputKind(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): OutputKind {
        return effectiveOutputKind(serializerParent, tagParent, true)
    }

    @OptIn(ExperimentalSerializationApi::class, ExperimentalXmlUtilApi::class)
    override fun effectiveOutputKind(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo,
        canBeAttribute: Boolean
    ): OutputKind {
        val serialDescriptor = overrideSerializerOrNull(serializerParent, tagParent)?.descriptor
            ?: serializerParent.elementSerialDescriptor

        return when (val overrideOutputKind =
            serializerParent.elementUseOutputKind) {
            null -> {
                val useAnnotations = tagParent.elementUseAnnotations
                val isValue =
                    useAnnotations.firstOrNull<XmlValue>()?.value == true
                var parentChildDesc = tagParent.elementSerialDescriptor
                while (parentChildDesc.isInline) {
                    parentChildDesc =
                        parentChildDesc.getElementDescriptor(0)
                }
                val elementKind = parentChildDesc.kind
                // If we can't be an attribue
                when {
                    elementKind == StructureKind.CLASS
                    -> OutputKind.Element

                    isValue -> OutputKind.Mixed

                    !canBeAttribute && (tagParent.elementUseOutputKind == OutputKind.Attribute)
                    -> handleAttributeOrderConflict(serializerParent, tagParent, OutputKind.Attribute)

                    !canBeAttribute -> OutputKind.Element

                    else -> tagParent.elementUseOutputKind
                        ?: serialDescriptor.declOutputKind()
                        ?: defaultOutputKind(serialDescriptor.kind)
                }
            }

            OutputKind.Mixed -> {
                if (serializerParent.descriptor is XmlListDescriptor) {
                    if (tagParent.elementSerialDescriptor.kind == StructureKind.CLASS) {
                        OutputKind.Element
                    } else {
                        OutputKind.Mixed
                    }
                } else {
                    val outputKind = tagParent.elementUseOutputKind
                        ?: serialDescriptor.declOutputKind()
                        ?: defaultOutputKind(serialDescriptor.kind)

                    when (outputKind) {
                        OutputKind.Attribute -> OutputKind.Text
                        else -> outputKind
                    }
                }
            }

            else -> overrideOutputKind

        }
    }


    @Deprecated("It is recommended to override serialTypeNameToQName and serialUseNameToQName instead")
    override fun serialNameToQName(
        serialName: String,
        parentNamespace: Namespace
    ): QName {
        return when (serialName) {
            "kotlin.Boolean" -> QName(XMLConstants.XSD_NS_URI, "boolean", XMLConstants.XSD_PREFIX)
            "kotlin.Byte" -> QName(XMLConstants.XSD_NS_URI, "byte", XMLConstants.XSD_PREFIX)
            "kotlin.UByte" -> QName(XMLConstants.XSD_NS_URI, "unsignedByte", XMLConstants.XSD_PREFIX)
            "kotlin.Short" -> QName(XMLConstants.XSD_NS_URI, "short", XMLConstants.XSD_PREFIX)
            "kotlin.UShort" -> QName(XMLConstants.XSD_NS_URI, "unsignedShort", XMLConstants.XSD_PREFIX)
            "kotlin.Int" -> QName(XMLConstants.XSD_NS_URI, "int", XMLConstants.XSD_PREFIX)
            "kotlin.UInt" -> QName(XMLConstants.XSD_NS_URI, "unsignedInt", XMLConstants.XSD_PREFIX)
            "kotlin.Long" -> QName(XMLConstants.XSD_NS_URI, "long", XMLConstants.XSD_PREFIX)
            "kotlin.ULong" -> QName(XMLConstants.XSD_NS_URI, "unsignedLong", XMLConstants.XSD_PREFIX)
            "kotlin.Float",
            "kotlin.Double" -> QName(XMLConstants.XSD_NS_URI, "double", XMLConstants.XSD_PREFIX)

            "kotlin.String" -> QName(XMLConstants.XSD_NS_URI, "string", XMLConstants.XSD_PREFIX)

            else -> serialName.substringAfterLast('.').toQname(parentNamespace)
        }
    }

    @OptIn(ExperimentalSerializationApi::class)
    override fun effectiveName(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo,
        outputKind: OutputKind,
        useName: DeclaredNameInfo
    ): QName {
        val typeDescriptor = serializerParent.elementTypeDescriptor
        val serialKind = typeDescriptor.serialDescriptor.kind
        val typeNameInfo = typeDescriptor.typeNameInfo
        val parentNamespace: Namespace = tagParent.namespace

        assert(typeNameInfo == typeDescriptor.typeNameInfo) {
            "Type name info should match"
        }

        val parentSerialKind = tagParent.descriptor?.serialKind

        return when {
            outputKind == OutputKind.Attribute -> when {
                useName.isDefaultNamespace -> QName(useName.annotatedName?.getLocalPart() ?: useName.serialName)
                useName.annotatedName != null -> useName.annotatedName
                else -> QName(useName.serialName)
            } // Use non-prefix attributes by default

            useName.annotatedName != null -> useName.annotatedName


            serialKind is PrimitiveKind ||
                    serialKind == StructureKind.MAP ||
                    serialKind == StructureKind.LIST ||
                    serialKind == PolymorphicKind.OPEN ||
                    typeNameInfo.serialName == "kotlin.Unit" || // Unit needs a special case
                    parentSerialKind is PolymorphicKind // child of explict polymorphic uses predefined names
            -> serialUseNameToQName(useName, parentNamespace)

            typeNameInfo.annotatedName != null -> typeNameInfo.annotatedName

            else -> serialTypeNameToQName(typeNameInfo, parentNamespace)
        }
    }

    override fun shouldEncodeElementDefault(elementDescriptor: XmlDescriptor?): Boolean {
        return when (encodeDefault) {
            XmlEncodeDefault.NEVER -> false
            XmlEncodeDefault.ALWAYS -> true
            XmlEncodeDefault.ANNOTATED -> (elementDescriptor as? XmlValueDescriptor)?.default == null
        }
    }

    @ExperimentalXmlUtilApi
    @Suppress("DirectUseOfResultType")
    override fun handleUnknownContentRecovering(
        input: XmlReader,
        inputKind: InputKind,
        descriptor: XmlDescriptor,
        name: QName?,
        candidates: Collection<Any>
    ): List<XML.ParsedData<*>> {
        return unknownChildHandler.handleUnknownChildRecovering(input, inputKind, descriptor, name, candidates)
    }

    @Deprecated("Don't use anymore, use the version that allows for recovery")
    override fun handleUnknownContent(
        input: XmlReader,
        inputKind: InputKind,
        name: QName?,
        candidates: Collection<Any>
    ) {
        throw UnsupportedOperationException("this function should not be called")
    }

    override fun onElementRepeated(parentDescriptor: XmlDescriptor, childIndex: Int) {
        if (throwOnRepeatedElement) {
            throw XmlSerialException("Duplicate child (${parentDescriptor.getElementDescriptor(childIndex)} found in ${parentDescriptor} outside of eluded list context")
        }
    }

    @OptIn(ExperimentalSerializationApi::class)
    override fun overrideSerializerOrNull(
        serializerParent: SafeParentInfo,
        tagParent: SafeParentInfo
    ): KSerializer<*>? =
        when (serializerParent.elementSerialDescriptor.serialName) {
            "javax.xml.namespace.QName?",
            "javax.xml.namespace.QName" -> when {
                serializerParent.elementSerialDescriptor.isNullable -> XmlQNameSerializer.nullable
                else -> XmlQNameSerializer
            }

            else -> null
        }

    /**
     * Default implementation that uses [XmlBefore] and [XmlAfter]. It does
     * not use the parent descriptor at all.
     */
    @OptIn(ExperimentalSerializationApi::class)
    override fun initialChildReorderMap(
        parentDescriptor: SerialDescriptor
    ): Collection<XmlOrderConstraint>? {
        val nameToIdx =
            (0 until parentDescriptor.elementsCount).associateBy {
                parentDescriptor.getElementName(it)
            }

        fun String.toChildIndex(): Int = when (this) {
            "*" -> XmlOrderConstraint.OTHERS
            else -> nameToIdx[this]
                ?: throw XmlSerialException("Could not find the attribute in ${parentDescriptor.serialName} with the name: $this\n  Candidates were: ${nameToIdx.keys.joinToString()}")
        }

        val orderConstraints = HashSet<XmlOrderConstraint>()
        val orderNodes = mutableMapOf<String, XmlOrderNode>()
        for (elementIdx in 0 until parentDescriptor.elementsCount) {
            var xmlBefore: Array<out String>? = null
            var xmlAfter: Array<out String>? = null
            for (annotation in parentDescriptor.getElementAnnotations(elementIdx)) {
                if (annotation is XmlBefore && annotation.value.isNotEmpty()) {
                    annotation.value.mapTo(orderConstraints) {
                        val successorIdx = it.toChildIndex()
                        XmlOrderConstraint(elementIdx, successorIdx)
                    }
                    xmlBefore = annotation.value
                } else if (annotation is XmlAfter && annotation.value.isNotEmpty()) {
                    annotation.value.mapTo(orderConstraints) {
                        val predecessorIdx = it.toChildIndex()
                        XmlOrderConstraint(predecessorIdx, elementIdx)
                    }
                    xmlAfter = annotation.value
                }
                if (xmlBefore != null || xmlAfter != null) {
                    val node = orderNodes.getOrPut(
                        parentDescriptor.getElementName(elementIdx)
                    ) {
                        XmlOrderNode(
                            elementIdx
                        )
                    }
                    if (xmlBefore != null) {
                        val befores = Array(xmlBefore.size) {
                            val name = xmlBefore[it]
                            orderNodes.getOrPut(name) { XmlOrderNode(name.toChildIndex()) }
                        }
                        node.addSuccessors(*befores)
                    }
                    if (xmlAfter != null) {
                        val afters = Array(xmlAfter.size) {
                            val name = xmlAfter[it]
                            orderNodes.getOrPut(name) { XmlOrderNode(name.toChildIndex()) }
                        }
                        node.addPredecessors(*afters)
                    }

                }
            }
        }
        if (orderNodes.isEmpty()) return null // no order nodes, no reordering

        return if (orderConstraints.isEmpty()) null else orderConstraints.toList()
    }

    override fun updateReorderMap(
        original: Collection<XmlOrderConstraint>,
        children: List<XmlDescriptor>
    ): Collection<XmlOrderConstraint> {

        fun Int.isAttribute(): Boolean = children[this].outputKind == OutputKind.Attribute

        return original.filter { constraint ->
            val (isBeforeAttribute, isAfterAttribute) = constraint.map(Int::isAttribute)

            isBeforeAttribute || (!isAfterAttribute)
        }
    }

    @OptIn(ExperimentalSerializationApi::class)
    @ExperimentalXmlUtilApi
    override fun preserveSpace(serializerParent: SafeParentInfo, tagParent: SafeParentInfo): Boolean {
        serializerParent.elementUseAnnotations.firstOrNull<XmlIgnoreWhitespace>()?.apply { return !value }
        return !(serializerParent.elementSerialDescriptor.annotations
            .firstOrNull<XmlIgnoreWhitespace>()?.value ?: false)
    }

    override fun mapKeyName(serializerParent: SafeParentInfo): DeclaredNameInfo {
        return DeclaredNameInfo("key")
    }

    override fun mapValueName(serializerParent: SafeParentInfo, isListEluded: Boolean): DeclaredNameInfo {
        val childAnnotation = serializerParent.elementUseAnnotations.firstOrNull<XmlChildrenName>()
        val childrenName = childAnnotation?.toQName()
        return DeclaredNameInfo("value", childrenName, childAnnotation?.namespace == UNSET_ANNOTATION_VALUE)
    }

    override fun mapEntryName(serializerParent: SafeParentInfo, isListEluded: Boolean): QName {
        if (isListEluded) { // If we don't have list tags, use the list name, otherwise use the default
            serializerParent.elementUseNameInfo.annotatedName?.let { return it }
        }
        return QName(serializerParent.namespace.namespaceURI, "entry")
    }

    @OptIn(ExperimentalSerializationApi::class)
    override fun isMapValueCollapsed(mapParent: SafeParentInfo, valueDescriptor: XmlDescriptor): Boolean {
        val keyDescriptor = mapParent.elementSerialDescriptor.getElementDescriptor(0)
        val keyUseName = mapKeyName(mapParent)

        val pseudoKeyParent =
            InjectedParentTag(0, XmlTypeDescriptor(keyDescriptor, mapParent.namespace), keyUseName, mapParent.namespace)
        val keyEffectiveOutputKind = effectiveOutputKind(pseudoKeyParent, pseudoKeyParent, true)
        if (!keyEffectiveOutputKind.isTextual) return false

        val keyName = effectiveName(pseudoKeyParent, pseudoKeyParent, keyEffectiveOutputKind, keyUseName)

        (0 until valueDescriptor.elementsCount)
            .map { valueDescriptor.getElementDescriptor(it) }
            .forEach { elem ->
                if (elem.tagName.isEquivalent(keyName)) return false
            }
        return true
    }

    @OptIn(ExperimentalSerializationApi::class)
    @ExperimentalXmlUtilApi
    override fun elementNamespaceDecls(serializerParent: SafeParentInfo): List<Namespace> {
        val annotations = (serializerParent.elementUseAnnotations.asSequence() +
                serializerParent.elementTypeDescriptor.serialDescriptor.annotations)
        return annotations
            .filterIsInstance<XmlNamespaceDeclSpec>()
            .flatMap { decl ->
                decl.namespaces
            }.toList()
    }

    override fun ignoredSerialInfo(message: String) {
        if (pedantic) throw XmlSerialException(message)
    }

    /**
     * Create a builder for this policy. This function allows subclasses to have their own configuration.
     */
    public open fun builder(): Builder = Builder(this)

    /**
     * Create a copy of this configuration with the changes specified through the config parameter.
     */
    @ExperimentalSerializationApi
    public inline fun copy(config: Builder.() -> Unit): DefaultXmlSerializationPolicy {
        return builder().apply(config).build()
    }

    @OptIn(ExperimentalSerializationApi::class)
    @Deprecated("Use the copy that uses the builder to configure changes")
    public fun copy(
        pedantic: Boolean = this.pedantic,
        autoPolymorphic: Boolean = this.autoPolymorphic,
        encodeDefault: XmlEncodeDefault = this.encodeDefault,
        typeDiscriminatorName: QName? = this.typeDiscriminatorName
    ): DefaultXmlSerializationPolicy {
        return copy {
            this.pedantic = pedantic
            this.autoPolymorphic = autoPolymorphic
            this.encodeDefault = encodeDefault
            this.typeDiscriminatorName = typeDiscriminatorName
        }
    }

    @OptIn(ExperimentalSerializationApi::class)
    @Deprecated("Use the copy that uses the builder to configure changes")
    @ExperimentalXmlUtilApi
    public fun copy(
        pedantic: Boolean = this.pedantic,
        autoPolymorphic: Boolean = this.autoPolymorphic,
        encodeDefault: XmlEncodeDefault = this.encodeDefault,
        unknownChildHandler: UnknownChildHandler,
        typeDiscriminatorName: QName? = this.typeDiscriminatorName
    ): DefaultXmlSerializationPolicy {
        return copy {
            this.pedantic = pedantic
            this.autoPolymorphic = autoPolymorphic
            this.encodeDefault = encodeDefault
            this.unknownChildHandler = unknownChildHandler
            this.typeDiscriminatorName = typeDiscriminatorName
        }
    }

    /**
     * A configuration builder for the default serialization policy.
     *
     * @property pedantic Enable some stricter behaviour
     * @property autoPolymorphic Rather than using type wrappers use the tag name to distinguish polymorphic types
     * @property encodeDefault Determine whether defaults need to be encoded
     * @property unknownChildHandler Function called when an unknown child is encountered. By default it throws an
     *   exception, but this function can use its own recovery behaviour
     * @property typeDiscriminatorName When set, use a type discriminator property
     * @property throwOnRepeatedElement When a single-value elemement is repeated in the content, will this throw an
     *   exception or only retain the final value
     * @property verifyElementOrder Verify that element children are in the order required by order annotations (and
     *   fail if not correct). Note that attribute order in XML is arbitrary and not meaningful.
     * @property isStrictAttributeNames Process attribute name reading strictly according to the XML standard, or a
     *   name handling that is a bit more lenient
     * @property isStrictBoolean Parse boolean data according to the requirements of XML, rather than the (very lenient)
     *   toBoolean function from the Kotlin standard library.
     */
    @OptIn(ExperimentalXmlUtilApi::class)
    public open class Builder internal constructor(
        public var pedantic: Boolean = false,
        public var autoPolymorphic: Boolean = false,
        public var encodeDefault: XmlEncodeDefault = XmlEncodeDefault.ANNOTATED,
        public var unknownChildHandler: UnknownChildHandler = XmlConfig.DEFAULT_UNKNOWN_CHILD_HANDLER,
        public var typeDiscriminatorName: QName? = null,
        public var throwOnRepeatedElement: Boolean = false,
        public var verifyElementOrder: Boolean = false,
        public var isStrictAttributeNames: Boolean = false,
        public var isStrictBoolean: Boolean = false,
    ) {
        /**
         * Constructor for default builder. To set any values, use the property setters. The primary constructor
         * is internal as it is not stable as new configuration options are added.
         */
        public constructor() : this(pedantic = false)

        @ExperimentalXmlUtilApi
        public constructor(policy: DefaultXmlSerializationPolicy) : this(
            pedantic = policy.pedantic,
            autoPolymorphic = policy.autoPolymorphic,
            encodeDefault = policy.encodeDefault,
            unknownChildHandler = policy.unknownChildHandler,
            typeDiscriminatorName = policy.typeDiscriminatorName,
            throwOnRepeatedElement = policy.throwOnRepeatedElement,
            verifyElementOrder = policy.verifyElementOrder,
            isStrictAttributeNames = policy.isStrictNames,
        )

        public fun ignoreUnknownChildren() {
            unknownChildHandler = XmlConfig.IGNORING_UNKNOWN_CHILD_HANDLER
        }

        public fun ignoreNamespaces() {
            unknownChildHandler = XmlConfig.IGNORING_UNKNOWN_NAMESPACE_HANDLER
        }

        public fun build(): DefaultXmlSerializationPolicy = DefaultXmlSerializationPolicy(this)
    }
}
