package shz;

import java.io.File;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;

public final class CmdHelp {
    private CmdHelp() {
        throw new IllegalStateException();
    }

    private static final Map<String, Long> PID = new ConcurrentHashMap<>();

    private static final BiConsumer<Process, Long> END = (t, u) -> {
        try {
            if (u == null || u <= 0) t.waitFor();
            else t.waitFor(u, TimeUnit.MILLISECONDS);
        } catch (InterruptedException e) {
            throw PRException.of(e);
        }
    };

    private static final String OK = "ok";

    public static final class ExecuteParam {
        private String[] envp;
        private File file;
        private Consumer<String> consumer;
        private Function<Process, Long> pid;
        private Consumer<Process> end;
        private String path;
        private long timeout;
        private boolean silent = true;
        private String[] args;

        private ExecuteParam() {
        }

        public static ExecuteParam of() {
            return new ExecuteParam();
        }

        public ExecuteParam envp(String[] envp) {
            this.envp = envp;
            return this;
        }

        public ExecuteParam file(File file) {
            this.file = file;
            return this;
        }

        public ExecuteParam consumer(Consumer<String> consumer) {
            this.consumer = consumer;
            return this;
        }

        public ExecuteParam pid(Function<Process, Long> pid) {
            this.pid = pid;
            return this;
        }

        public ExecuteParam end(Consumer<Process> end) {
            this.end = end;
            return this;
        }

        public ExecuteParam path(String path) {
            this.path = path;
            return this;
        }

        public ExecuteParam timeout(long timeout) {
            this.timeout = timeout;
            return this;
        }

        public ExecuteParam silent(boolean silent) {
            this.silent = silent;
            return this;
        }

        public ExecuteParam args(String... args) {
            this.args = args;
            return this;
        }
    }

    public static void execute(String command, ExecuteParam param) {
        try {
            Process process = Runtime.getRuntime().exec("cmd /c " + command, param.envp, param.file == null ? null : param.file.getParentFile());
            if (param.consumer != null) IOHelp.read(IOHelp.getBr(process.getInputStream()), param.consumer);
            if (param.pid != null && param.file != null)
                PID.put(param.file.getAbsolutePath(), param.pid.apply(process));
            if (param.end != null) param.end.accept(process);
        } catch (Throwable t) {
            throw PRException.of(t);
        }
    }

    public static void execute(String command) {
        execute(command, ExecuteParam.of());
    }

    public static Future<Boolean> isStarted(String name, long timeout) {
        return CompletableFuture.supplyAsync(() -> {
            if (Validator.isBlank(name)) return false;
            AtomicBoolean result = new AtomicBoolean();
            try {
                execute("tasklist", ExecuteParam.of().consumer(t -> {
                    if (ToSet.asSet(t.split("\\s+")).contains(name)) result.set(true);
                }).end(t -> END.accept(t, timeout)));
            } catch (Throwable e) {
                throw PRException.of(e);
            }
            return result.get();
        });
    }

    public static Future<Boolean> isStarted(String name) {
        return isStarted(name, 1500);
    }

    public static Future<String> start(ExecuteParam param) {
        if (param.file == null) param.file = FileHelp.findFile(param.path, param.timeout);
        return CompletableFuture.supplyAsync(() -> {
            String validFile = validFile(param.file);
            if (validFile != null) return validFile;
            String name = param.file.getName();
            Future<Boolean> started = isStarted(name);
            try {
                if (started.get()) return "[" + name + "] already running";
            } catch (InterruptedException | ExecutionException e) {
                return e.getMessage();
            }
            String params = name;
            if (Validator.nonBlank(param.args))
                params += Arrays.stream(param.args).collect(Collectors.joining(" ", " ", ""));
            try {
                if (param.silent) execute(params, param);
                else execute("start " + params, param);
            } catch (Throwable t) {
                return t.getMessage();
            }
            return OK;
        });
    }

    private static String validFile(File execute) {
        if (execute == null) return "the program is not exist";
        if (!execute.isFile()) return "[" + execute.getName() + "] is not file";
        if (!execute.canExecute()) return "[" + execute.getName() + "] is can't execute";
        return null;
    }

    public static List<Future<String>> start(ExecuteParam... ps) {
        return ToList.explicitCollect(Arrays.stream(ps).map(CmdHelp::start), ps.length);
    }

    public static Future<String> close(ExecuteParam param) {
        if (param.file == null) param.file = FileHelp.findFile(param.path, param.timeout);
        return CompletableFuture.supplyAsync(() -> {
            String validFile = validFile(param.file);
            if (validFile != null) return validFile;
            String name = param.file.getName();
            Future<Boolean> started = isStarted(name);
            try {
                if (!started.get()) return "[" + name + "] not in running";
            } catch (InterruptedException | ExecutionException e) {
                return e.getMessage();
            }
            String path = param.file.getAbsolutePath();
            boolean flag = PID.containsKey(path);
            String command = flag ? "taskkill /pid " + PID.get(path) + " /f /t " : "taskkill /f /im " + name;
            try {
                execute(command, param);
            } catch (Throwable t) {
                return t.getMessage();
            }
            if (flag) PID.remove(path);
            return OK;
        });
    }

    public static Future<String> close(String path) {
        return close(ExecuteParam.of().path(path).end(t -> {
            END.accept(t, null);
            t.destroyForcibly();
        }));
    }

    public static List<Future<String>> close(ExecuteParam... ps) {
        return ToList.explicitCollect(Arrays.stream(ps).map(CmdHelp::close), ps.length);
    }

    public static List<Future<String>> close(String... ps) {
        return ToList.explicitCollect(Arrays.stream(ps).map(CmdHelp::close), ps.length);
    }

    public static Future<String> restart(ExecuteParam param) {
        if (param.file == null) param.file = FileHelp.findFile(param.path, param.timeout);
        return CompletableFuture.supplyAsync(() -> {
            try {
                Consumer<Process> end = param.end;
                String result = close(param.end(t -> {
                    END.accept(t, null);
                    t.destroyForcibly();
                    if (t.isAlive()) throw new RuntimeException("the program is alive");
                })).get();
                if (OK.equals(result)) return restart(param.end(end)).get();
                return result;
            } catch (InterruptedException | ExecutionException e) {
                return e.getMessage();
            }
        });
    }

    public static List<Future<String>> restart(ExecuteParam... ps) {
        return ToList.explicitCollect(Arrays.stream(ps).map(CmdHelp::restart), ps.length);
    }

    /**
     * 显示dns缓存
     */
    public static void displaydns() {
        execute("ipconfig /displaydns");
    }

    /**
     * 刷新DNS记录
     */
    public static void flushdns() {
        execute("ipconfig /flushdns");
    }

    /**
     * 重新从DHCP服务器获得IP
     */
    public static void renew() {
        execute("ipconfig /renew");
    }

    /**
     * maven 安装依赖
     */
    public static void mvnInstall(String groupId, String artifactId, String version, String packaging, String file) {
        execute("mvn install:install-file" +
                " -DgroupId=" + groupId +
                " -DartifactId=" + artifactId +
                " -Dversion=" + version +
                " -Dpackaging=" + packaging +
                " -Dfile=" + file);
    }

    /**
     * 定时关机
     */
    public static void shutdown(int seconds) {
        execute("shutdown -s -t " + seconds);
    }

    /**
     * 取消定时关机
     */
    public static void cancelShutdown() {
        execute("shutdown -a");
    }
}
