package shz;

import shz.model.Sphere;

import java.awt.geom.Point2D;
import java.math.BigDecimal;

/**
 * 坐标系转换工具
 */
public class CoordinateHelp {
    private CoordinateHelp() {
        throw new IllegalStateException();
    }

    private static final double ee = 0.00669342162296594323D;

    /**
     * WGS84坐标系：地心坐标系，即GPS原始坐标体系，国际上通用的坐标系。
     * 设备一般包含GPS芯片或者北斗芯片获取的经纬度为WGS84地理坐标系,
     * 谷歌地图采用的是WGS84地理坐标系（中国范围除外）。
     * <p>
     * GCJ02坐标系：即火星坐标系，是由中国国家测绘局制订的地理信息系统的坐标系统。由WGS84坐标系经加密后的坐标系。
     * 高德、腾讯、Google中国地图使用
     */
    public static Point2D.Double gps84ToGcj02(double lon, double lat) {
        if (Sphere.outOfChina(lon, lat)) return new Point2D.Double(lon, lat);
        double radLat = lat * Sphere.RADIAN;
        double magic = Math.sin(radLat);
        magic = 1 - ee * magic * magic;
        double sqrtMagic = Math.sqrt(magic);
        Point2D.Double gcj02 = transform(lon - 105.0, lat - 35.0);
        gcj02.x = setScale(lon + gcj02.x * 180.0 / (Sphere.EARTH_RADIUS / sqrtMagic * Math.cos(radLat) * Math.PI));
        gcj02.y = setScale(lat + gcj02.y * 180.0 / (Sphere.EARTH_RADIUS * (1 - ee) / (magic * sqrtMagic) * Math.PI));
        return gcj02;
    }

    private static Point2D.Double transform(double lon, double lat) {
        Point2D.Double point = new Point2D.Double();
        double sqrt = Math.sqrt(Math.abs(lon));
        point.x = 300.0 + lon + 2.0 * lat + 0.1 * lon * lon + 0.1 * lon * lat + 0.1 * sqrt;
        point.y = -100.0 + 2.0 * lon + 3.0 * lat + 0.2 * lat * lat + 0.1 * lon * lat + 0.2 * sqrt;

        double v = (20.0 * Math.sin(6.0 * lon * Math.PI) + 20.0 * Math.sin(2.0 * lon * Math.PI)) * 2.0 / 3.0;
        point.x += v;
        point.y += v;

        point.x += (20.0 * Math.sin(lon * Math.PI) + 40.0 * Math.sin(lon / 3.0 * Math.PI)) * 2.0 / 3.0;
        point.x += (150.0 * Math.sin(lon / 12.0 * Math.PI) + 300.0 * Math.sin(lon / 30.0 * Math.PI)) * 2.0 / 3.0;

        point.y += (20.0 * Math.sin(lat * Math.PI) + 40.0 * Math.sin(lat / 3.0 * Math.PI)) * 2.0 / 3.0;
        point.y += (160.0 * Math.sin(lat / 12.0 * Math.PI) + 320 * Math.sin(lat * Math.PI / 30.0)) * 2.0 / 3.0;
        return point;
    }

    private static double setScale(double val) {
        BigDecimal result = new BigDecimal(Double.toString(val));
        if (result.scale() > 6) result = result.setScale(6, BigDecimal.ROUND_HALF_UP);
        return result.doubleValue();
    }

    public static Point2D.Double gcj02ToGps84(double lon, double lat) {
        Point2D.Double gcj02 = gps84ToGcj02(lon, lat);
        Point2D.Double gps84 = new Point2D.Double();
        gps84.x = setScale(lon * 2 - gcj02.x);
        gps84.y = setScale(lat * 2 - gcj02.y);
        return gps84;
    }

    private static final double x_pi = 3000.0 * Sphere.RADIAN;

    /**
     * BD09坐标系：即百度坐标系，GCJ02坐标系经加密后的坐标系。
     */
    public static Point2D.Double gcj02ToBd09(double lon, double lat) {
        double z = Math.sqrt(lon * lon + lat * lat) + 0.00002 * Math.sin(lat * x_pi);
        double theta = Math.atan2(lat, lon) + 0.000003 * Math.cos(lon * x_pi);
        Point2D.Double bd09 = new Point2D.Double();
        bd09.x = setScale(z * Math.cos(theta) + 0.0065);
        bd09.y = setScale(z * Math.sin(theta) + 0.006);
        return bd09;
    }

    public static Point2D.Double bd09ToGcj02(double lon, double lat) {
        double x = lon - 0.0065, y = lat - 0.006;
        double z = Math.sqrt(x * x + y * y) - 0.00002 * Math.sin(y * x_pi);
        double theta = Math.atan2(y, x) - 0.000003 * Math.cos(x * x_pi);
        Point2D.Double gcj02 = new Point2D.Double();
        gcj02.x = setScale(z * Math.cos(theta));
        gcj02.y = setScale(z * Math.sin(theta));
        return gcj02;
    }
}
