package shz;

import shz.msg.ServerFailureMsg;

import java.io.*;
import java.net.JarURLConnection;
import java.net.URL;
import java.net.URLConnection;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.concurrent.*;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.jar.JarFile;
import java.util.regex.Matcher;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@SuppressWarnings({"restriction"})
public final class FileHelp {
    private FileHelp() {
        throw new IllegalStateException();
    }

    public static File fromUrl(URL url, Charset charset) {
        return new File(Coder.urlDecode(url.getFile(), charset));
    }

    public static File fromUrl(URL url) {
        return fromUrl(url, StandardCharsets.UTF_8);
    }

    public static File fromCls(Class<?> cls, Charset charset) {
        return fromUrl(cls.getProtectionDomain().getCodeSource().getLocation(), charset);
    }

    public static File fromCls(Class<?> cls) {
        return fromCls(cls, StandardCharsets.UTF_8);
    }

    /**
     * 搜索单个文件,支持模糊匹配
     */
    public static File findFile(String path, boolean tryFind, List<String> includes, Set<String> excludes, Executor executor, long timeout) {
        if (Validator.isBlank(path)) return null;
        String path_ = Coder.urlDecode(path, StandardCharsets.UTF_8);
        File file = new File(path_);
        if (file.exists()) return file;
        if (!tryFind) return null;
        AtomicBoolean stop = new AtomicBoolean();
        if (executor == null || timeout <= 0) return tryFind(path_, includes, excludes, stop);
        try {
            return CompletableFuture.supplyAsync(() -> tryFind(path_, includes, excludes, stop), executor).get(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw PRException.of(e);
        } finally {
            stop.set(true);
        }
    }

    private static File tryFind(String path, List<String> includes, Set<String> excludes, AtomicBoolean stop) {
        char[] array = format(path);
        File include = getInclude(array);
        List<File> includes_ = include == null ? getIncludes(includes) : Collections.singletonList(include);
        if (Validator.isEmpty(includes_)) return null;
        Set<char[]> excludes_ = getExcludes(excludes);
        if (excludes_ != null) {
            includes_ = ToList.explicitCollect(includes_.stream().filter(f -> !contains(f, excludes_, false)), includes_.size());
            if (Validator.isEmpty(includes_)) return null;
        }
        File result = null;
        for (File file : includes_) {
            if (stop.get()) break;
            result = tryFind(file, array, excludes_, stop);
            if (result != null) break;
        }
        return result;
    }

    private static char[] format(String path) {
        StringBuilder sb = new StringBuilder(path.length());
        char prev1 = ' ';
        char prev2 = ' ';
        for (int i = 0; i < path.length(); ++i) {
            char c = path.charAt(i);
            if (c == '/' || c == '\\') {
                if (prev1 == '/') continue;
                sb.append('/');
                prev2 = prev1;
                prev1 = '/';
            } else if (c == '*') {
                if (prev1 == '*' && prev2 == '*') continue;
                sb.append('*');
                prev2 = prev1;
                prev1 = '*';
            } else {
                sb.append(c);
                prev2 = prev1;
                prev1 = c;
            }
        }
        path = sb.toString().replaceAll("^(\\*\\*/){2,}", "**/").replaceAll("(/\\*\\*){2,}/", "/**/");
        if (path.charAt(path.length() - 1) == '/') path = path.substring(0, path.length() - 1);
        char[] array = path.toCharArray();
        int idx = Help.indexOf(':', array);
        if (idx <= 0 || !isWindows()) return array;
        for (int i = 0; i < idx; ++i) if (!Character.isUpperCase(array[i])) array[i] = Character.toUpperCase(array[i]);
        return array;
    }

    private static boolean isWindows() {
        return System.getProperty("os.name").toLowerCase().startsWith("win");
    }

    private static File getInclude(char[] array) {
        boolean win;
        if ((win = isWindows()) && (array[0] == '/' || Help.indexOf(':', array) == -1)) return null;
        if (!win && array[0] != '/') return null;
        int end = -1;
        for (int i = 0; i < array.length; ++i) {
            if (array[i] == '*') break;
            if (array[i] == '/') end = i;
        }
        if (end == -1) return null;
        File file = new File(new String(array, 0, end + 1));
        return file.exists() ? file : null;
    }

    private static List<File> getIncludes(List<String> includes) {
        List<File> includes_;
        if (Validator.nonBlank(includes)) {
            includes_ = ToList.explicitCollect(includes.stream().filter(Validator::nonBlank)
                    .map(e -> Coder.urlDecode(e, StandardCharsets.UTF_8)).map(File::new).filter(File::exists), includes.size());
            if (Validator.nonEmpty(includes_)) return includes_;
        }
        if (isWindows())
            return ToList.explicitCollect(Stream.of(
                    System.getProperty("user.dir"),
                    System.getProperty("user.home"),
                    "C:", "D:", "E:", "F:", "G:", "H:"
            ).map(File::new), 8);
        return ToList.explicitCollect(Stream.of(
                System.getProperty("user.dir"),
                System.getProperty("user.home"),
                System.getProperty("file.separator")
        ).map(File::new), 3);
    }

    private static Set<char[]> getExcludes(Set<String> excludes) {
        if (Validator.isBlank(excludes)) return null;
        return ToSet.explicitCollect(excludes.parallelStream().filter(Validator::nonBlank).map(e -> Coder.urlDecode(e, StandardCharsets.UTF_8)).map(FileHelp::format), excludes.size());
    }

    private static File tryFind(File file, char[] path, Set<char[]> excludes, AtomicBoolean stop) {
        if (stop.get()) return null;
        if (equals(format(file.getAbsolutePath()), path)) return file;
        if (file.isFile()) return null;
        File[] listFiles = file.listFiles(f -> f.isFile() && !contains(f, excludes, false));
        if (Validator.nonEmpty(listFiles)) {
            File file_ = Arrays.stream(listFiles).filter(f -> equals(format(f.getAbsolutePath()), path)).findAny().orElse(null);
            if (file_ != null) return file_;
        }
        listFiles = file.listFiles(f -> f.isDirectory() && !contains(f, excludes, false));
        if (Validator.isEmpty(listFiles)) return null;
        return Arrays.stream(listFiles).map(f -> tryFind(f, path, excludes, stop)).filter(Objects::nonNull).findAny().orElse(null);
    }

    private static final int offset = 10;
    //0x3ff
    private static final int mask = ~(-1 << offset);

    private static boolean equals(char[] file, char[] path) {
        int checkTail = checkTail(file, path);
        if (checkTail == N) return false;
        int plen = spLen(path);
        if (plen == 1) return checkTail == S || spLen(file) == 1;
        int checkHead = checkHead(file, path);
        if (checkHead == N) return false;
        if (plen == 2) return checkTail == S || checkHead == S || spLen(file) == 2;
        int flen = spLen(file);
        int[] fsps = sps(file, flen);
        int[] psps = sps(path, plen);
        int[][] dp = new int[flen][plen];
        dp[0][0] = checkHead;
        dp[0][0] |= OK;
        for (int i = 1; i < flen; ++i) {
            dp[i][0] = match(file, fsps[i] >> offset, fsps[i] & mask, path, psps[0] >> offset, psps[0] & mask);
            if (dp[i][0] == N) break;
            dp[i][0] |= OK;
        }
        for (int j = 1; j < plen; ++j) {
            dp[0][j] = match(file, fsps[0] >> offset, fsps[0] & mask, path, psps[j] >> offset, psps[j] & mask);
            if (dp[0][j] == N) break;
            dp[0][j] |= OK;
        }
        for (int i = 1; i < flen; ++i) {
            for (int j = 1; j < plen; ++j) {
                dp[i][j] = match(file, fsps[i] >> offset, fsps[i] & mask, path, psps[j] >> offset, psps[j] & mask);
                if (dp[i][j] != N) {
                    dp[i][j] |= dp[i - 1][j - 1];
                    if ((dp[i][j] & S) != 0 || (dp[i][j - 1] & S) != 0) {
                        dp[i][j] |= dp[i][j - 1];
                        dp[i][j] |= dp[i - 1][j];
                    }
                }
            }
        }
        return (dp[flen - 1][plen - 1] & OK) != 0;
    }

    private static int checkTail(char[] file, char[] path) {
        if (path.length < 2) return N;
        int PL = lastSp(path);
        PL = PL == -1 ? 0 : PL + 1;
        return match(file, lastSp(file) + 1, file.length, path, PL, path.length);
    }

    private static int lastSp(char[] array) {
        for (int i = array.length - 1; i >= 0; --i) if (array[i] == '/') return i;
        return -1;
    }

    private static final int N = 1;
    private static final int M = 1 << 1;
    private static final int S = 1 << 2;
    private static final int OK = 1 << 3;

    private static int match(char[] file, int FL, int FR, char[] path, int PL, int PR) {
        if (file[FR - 1] != path[PR - 1] && path[PR - 1] != '*') return N;
        try {
            if (file[FL] != path[PL] && path[PL] != '*') return N;
        } catch (Throwable t) {
            System.out.println(new String(file));
            return N;
        }

        if (PR - PL == 1) return path[PL] == '*' ? M : FR - FL > 1 || file[FL] != path[PL] ? N : M;
        if (PR - PL == 2) {
            if (path[PL] == '*' && path[PL + 1] == '*') return S;
            if (FR - FL != 2) return N;
            if (path[PL + 1] == '*') return file[FL] == path[PL] ? M : N;
            if (path[PL] == '*') return file[FL + 1] == path[PL + 1] ? M : N;
            return file[FL] == path[PL] && file[FL + 1] == path[PL + 1] ? M : N;
        }
        boolean[][] dp = new boolean[FR - FL][PR - PL];
        dp[0][0] = true;
        for (int i = 1; i < FR - FL; ++i) {
            for (int j = 1; j < PR - PL; ++j) {
                if (file[i + FL] == path[j + PL] || path[j + PL] == '*') {
                    dp[i][j] |= dp[i - 1][j - 1];
                    if (path[j - 1 + PL] == '*' && (path[j + PL] == '*' || (j >= 2 && path[j - 2 + PL] == '*'))) {
                        dp[i][j] |= dp[i][j - 1];
                        dp[i][j] |= dp[i - 1][j];
                    }
                }
            }
        }
        return dp[FR - FL - 1][PR - PL - 1] ? M : N;
    }

    private static int spLen(char[] array) {
        int len = 1;
        int idx = 0;
        while ((idx = nextSp(array, idx)) != -1) ++len;
        return len;
    }

    private static int nextSp(char[] array, int idx) {
        for (int i = idx + 1; i < array.length; ++i) if (array[i] == '/') return i;
        return -1;
    }

    private static int checkHead(char[] file, char[] path) {
        int[] fsps = sps(file, 1);
        int[] psps = sps(path, 1);
        return match(file, fsps[0] >> offset, fsps[0] & mask, path, psps[0] >> offset, psps[0] & mask);
    }

    private static int[] sps(char[] array, int len) {
        int[] sps = new int[len];
        int L = array[0] == '/' ? 1 : 0;
        int R = nextSp(array, L);
        R = R == -1 ? array.length : R;
        sps[0] = (L << offset) + R;
        for (int i = 1; i < len; ++i) {
            sps[i] = (R + 1) << offset;
            R = nextSp(array, R + 1);
            R = R == -1 ? array.length : R;
            sps[i] += R;
        }
        return sps;
    }

    private static boolean contains(File file, Set<char[]> paths, boolean remove) {
        if (paths == null) return false;
        char[] array = format(file.getAbsolutePath());
        for (Iterator<char[]> it = paths.iterator(); it.hasNext(); ) {
            if (equals(array, it.next())) {
                if (remove) it.remove();
                return true;
            }
        }
        return false;
    }

    public static File findFile(String path, long timeout) {
        return findFile(path, true, null, null, timeout <= 0L ? null : ForkJoinPool.commonPool(), timeout);
    }

    public static File findFile(String path) {
        return findFile(path, true, null, null, null, 0L);
    }

    /**
     * 搜索多个文件,支持模糊匹配
     */
    public static Set<File> findFiles(Set<String> paths, boolean tryFind, boolean regex, List<String> includes, Set<String> excludes, Executor executor, long timeout) {
        if (Validator.isBlank(paths)) return Collections.emptySet();
        Set<String> paths_ = ToSet.explicitCollect(paths.stream().filter(Validator::nonBlank).map(e -> Coder.urlDecode(e, StandardCharsets.UTF_8)), paths.size());
        Stream<File> filter = paths_.stream().map(File::new).filter(File::exists);
        Set<File> result = filter.collect(Collectors.toSet());
        if (!tryFind || (!regex && result.size() == paths_.size())) return result;
        if (!result.isEmpty())
            paths_.removeAll(ToSet.explicitCollect(result.stream().map(File::getAbsolutePath), result.size()));
        AtomicBoolean stop = new AtomicBoolean();
        if (executor == null || timeout <= 0) tryFinds(paths_, regex, includes, excludes, result, stop);
        else try {
            CompletableFuture.runAsync(() -> tryFinds(paths_, regex, includes, excludes, result, stop), executor).get(timeout, TimeUnit.MILLISECONDS);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw PRException.of(e);
        } finally {
            stop.set(true);
        }
        return result.isEmpty() ? Collections.emptySet() : result;
    }

    private static void tryFinds(Set<String> paths, boolean regex, List<String> includes, Set<String> excludes, Set<File> result, AtomicBoolean stop) {
        Set<char[]> set = ToSet.explicitCollect(paths.stream().map(FileHelp::format), paths.size());
        List<File> includes_ = ToList.explicitCollect(set.stream().map(FileHelp::getInclude).filter(Objects::nonNull), set.size());
        if (includes_.size() != set.size()) {
            if (includes_.isEmpty()) includes_ = getIncludes(includes);
            else includes_.addAll(getIncludes(includes));
        }
        if (includes_.isEmpty()) return;
        Set<char[]> excludes_ = getExcludes(excludes);
        if (excludes_ != null) {
            includes_ = ToList.collect(includes_.stream().filter(f -> !contains(f, excludes_, false)));
            if (includes_.isEmpty()) return;
        }
        for (File file : includes_) {
            if (stop.get()) break;
            tryFinds(file, set, regex, excludes_, result, stop);
            if (!regex && set.isEmpty()) break;
        }
    }

    private static void tryFinds(File file, Set<char[]> paths, boolean regex, Set<char[]> excludes, Set<File> result, AtomicBoolean stop) {
        if (stop.get() || (!regex && paths.isEmpty())) return;
        File[] listFiles = file.listFiles(f -> f.isFile() && !contains(f, excludes, false));
        if (Validator.nonEmpty(listFiles))
            Arrays.stream(listFiles).filter(f -> contains(f, paths, !regex)).forEach(result::add);
        listFiles = file.listFiles(f -> f.isDirectory() && !contains(f, excludes, false));
        if (Validator.isEmpty(listFiles)) return;
        Arrays.stream(listFiles).forEach(f -> {
            if (contains(f, paths, !regex)) result.add(f);
            else tryFinds(f, paths, regex, excludes, result, stop);
        });
    }

    public static Set<File> findFiles(Set<String> paths, long timeout) {
        return findFiles(paths, true, false, null, null, timeout <= 0L ? null : ForkJoinPool.commonPool(), timeout);
    }

    public static Set<File> findFiles(Set<String> paths) {
        return findFiles(paths, true, false, null, null, null, 0L);
    }

    public static String getFileType(String hexStr) {
        switch (hexStr) {
            case "FFD8FF":
                return "jpg";
            case "89504E":
                return "png";
            case "474946":
                return "jif";
            case "49492A":
                return "tif";
            case "424D":
                return "bmp";
            case "414331":
                return "dwg";
            case "384250":
                return "psd";
            case "7B5C72":
                return "rtf";
            case "3C3F78":
                return "xml";
            case "68746D":
                return "html";
            case "44656C":
                return "eml";
            case "CFAD12":
                return "dbx";
            case "214244":
                return "pst";
            case "D0CF11":
                return "xls doc";
            case "537461":
                return "mdb";
            case "FF5750":
                return "wpd";
            case "252150":
                return "eps ps";
            case "255044":
                return "pdf";
            case "AC9EBD":
                return "qdf";
            case "E38285":
                return "pwl";
            case "504B03":
                return "zip";
            case "526172":
                return "rar";
            case "574156":
                return "wav";
            case "415649":
                return "avi";
            case "2E7261":
                return "ram";
            case "2E524D":
                return "rm";
            case "000001":
                return "mpg";
            case "6D6F6F":
                return "mov";
            case "3026B2":
                return "asf";
            case "4D5468":
                return "mid";
            case "706163":
                return "java";
            case "CAFEBA":
                return "class";
            default:
                return "txt";
        }
    }

    public static String getFileType(byte[] bytes) {
        if (Validator.isEmpty(bytes)) return null;
        int len = Math.min(bytes.length, 3);
        StringBuilder sb = new StringBuilder(6);
        for (int i = 0; i < len; ++i) {
            String hex = Integer.toHexString(bytes[i] & 0xff);
            if (hex.length() < 2) sb.append(0);
            sb.append(hex);
        }
        return getFileType(sb.toString().toUpperCase());
    }

    public static String getFileType(InputStream is) {
        Validator.requireNon(!is.markSupported());
        is.mark(3);
        byte[] bytes = new byte[3];
        try {
            int len = is.read(bytes, 0, 3);
            if (len < 3) bytes = Arrays.copyOf(bytes, len);
            is.reset();
        } catch (IOException e) {
            throw PRException.of(e);
        }
        return getFileType(bytes);
    }

    public static Set<File> findFilesFromURL(URL url, boolean tryFind, boolean regex, Set<String> paths, long timeout) {
        url = url == null ? Thread.currentThread().getContextClassLoader().getResource("") : url;
        Objects.requireNonNull(url);
        String protocol = url.getProtocol();
        try {
            URLConnection conn = url.openConnection();
            if ("jar".equals(protocol)) {
                JarFile jarFile = ((JarURLConnection) conn).getJarFile();
                return findFiles(ToSet.collect(paths.stream().map(e -> "**/" + e)), tryFind, regex,
                        Collections.singletonList(ZipHelp.getFileFromJarFile(jarFile, IOHelp.DEFAULT_DATA_SIZE, paths)),
                        null, timeout <= 0L ? null : ForkJoinPool.commonPool(), timeout);
            } else {
                File file;
                if ("file".equals(protocol)) file = fromUrl(conn.getURL());
                else {
                    InputStream is = conn.getInputStream();
                    file = fromUrl(url);
                    IOHelp.read(is, IOHelp.getBos(file));
                }
                if (file.isDirectory())
                    return findFiles(paths, tryFind, regex, Collections.singletonList(file.getAbsolutePath()), null,
                            timeout <= 0L ? null : ForkJoinPool.commonPool(), timeout);
                if (paths.size() > 1) return null;
                if (equals(file, paths.iterator().next())) return Collections.singleton(file);
            }
        } catch (IOException e) {
            throw PRException.of(e);
        }
        return Collections.emptySet();
    }

    public static Set<File> findFilesFromURL(Set<String> paths) {
        return findFilesFromURL(null, true, false, paths, 0L);
    }

    public static boolean equals(File file, String path) {
        return equals(format(file.getAbsolutePath()), format(path));
    }

    public static void readBytes(File file, long offset, long size, int dataSize, Collection<byte[]> bytes) {
        IOHelp.read(IOHelp.getBis(file), offset, size, dataSize, bytes, null);
    }

    public static void readBytes(File file, Collection<byte[]> bytes) {
        readBytes(file, 0L, 0L, IOHelp.DEFAULT_DATA_SIZE, bytes);
    }

    public static void readBytes(File file, long offset, long size, int dataSize, byte[] bytes, int dataOffset) {
        IOHelp.read(IOHelp.getBis(file), offset, size, dataSize, bytes, dataOffset, null);
    }

    public static void readBytes(File file, byte[] bytes) {
        readBytes(file, 0L, 0L, IOHelp.DEFAULT_DATA_SIZE, bytes, 0);
    }

    public static byte[] readBytes(File file, long offset, long size, int dataSize) {
        return IOHelp.read(IOHelp.getBis(file), offset, size, dataSize, null);
    }

    public static byte[] readBytes(File file) {
        return readBytes(file, 0L, 0L, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static <T> T readObject(File file, long offset, long size, int dataSize) {
        return Serializer.deserialize(IOHelp.read(IOHelp.getBis(file), offset, size, dataSize, null));
    }

    public static <T> T readObject(File file) {
        return readObject(file, 0L, 0L, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static void readString(File file, Consumer<String> consumer) {
        IOHelp.read(IOHelp.getBr(file), consumer);
    }

    public static void readString(File file, Collection<String> strings) {
        IOHelp.read(IOHelp.getBr(file), (Consumer<String>) strings::add);
    }

    public static String readString(File file) {
        StringWriter sw = new StringWriter();
        IOHelp.read(IOHelp.getBr(file), sw);
        return sw.toString();
    }

    public static void writeBytes(File file, boolean append, Collection<byte[]> bytes) {
        IOHelp.write(IOHelp.getBos(checkWriteFile(file), append, IOHelp.DEFAULT_BUFFER_SIZE), bytes);
    }

    public static File checkWriteFile(File file) {
        ServerFailureMsg.requireNon(file == null, "写出文件为空");
        File parentFile;
        ServerFailureMsg.requireNon((parentFile = file.getParentFile()) == null || !parentFile.mkdirs() && !parentFile.exists(), "写出文件父文件不存在");
        ServerFailureMsg.requireNon(!parentFile.canWrite(), "写出文件父文件不可写");
        return file;
    }

    public static void writeBytes(File file, Collection<byte[]> bytes) {
        writeBytes(file, false, bytes);
    }

    public static void writeBytes(File file, boolean append, byte[] bytes) {
        IOHelp.write(IOHelp.getBos(checkWriteFile(file), append, IOHelp.DEFAULT_BUFFER_SIZE), bytes);
    }

    public static void writeBytes(File file, byte[] bytes) {
        writeBytes(file, false, bytes);
    }

    public static void writeChars(File file, boolean append, char[] chars) {
        IOHelp.write(IOHelp.getBw(checkWriteFile(file), append, IOHelp.DEFAULT_BUFFER_SIZE), chars);
    }

    public static void writeChars(File file, char[] chars) {
        writeChars(file, false, chars);
    }

    public static void writeString(File file, boolean append, Collection<String> strings) {
        IOHelp.write(IOHelp.getBw(checkWriteFile(file), append, IOHelp.DEFAULT_BUFFER_SIZE), strings);
    }

    public static void writeString(File file, Collection<String> strings) {
        writeString(file, false, strings);
    }

    public static void writeObject(File file, boolean append, Object data) {
        IOHelp.write(new DataOutputStream(IOHelp.getBos(checkWriteFile(file), append, IOHelp.DEFAULT_BUFFER_SIZE)), Serializer.serialize(data));
    }

    public static void writeObject(File file, Object data) {
        writeObject(file, false, data);
    }

    public static void copy(File src, File des, int dataSize) {
        IOHelp.read(IOHelp.getBis(src, IOHelp.DEFAULT_BUFFER_SIZE), IOHelp.getBos(des, false, IOHelp.DEFAULT_BUFFER_SIZE), dataSize, null, null);
    }

    public static void copy(File src, File des) {
        copy(src, des, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static void copyText(File src, File des) {
        IOHelp.read(IOHelp.getBr(src, IOHelp.DEFAULT_BUFFER_SIZE), IOHelp.getBw(des, false, IOHelp.DEFAULT_BUFFER_SIZE));
    }

    public static void copy(Set<File> srcs, File des, int dataSize) {
        File desFile = checkCopyFile(des);
        srcs.parallelStream().filter(Objects::nonNull).filter(f -> f.exists() && f.isFile() && f.canRead()).forEach(f -> {
            File new_folder = new File(desFile, Help.uuid());
            if (new_folder.mkdirs())
                IOHelp.read(IOHelp.getBis(f, IOHelp.DEFAULT_BUFFER_SIZE), IOHelp.getBos(new File(new_folder, f.getName()), false, IOHelp.DEFAULT_BUFFER_SIZE), dataSize, null, null);
        });
    }

    public static File checkCopyFile(File file) {
        ServerFailureMsg.requireNon(file == null, "拷贝输出文件为空");
        ServerFailureMsg.requireNon(!file.mkdirs() && !file.exists(), "拷贝输出文件不存在");
        ServerFailureMsg.requireNon(!file.canWrite(), "拷贝输出文件不可写");
        ServerFailureMsg.requireNon(!file.isDirectory(), "拷贝输出文件不是目录");
        return file;
    }

    public static void copy(Set<File> srcs, File des) {
        copy(srcs, des, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static void copyText(Set<File> srcs, File des) {
        File desFile = checkCopyFile(des);
        srcs.parallelStream().filter(Objects::nonNull).filter(f -> f.exists() && f.isFile() && f.canRead()).forEach(
                f -> IOHelp.read(IOHelp.getBr(f, IOHelp.DEFAULT_BUFFER_SIZE), IOHelp.getBw(new File(desFile, Help.uuid() + "-copy-" + f.getName()), false, IOHelp.DEFAULT_BUFFER_SIZE))
        );
    }

    public static void copyFolder(File src, File des, Set<String> excludes, int dataSize) {
        copyFolder0(src, checkCopyFile(des), getExcludes(excludes), dataSize);
    }

    private static void copyFolder0(File oldFolder, File newFolder, Set<char[]> excludes, int dataSize) {
        Optional.ofNullable(oldFolder.listFiles(f -> !contains(f, excludes, false))).ifPresent(fs -> {
            if (newFolder.mkdirs() || newFolder.exists())
                Arrays.stream(fs).parallel().forEach(f -> {
                    if (f.isFile())
                        IOHelp.read(IOHelp.getBis(f, IOHelp.DEFAULT_BUFFER_SIZE), IOHelp.getBos(new File(newFolder, f.getName()), false, IOHelp.DEFAULT_BUFFER_SIZE), dataSize, null, null);
                    else copyFolder0(f, new File(newFolder, f.getName()), excludes, dataSize);
                });
        });
    }

    public static void copyFolder(File src, File des) {
        copyFolder(src, des, null, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static void copyFolder(File file) {
        copyFolder0(file, getDcf(file), null, IOHelp.DEFAULT_DATA_SIZE);
    }

    private static File getDcf(File file) {
        String name = file.getName();
        int idx = name.indexOf(".");
        if (idx != -1) name = name.substring(0, idx);
        File des = new File(file.getParentFile(), name + "-copy");
        if (des.exists()) deleteFile(des, null);
        return des.mkdirs() ? des : null;
    }

    public static void copyFolder(Set<File> srcs, File des, Set<String> excludes, int dataSize) {
        File desFile = checkCopyFile(des);
        Set<char[]> excludes_ = getExcludes(excludes);
        srcs.parallelStream().filter(Objects::nonNull).filter(f -> f.exists() && f.isDirectory() && f.canRead())
                .forEach(f -> copyFolder0(f, new File(desFile, Help.uuid() + "-copy-" + f.getName()), excludes_, dataSize));
    }

    public static void copyFolder(Set<File> srcs, File des) {
        copyFolder(srcs, des, null, IOHelp.DEFAULT_DATA_SIZE);
    }

    public static void copyFolder(Set<File> files) {
        copyFolder(files, getDcf(files), null, IOHelp.DEFAULT_DATA_SIZE);
    }

    private static File getDcf(Set<File> files) {
        return new File(System.getProperty("user.dir"), Coder.md5(Arrays.toString(files.toArray()).getBytes()));
    }

    public static void deleteFile(File file, Set<String> excludes) {
        if (file == null) return;
        Set<char[]> excludes_;
        if (contains(file, excludes_ = getExcludes(excludes), false)) return;
        deleteFile0(file, excludes_);
    }

    private static void deleteFile0(File file, Set<char[]> excludes) {
        if (file.isFile()) ServerFailureMsg.requireNon(!file.delete(), "删除文件%s异常", file.getAbsolutePath());
        else Optional.ofNullable(file.listFiles(f -> !contains(f, excludes, false))).ifPresent(
                fs -> Arrays.stream(fs).parallel().forEach(f -> deleteFile0(f, excludes))
        );
    }

    public static void deleteFile(File file) {
        deleteFile(file, null);
    }

    public static void deleteFile(String path, Set<String> excludes) {
        deleteFile(findFile(path), excludes);
    }

    public static void deleteFile(String path) {
        deleteFile(findFile(path), null);
    }

    public static void deleteFile(Set<String> paths, Set<String> excludes) {
        Set<char[]> excludes_ = getExcludes(excludes);
        paths.parallelStream().map(File::new).filter(f -> f.exists() && f.canExecute()).forEach(f -> deleteFile0(f, excludes_));
    }

    public static void deleteFile(Set<String> paths) {
        deleteFile(paths, null);
    }

    public static final String F_R = "r";
    public static final String F_RW = "rw";
    public static final String F_RWS = "rws";
    public static final String F_RWD = "rwd";

    public static RandomAccessFile getRandomAccessFile(File file, String mode) {
        try {
            return new RandomAccessFile(file, mode);
        } catch (FileNotFoundException e) {
            throw PRException.of(e);
        }
    }

    public static void consumer(File file, Consumer<File> consumer) {
        if (file.isFile()) {
            consumer.accept(file);
            return;
        }
        Optional.ofNullable(file.listFiles()).ifPresent(fs -> Arrays.stream(fs).parallel().forEach(f -> {
            if (f.isFile()) consumer.accept(f);
            else consumer(f, consumer);
        }));
    }

    public static void consumer(File src, File des, BiConsumer<File, File> consumer) {
        consumer0(src, checkCopyFile(des), consumer);
    }

    private static void consumer0(File src, File des, BiConsumer<File, File> consumer) {
        if (src.isFile()) {
            consumer.accept(src, des);
            return;
        }
        Optional.ofNullable(src.listFiles()).ifPresent(fs -> Arrays.stream(fs).parallel().forEach(f -> {
            if (f.isFile()) consumer.accept(f, des);
            else consumer0(f, new File(des, f.getName()), consumer);
        }));
    }

    public static void consumer(File file, BiConsumer<File, File> consumer) {
        consumer0(file, getDcf(file), consumer);
    }

    public static void copyAndUpdate(File src, File des, Function<String, String> update) {
        copyAndUpdate0(src, checkCopyFile(des), update);
    }

    private static void copyAndUpdate0(File src, File des, Function<String, String> update) {
        consumer0(src, des, (s, d) -> {
            if (d.mkdirs() || d.exists())
                IOHelp.read(IOHelp.getBr(s), IOHelp.getBw(new File(d, s.getName())), update, null);
        });
    }

    public static void copyAndUpdate(File file, Function<String, String> update) {
        copyAndUpdate0(file, getDcf(file), update);
    }

    public static void copyAndUpdate(File src, File des, String regex, String newStr) {
        copyAndUpdate(src, des, l -> RegexHelp.isMatch(l, regex) ? l.replaceAll(regex, newStr) : l);
    }

    public static void copyAndUpdate(File src, String regex, String newStr) {
        copyAndUpdate(src, l -> RegexHelp.isMatch(l, regex) ? l.replaceAll(regex, newStr) : l);
    }

    public static void updateJavaFileImport(File src, File des, String oldStr, String newStr) {
        copyAndUpdate0(src, checkCopyFile(des),
                l -> RegexHelp.isMatch(l, "^(?:package|import) [a-z0-9]+(?:\\.[a-z0-9]+)*(?:.[a-zA-Z0-9]+)+;\\s*")
                        ? l.replaceFirst(oldStr, newStr)
                        : l
        );
    }

    public static void updateJavaFileImport(File src, String oldStr, String newStr) {
        updateJavaFileImport(src, getDcf(src), oldStr, newStr);
    }

    public static void replace(File src, File des, String regex, Function<Matcher, String> func) {
        copyAndUpdate0(src, checkCopyFile(des), l -> RegexHelp.replace(l, regex, func));
    }

    public static void replace(File src, String regex, Function<Matcher, String> func) {
        copyAndUpdate0(src, getDcf(src), l -> RegexHelp.replace(l, regex, func));
    }

    public static void copyAndUpdate(File src, File des, BiConsumer<String, BufferedWriter> update) {
        copyAndUpdate0(src, checkCopyFile(des), update);
    }

    private static void copyAndUpdate0(File src, File des, BiConsumer<String, BufferedWriter> update) {
        consumer0(src, des, (s, d) -> {
            if (d.mkdirs() || d.exists()) IOHelp.read(IOHelp.getBr(s), IOHelp.getBw(new File(d, s.getName())), update);
        });
    }

    public static void copyAndUpdate(File src, BiConsumer<String, BufferedWriter> update) {
        copyAndUpdate0(src, getDcf(src), update);
    }

    public static void formatJavaFile(File src, File des) {
        copyAndUpdate0(src, checkCopyFile(des), (s, w) -> {
            try {
                String str = s.trim();
                if (str.length() == 0) return;
                switch (str.charAt(0)) {
                    case '/':
                    case '*':
                        return;
                    default:
                }
                switch (str.charAt(str.length() - 1)) {
                    case ';':
                    case '{':
                    case '}':
                        break;
                    default:
                        str += " ";
                }
                w.write(str);
            } catch (IOException e) {
                throw PRException.of(e);
            }
        });
    }

    public static void formatJavaFile(File src) {
        formatJavaFile(src, getDcf(src));
    }

    public static final class MatchColumnInfo {
        private final int row;
        private final List<Integer> columns;

        public MatchColumnInfo(int row, List<Integer> columns) {
            this.row = row;
            this.columns = columns;
        }

        public int row() {
            return row;
        }

        public List<Integer> columns() {
            return columns;
        }

        @Override
        public String toString() {
            return "MatchColumnInfo{" +
                    "row=" + row +
                    ", columns=" + columns +
                    '}';
        }
    }

    public static final class MatchFileInfo {
        private final String path;
        private final List<MatchColumnInfo> matchColumnInfo;

        public MatchFileInfo(String path) {
            this.path = path;
            matchColumnInfo = new LinkedList<>();
        }

        public String path() {
            return path;
        }

        public List<MatchColumnInfo> matchColumnInfo() {
            return matchColumnInfo;
        }

        @Override
        public String toString() {
            return "MatchFileInfo{" +
                    "path='" + path + '\'' +
                    ", matchColumnInfo=" + matchColumnInfo +
                    '}';
        }
    }

    public static final class MatchInfo {
        private int totalFiles;
        private int totalRows;
        private int totalColumns;
        private final List<MatchFileInfo> matchFileInfo;

        public MatchInfo() {
            matchFileInfo = new LinkedList<>();
        }

        public int totalFiles() {
            return totalFiles;
        }

        public void totalFiles(int totalFiles) {
            this.totalFiles = totalFiles;
        }

        public int totalRows() {
            return totalRows;
        }

        public void totalRows(int totalRows) {
            this.totalRows = totalRows;
        }

        public int totalColumns() {
            return totalColumns;
        }

        public void totalColumns(int totalColumns) {
            this.totalColumns = totalColumns;
        }

        public List<MatchFileInfo> matchFileInfo() {
            return matchFileInfo;
        }

        @Override
        public String toString() {
            return "MatchInfo{" +
                    "totalFiles=" + totalFiles +
                    ", totalRows=" + totalRows +
                    ", totalColumns=" + totalColumns +
                    ", matchFileInfo=" + matchFileInfo +
                    '}';
        }
    }

    /**
     * 匹配指定文件信息
     */
    public static MatchInfo matchFiles(File file, Function<String, List<Integer>> func) {
        MatchInfo matchInfo = new MatchInfo();
        consumer(file, f -> matchColumns(f, func, matchInfo));
        return matchInfo;
    }

    private static void matchColumns(File file, Function<String, List<Integer>> func, MatchInfo matchInfo) {
        BufferedReader br = IOHelp.getBr(file);
        MatchFileInfo matchFileInfo = new MatchFileInfo(file.getAbsolutePath());
        matchInfo.matchFileInfo.add(matchFileInfo);
        int row = 0;
        String line;
        try {
            while ((line = br.readLine()) != null) {
                ++row;
                List<Integer> columns = func.apply(line);
                if (Validator.nonEmpty(columns)) matchFileInfo.matchColumnInfo.add(new MatchColumnInfo(row, columns));
            }
        } catch (IOException ignored) {
        } finally {
            if (!matchFileInfo.matchColumnInfo.isEmpty()) {
                matchInfo.totalFiles(matchInfo.totalFiles + 1);
                matchInfo.totalRows(matchInfo.totalRows + matchFileInfo.matchColumnInfo.size());
                matchInfo.totalColumns(matchInfo.totalColumns + matchFileInfo.matchColumnInfo.parallelStream()
                        .map(ci -> ci.columns().size()).reduce(Integer::sum).orElse(0));
            } else matchInfo.matchFileInfo.remove(matchInfo.matchFileInfo.size() - 1);
            IOHelp.close(br);
        }
    }

    /**
     * 查找指定文件中包含指定字符串的信息
     */
    public static MatchInfo findByContains(File file, String target) {
        return matchFiles(file, line -> {
            int index = line.indexOf(target);
            if (index == -1) return null;
            return getLineColumnsOfStr(line, target, new LinkedList<>(), 0);
        });
    }

    private static List<Integer> getLineColumnsOfStr(String line, String target, List<Integer> columns, int offset) {
        int index = line.indexOf(target);
        if (index == -1) return columns;
        columns.add(index + 1 + offset);
        offset += index + target.length();
        return getLineColumnsOfStr(line.substring(index + target.length()), target, columns, offset);
    }

    public static String formatPath(String path) {
        if (Validator.isBlank(path)) return path;
        StringBuilder sb = new StringBuilder(path.length());
        append(sb, path);
        return sb.toString();
    }

    private static void append(StringBuilder sb, String path) {
        boolean mark = true;
        for (int i = 0; i < path.length(); ++i) {
            char c = path.charAt(i);
            if (c != '/' && c != '\\') {
                sb.append(c);
                mark = true;
            } else if (mark) {
                sb.append('/');
                mark = false;
            }
        }
    }

    public static String jointPath(String prefix, String path) {
        if (Validator.isBlank(prefix)) return formatPath(path);
        if (Validator.isBlank(path)) return formatPath(prefix);
        StringBuilder sb = new StringBuilder(prefix.length() + path.length() + 1);
        append(sb, prefix);

        char c = path.charAt(0);
        if (sb.charAt(sb.length() - 1) == '/') {
            if (c == '/' || c == '\\') sb.deleteCharAt(sb.length() - 1);
        } else if (c != '/' && c != '\\') sb.append('/');

        append(sb, path);
        return sb.toString();
    }
}