package shz;

import shz.msg.FailureMsg;

import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.X509TrustManager;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;
import java.net.HttpURLConnection;
import java.net.InetSocketAddress;
import java.net.Proxy;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.security.*;
import java.security.cert.Certificate;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.Arrays;
import java.util.Map;
import java.util.function.Supplier;

/**
 * GET(幂等) 一般来说应该只用于数据的读取，而不应当用于会产生副作用的非幂等的操作中
 * HEAD(幂等) 与GET方法一样，都是向服务器发出指定资源的请求。但是，服务器在响应HEAD请求时不会回传资源的内容部分，即：响应主体。
 * 这样，我们可以不传输全部内容的情况下，就可以获取服务器的响应头信息。HEAD方法常被用于客户端查看服务器的性能
 * POST 向指定资源提交数据，请求服务器进行处理，如：表单数据提交、文件上传等，请求数据会被包含在请求体中。
 * 是非幂等的方法，因为这个请求可能会创建新的资源或修改现有资源
 * PUT(幂等) 向指定资源位置上传其最新内容，通过该方法客户端可以将指定资源的最新数据传送给服务器取代指定的资源的内容
 * DELETE(幂等) 用于请求服务器删除所请求URI（统一资源标识符，Uniform Resource Identifier）所标识的资源。DELETE请求后指定资源会被删除
 * CONNECT 是HTTP/1.1协议预留的，能够将连接改为管道方式的代理服务器。通常用于SSL加密服务器的链接与非加密的HTTP代理服务器的通信。
 * OPTIONS(幂等) 与HEAD类似，一般也是用于客户端查看服务器的性能。 这个方法会请求服务器返回该资源所支持的所有HTTP请求方法，该方法会用’*’来代替资源名称，
 * 向服务器发送OPTIONS请求，可以测试服务器功能是否正常。JavaScript的XMLHttpRequest对象进行CORS跨域资源共享时，就是使用OPTIONS方法发送嗅探请求，以判断是否有对指定资源的访问权限
 * TRACE 请求服务器回显其收到的请求信息，该方法主要用于HTTP请求的测试或诊断
 * PATCH(幂等) 方法出现的较晚，它在2010年的RFC 5789标准中被定义。PATCH请求与PUT请求类似，同样用于资源的更新。二者有以下两点不同：
 * 1、PATCH一般用于资源的部分更新，而PUT一般用于资源的整体更新。
 * 2、当资源不存在时，PATCH会创建一个新的资源，而PUT只会对已在资源进行更新
 */
public final class HttpHelp {
    private HttpHelp() {
        throw new IllegalStateException();
    }

    public static String jointUrl(String url, Map<String, ?> params) {
        if (Validator.isEmpty(params)) return url;
        StringBuilder sb = new StringBuilder();
        params.forEach((k, v) -> sb.append("&").append(Coder.urlEncode(k)).append("=").append(Coder.urlEncode(String.valueOf(v))));
        return url.contains("?") ? url + sb.toString() : url + "?" + sb.substring(1);
    }

    //GET, HEAD, POST, PUT, PATCH, DELETE, OPTIONS, TRACE, CONNECT;

    public static InputStream requestIs(String url, Supplier<SSLContext> sslContext, Map<String, String> header, String method, byte[] body, String proxyHost, int proxyPort, int connectTimeout, int readTimeout) {
        try {
            URL uri = new URL(url);
            HttpURLConnection conn;
            if (Validator.nonBlank(proxyHost) && proxyPort > 0) {
                Proxy proxy = new Proxy(Proxy.Type.HTTP, new InetSocketAddress(proxyHost, proxyPort));
                conn = (HttpURLConnection) uri.openConnection(proxy);
            } else conn = (HttpURLConnection) uri.openConnection();
            if (sslContext != null && conn instanceof HttpsURLConnection) {
                SSLContext ssl = sslContext.get();
                if (ssl != null) ((HttpsURLConnection) conn).setSSLSocketFactory(ssl.getSocketFactory());
            }
            if (Validator.nonEmpty(header)) header.forEach(conn::setRequestProperty);
            conn.setDoInput(true);
            conn.setUseCaches(false);
            conn.setRequestMethod(method);
            conn.setConnectTimeout(connectTimeout);
            conn.setReadTimeout(readTimeout);
            if (Validator.nonEmpty(body)) {
                conn.setDoOutput(true);
                conn.getOutputStream().write(body);
            }
            conn.connect();
            int responseCode = conn.getResponseCode();
            if (responseCode != HttpURLConnection.HTTP_OK)
                throw PRException.of(FailureMsg.fail(responseCode, conn.getResponseMessage()));
            return conn.getInputStream();
        } catch (Throwable t) {
            throw PRException.of(t);
        }
    }

    public static InputStream requestIs(String url, Map<String, String> header, String method, byte[] body) {
        return requestIs(url, HttpHelp::getDefaultSSLContext, header, method, body, null, 0, 10000, 10000);
    }

    public static SSLContext getDefaultSSLContext(Certificate... certificates) {
        try {
            SSLContext context = SSLContext.getInstance("TLS");
            context.init(null, Arrays.stream(certificates).map(SSL509TrustManager::new).toArray(SSL509TrustManager[]::new), null);
            return context;
        } catch (NoSuchAlgorithmException | KeyManagementException e) {
            return null;
        }
    }

    private static final class SSL509TrustManager implements X509TrustManager {
        private final Certificate certificate;

        SSL509TrustManager(Certificate certificate) {
            this.certificate = certificate;
        }

        @Override
        public void checkClientTrusted(X509Certificate[] chain, String authType) {
        }

        @Override
        public void checkServerTrusted(X509Certificate[] chain, String authType) throws CertificateException {
            for (X509Certificate cert : chain) {
                cert.checkValidity();
                try {
                    cert.verify(certificate.getPublicKey());
                } catch (NoSuchAlgorithmException | InvalidKeyException | NoSuchProviderException | SignatureException e) {
                    throw PRException.of(e);
                }
            }
        }

        @Override
        public X509Certificate[] getAcceptedIssuers() {
            return new X509Certificate[0];
        }
    }

    public static String request(String url, Supplier<SSLContext> sslContext, Map<String, String> header, String method, byte[] body, String proxyHost, int proxyPort, int connectTimeout, int readTimeout, Charset charset) {
        InputStream is = requestIs(url, sslContext, header, method, body, proxyHost, proxyPort, connectTimeout, readTimeout);
        StringWriter sw = new StringWriter();
        IOHelp.read(IOHelp.getBr(is, charset), sw);
        return sw.toString();
    }

    public static String request(String url, Map<String, String> header, String method, byte[] body) {
        return request(url, HttpHelp::getDefaultSSLContext, header, method, body, null, 0, 10000, 10000, StandardCharsets.UTF_8);
    }

    public static String get(String url, Map<String, String> header) {
        return request(url, header, "GET", null);
    }

    public static String head(String url, Map<String, String> header) {
        return request(url, header, "HEAD", null);
    }

    public static String post(String url, Map<String, String> header, byte[] body) {
        return request(url, header, "POST", body);
    }

    public static String put(String url, Map<String, String> header, byte[] body) {
        return request(url, header, "PUT", body);
    }

    public static String patch(String url, Map<String, String> header, byte[] body) {
        return request(url, header, "PATCH", body);
    }

    public static String delete(String url, Map<String, String> header) {
        return request(url, header, "DELETE", null);
    }

    public static String options(String url, Map<String, String> header) {
        return request(url, header, "OPTIONS", null);
    }

    public static String trace(String url, Map<String, String> header, byte[] body) {
        return request(url, header, "TRACE", body);
    }

    public static String connect(String url, Map<String, String> header) {
        return request(url, header, "CONNECT", null);
    }

    public static String uploadMultipartFile(String url, String name, byte[] bytes, String filename) {
        String boundary = Help.uuid();
        Map<String, String> headers = ToMap.get(4)
                .put("Accept", "*/*")
                .put("Transfer-Encoding", "gzip,deflate,binary")
                .put("Connection", "keep-alive")
                .put("Content-Type", "multipart/form-data; boundary=" + boundary)
                .build();

        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try {
            baos.write(("--" + boundary + "\r\n").getBytes());
            baos.write(("Content-Disposition: form-data; name=\"" + name + "\"; filename=\"" + filename + "\"\r\n").getBytes());
            baos.write("\r\n".getBytes());
            baos.write(bytes);
            baos.write("\r\n".getBytes());
            baos.write(("--" + boundary + "--\r\n").getBytes());
        } catch (IOException e) {
            throw PRException.of(e);
        }

        return post(url, headers, baos.toByteArray());
    }

    public static String uploadMultipartFile(String url, byte[] bytes, String filename) {
        return uploadMultipartFile(url, "file", bytes, filename);
    }
}
