package shz;

import shz.constant.ArrayConstant;

import javax.imageio.ImageIO;
import java.awt.*;
import java.awt.color.ColorSpace;
import java.awt.geom.AffineTransform;
import java.awt.image.AffineTransformOp;
import java.awt.image.BufferedImage;
import java.awt.image.ColorConvertOp;
import java.awt.image.ColorModel;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.Base64;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Supplier;

public final class ImageHelp {
    private ImageHelp() {
        throw new IllegalStateException();
    }

    public static BufferedImage read(InputStream is) {
        try {
            return ImageIO.read(is);
        } catch (IOException e) {
            throw PRException.of(e);
        } finally {
            IOHelp.close(is);
        }
    }

    public static BufferedImage read(File file) {
        try {
            return ImageIO.read(file);
        } catch (IOException e) {
            throw PRException.of(e);
        }
    }

    public static BufferedImage read(byte[] bytes) {
        return read(new ByteArrayInputStream(bytes));
    }

    public static void write(BufferedImage image, File file) {
        try {
            String filename = file.getName();
            ImageIO.write(image, filename.substring(filename.lastIndexOf(".") + 1), IOHelp.getBos(file));
        } catch (IOException e) {
            throw PRException.of(e);
        }
    }

    public static void convert(File folder, Function<BufferedImage, BufferedImage> convert) {
        FileHelp.consumer(folder, (s, d) -> {
            if (d.mkdirs() || d.exists())
                write(convert.apply(read(IOHelp.getBis(s))), new File(d, s.getName()));
        });
    }

    public static BufferedImage convert(BufferedImage image, int colorSpace, int type) {
        if (image.getType() == type) return image;
        return new ColorConvertOp(ColorSpace.getInstance(colorSpace), null).filter(image, new BufferedImage(image.getWidth(), image.getHeight(), type));
    }

    public static BufferedImage toRgb(BufferedImage image) {
        return convert(image, ColorSpace.CS_sRGB, BufferedImage.TYPE_3BYTE_BGR);
    }

    public static BufferedImage toGray(BufferedImage image) {
        return convert(image, ColorSpace.CS_GRAY, BufferedImage.TYPE_BYTE_GRAY);
    }

    public static byte[] toBytes(BufferedImage image) {
        return (byte[]) image.getData().getDataElements(0, 0, image.getWidth(), image.getHeight(), null);
    }

    public static String toStr(BufferedImage image) {
        return Base64.getEncoder().encodeToString(toBytes(image));
    }

    public static byte[] strToBytes(String str) {
        if (Validator.isBlank(str)) return ArrayConstant.EMPTY_BYTE_ARRAY;
        byte[] decode = Base64.getDecoder().decode(str);
        for (int i = 0; i < decode.length; ++i) if (decode[i] < 0) decode[i] += 256;
        return decode;
    }

    public static BufferedImage resize(BufferedImage image, int w, int h) {
        ColorModel cm = image.getColorModel();
        BufferedImage resizeImage = new BufferedImage(cm, image.getRaster().createCompatibleWritableRaster(w, h), cm.isAlphaPremultiplied(), null);
        Graphics g = resizeImage.createGraphics();
        try {
            g.drawImage(image.getScaledInstance(w, h, Image.SCALE_SMOOTH), 0, 0, null);
            return resizeImage;
        } finally {
            if (g != null) g.dispose();
        }
    }

    public static BufferedImage scale(BufferedImage image, double sx, double sy) {
        return new AffineTransformOp(AffineTransform.getScaleInstance(sx, sy), null).filter(image, null);
    }

    public static BufferedImage scale(BufferedImage image, double ratio) {
        return scale(image, ratio, ratio);
    }

    public static BufferedImage scale(BufferedImage image, int w, int h) {
        return scale(image, w * 1.0D / image.getWidth(), h * 1.0D / image.getHeight());
    }

    /**
     * 字符串转图像
     */
    public static BufferedImage fromStr(String str, int w, int h, Color backc, Color bordc, Color fontc,
                                        String fontname, int fontstyle, int fontsize, int strx, int stry,
                                        int gapx, int gapy, int rightlimit) {
        BufferedImage image = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
        Graphics g = image.createGraphics();
        try {
            g.setColor(backc);
            g.fillRect(0, 0, w, h);
            if (bordc != null) {
                g.setColor(bordc);
                g.drawRect(0, 0, w - 1, h - 1);
            }
            g.setColor(fontc);
            g.setFont(new Font(fontname, fontstyle, fontsize));
            if (gapx == 0 || gapy == 0)
                g.drawString(str, bordc != null ? strx + 1 : strx, bordc != null ? stry - 1 : stry);
            else {
                int nx = -1;
                int ny = 0;
                char[] array = str.toCharArray();
                for (char c : array) {
                    if (strx + nx * gapx <= w - rightlimit) ++nx;
                    else {
                        ++ny;
                        nx = 0;
                    }
                    g.drawString(c + "", strx + nx * gapx, stry + ny * gapy);
                }
            }
        } finally {
            if (g != null) g.dispose();
        }
        return image;
    }

    public static BufferedImage fromStr(String str, int w, int h, String fontname, int fontstyle, int fontsize,
                                        int strx, int stry, int gapx, int gapy, int rightlimit) {
        return fromStr(str, w, h, Color.white, Color.white, Color.black, fontname, fontstyle, fontsize, strx, stry, gapx, gapy, rightlimit);
    }

    public static BufferedImage fromStr(String str, int w, int h, String fontname, int fontstyle, int fontsize,
                                        int strx, int stry) {
        return fromStr(str, w, h, Color.white, Color.white, Color.black, fontname, fontstyle, fontsize, strx, stry, 0, 0, 0);
    }

    public static BufferedImage fromStr(String str, int w, int h, String fontname, int fontstyle) {
        return fromStr(str, w, h, fontname, fontstyle, 20, 0, 0, 0, 0, 0);
    }

    /**
     * 多张图像合并一张大图
     */
    public static BufferedImage fromImages(Supplier<BufferedImage> supplier, int w, int h, int count, boolean cut, boolean compress) {
        int[] whg = Help.whAndGcd(w, h, count, 1, cut);
        BufferedImage image = new BufferedImage(whg[0], whg[1], BufferedImage.TYPE_INT_RGB);
        Graphics g = image.createGraphics();
        try {
            for (int y = 0; y < whg[1]; y += whg[2]) {
                for (int x = 0; x < whg[0]; x += whg[2]) {
                    BufferedImage img = supplier.get();
                    if (img == null || count == 0) {
                        y = whg[1];
                        break;
                    }
                    g.drawImage(img.getScaledInstance(whg[2], whg[2], Image.SCALE_SMOOTH), x, y, null);
                    --count;
                }
            }
        } finally {
            if (g != null) g.dispose();
        }
        if (whg[1] != h && compress) return resize(image, w, h);
        return image;
    }

    public static BufferedImage fromImages(Supplier<BufferedImage> supplier, int w, int h, int count) {
        return fromImages(supplier, w, h, count, true, true);
    }

    public static BufferedImage fromImages(File folder, int w, int h, boolean cut, boolean compress) {
        AtomicInteger count = new AtomicInteger();
        FileHelp.consumer(folder, f -> count.incrementAndGet());
        int[] whg = Help.whAndGcd(w, h, count.get(), 1, cut);
        BufferedImage image = new BufferedImage(whg[0], whg[1], BufferedImage.TYPE_INT_RGB);
        Graphics g = image.createGraphics();
        try {
            AtomicInteger x = new AtomicInteger();
            AtomicInteger y = new AtomicInteger();
            FileHelp.consumer(folder, f -> {
                if (y.get() >= whg[1]) return;
                g.drawImage(read(f).getScaledInstance(whg[2], whg[2], Image.SCALE_SMOOTH), x.get(), y.get(), null);
                if (x.addAndGet(whg[2]) >= whg[0]) {
                    x.set(0);
                    y.addAndGet(whg[2]);
                }
            });
        } finally {
            if (g != null) g.dispose();
        }
        if (whg[1] != h && compress) return resize(image, w, h);
        return image;
    }

    public static BufferedImage fromImages(File folder, int w, int h) {
        return fromImages(folder, w, h, true, true);
    }

    /**
     * 将指定文件中的字符串转成图片
     */
    public static BufferedImage fromText(File folder, int w, int h, int fontsize, boolean cut, boolean compress,
                                         boolean trim, Color backc, Color bordc, Color fontc, String fontname, int fontstyle) {
        AtomicInteger count = new AtomicInteger();
        FileHelp.consumer(folder, f -> IOHelp.read(IOHelp.getBr(f), (Consumer<String>) s -> {
            if (trim) for (char c : s.toCharArray())
                if (!Character.isWhitespace(c)) count.incrementAndGet();
                else count.addAndGet(s.length());
        }));
        int[] whg = Help.whAndGcd(w, h, count.get(), fontsize, cut);
        BufferedImage image = new BufferedImage(whg[0], whg[1], BufferedImage.TYPE_INT_RGB);
        Graphics g = image.createGraphics();
        try {
            AtomicInteger x = new AtomicInteger();
            AtomicInteger y = new AtomicInteger();
            FileHelp.consumer(folder, f -> {
                if (y.get() >= whg[1]) return;
                IOHelp.read(IOHelp.getBr(f), (Consumer<String>) s -> {
                    for (char c : s.toCharArray()) {
                        if (y.get() >= whg[1]) break;
                        if (trim && Character.isWhitespace(c)) continue;
                        g.drawImage(fromStr(c + "", whg[2], whg[2], backc, bordc, fontc, fontname, fontstyle,
                                whg[2], 0, whg[2], 0, 0, 0), x.get(), y.get(), null);
                        if (x.addAndGet(whg[2]) >= whg[0]) {
                            x.set(0);
                            y.addAndGet(whg[2]);
                        }
                    }
                });
            });
        } finally {
            if (g != null) g.dispose();
        }
        if (whg[1] != h && compress) return resize(image, w, h);
        return image;
    }

    public static BufferedImage fromText(File folder, int w, int h, int fontsize, String fontname) {
        return fromText(folder, w, h, fontsize, true, true, true, Color.white, Color.white, Color.black, fontname, Font.PLAIN);
    }

    public static BufferedImage fromText(File folder, int w, int h, String fontname) {
        return fromText(folder, w, h, 16, fontname);
    }

    public static BufferedImage fromFile(File file, int w, int h, Color backc, Color bordc, Color fontc, String fontname, int fontstyle, int fontsize, int strx, int stry, int gapy) {
        BufferedImage image = new BufferedImage(w, h, BufferedImage.TYPE_INT_RGB);
        Graphics g = image.createGraphics();
        try {
            g.setColor(backc);
            g.fillRect(0, 0, w, h);
            if (bordc != null) {
                g.setColor(bordc);
                g.drawRect(0, 0, w - 1, h - 1);
            }
            g.setColor(fontc);
            g.setFont(new Font(fontname, fontstyle, fontsize));
            int y = stry;
            List<String> strings = new LinkedList<>();
            FileHelp.readString(file, strings);
            for (String s : ToList.explicitCollect(strings.stream().map(s -> s.replaceAll("\t", "    ")), strings.size())) {
                g.drawString(s, strx, y);
                y += gapy;
            }
        } finally {
            if (g != null) g.dispose();
        }
        return image;
    }

    public static BufferedImage fromFile(File file, int w, int h, String fontname, int fontstyle, int fontsize, int strx, int stry, int gapy) {
        return fromFile(file, w, h, Color.white, Color.white, Color.black, fontname, fontstyle, fontsize, strx, stry, gapy);
    }
}