package shz;

import shz.msg.FailureMsg;
import shz.msg.ServerFailure;
import shz.msg.Success;
import shz.msg.SuccessMsg;

import java.io.Serializable;
import java.util.function.Function;

@SuppressWarnings("unchecked")
public class Response<T> implements Serializable {
    private static final long serialVersionUID = 4985263968380752221L;
    protected int code;
    protected String msg;
    protected T payload;

    public Response(int code, String msg, T payload) {
        this.code = code;
        this.msg = msg;
        this.payload = payload;
    }

    public Response(int code, String msg) {
        this(code, msg, null);
    }

    public Response() {
        this(Success.OK.code(), Success.OK.msg(), null);
    }

    public boolean isOk() {
        return code == Success.OK.code();
    }

    protected static final Response<?> SUCCESS = new Response<>();

    public static <T> Response<T> ok(SuccessMsg message, T payload) {
        if (message != null) return new Response<>(Success.OK.code(), message.msg(), payload);
        return payload == null ? (Response<T>) SUCCESS : new Response<>(Success.OK.code(), Success.OK.msg(), payload);
    }

    public static <T> Response<T> ok(SuccessMsg message) {
        return ok(message, null);
    }

    public static <T> Response<T> ok(T payload) {
        return ok(null, payload);
    }

    public static <T> Response<T> ok() {
        return ok(null, null);
    }

    protected static final Response<?> SERVER_ERROR = new Response<>(ServerFailure.INTERNAL_ERROR.code(), ServerFailure.INTERNAL_ERROR.msg(), null);

    public static <T> Response<T> fail(FailureMsg message, T payload) {
        if (message != null) return new Response<>(message.code(), message.msg(), payload);
        return payload == null ? (Response<T>) SERVER_ERROR : new Response<>(ServerFailure.INTERNAL_ERROR.code(), ServerFailure.INTERNAL_ERROR.msg(), payload);
    }

    public static <T> Response<T> fail(FailureMsg message) {
        return fail(message, null);
    }

    public static <T> Response<T> fail(T payload) {
        return fail(null, payload);
    }

    public static <T> Response<T> fail() {
        return fail(null, null);
    }

    public static <T> Response<T> of(Response<?> response, T payload) {
        return response == null ? fail() : new Response<>(response.code, response.msg, payload);
    }

    public static <T, D> Response<T> of(Response<? extends D> response, Function<D, T> mapper) {
        if (response == null || mapper == null) return fail();
        T payload = mapper.apply(response.payload);
        return new Response<>(response.code, response.msg, payload);
    }

    public static <T> Response<T> of(Response<?> response) {
        return of(response, (Function<Object, T>) Function.identity());
    }

    public static <T> Response<T> of(boolean flag) {
        return flag ? ok() : fail();
    }

    public final int getCode() {
        return code;
    }

    public final String getMsg() {
        return msg;
    }

    public final T getPayload() {
        return payload;
    }

    @Override
    public String toString() {
        return "Response{" +
                "code=" + code +
                ", msg='" + msg + '\'' +
                ", payload=" + payload +
                '}';
    }
}