package shz;

import shz.constant.ArrayConstant;

import java.io.*;
import java.util.Base64;
import java.util.Comparator;
import java.util.Objects;
import java.util.function.ToIntFunction;
import java.util.function.ToLongFunction;

@SuppressWarnings("unchecked")
public final class Serializer {
    private Serializer() {
        throw new IllegalStateException();
    }

    public static byte[] serialize(Object obj) {
        if (obj == null) return ArrayConstant.EMPTY_BYTE_ARRAY;
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try (ObjectOutputStream oos = new ObjectOutputStream(baos)) {
            oos.writeObject(obj);
            oos.flush();
            return baos.toByteArray();
        } catch (IOException e) {
            throw PRException.of(e);
        }
    }

    public static <T> T deserialize(byte[] bytes) {
        if (Validator.isEmpty(bytes)) return null;
        ByteArrayInputStream bais = new ByteArrayInputStream(bytes);
        try (ObjectInputStream ois = new ObjectInputStream(bais)) {
            return (T) ois.readObject();
        } catch (ClassNotFoundException | IOException e) {
            throw PRException.of(e);
        }
    }

    public static String toString(Object obj) {
        return Base64.getEncoder().encodeToString(serialize(obj));
    }

    public static <T> T fromString(String str) {
        return deserialize(Base64.getDecoder().decode(str));
    }

    @FunctionalInterface
    public interface IComparator<T> extends Comparator<T>, Serializable {
        static <T> IComparator<T> comparingInt(IToIntFunction<? super T> keyExtractor) {
            Objects.requireNonNull(keyExtractor);
            return (IComparator<T> & Serializable) (c1, c2) -> Integer.compare(keyExtractor.applyAsInt(c1), keyExtractor.applyAsInt(c2));
        }

        static <T> IComparator<T> comparingLong(IToLongFunction<? super T> keyExtractor) {
            Objects.requireNonNull(keyExtractor);
            return (IComparator<T> & Serializable) (c1, c2) -> Long.compare(keyExtractor.applyAsLong(c1), keyExtractor.applyAsLong(c2));
        }
    }

    public interface IToIntFunction<T> extends ToIntFunction<T>, Serializable {
    }

    public interface IToLongFunction<T> extends ToLongFunction<T>, Serializable {
    }
}