package shz;

import java.sql.Timestamp;
import java.time.*;
import java.time.format.DateTimeFormatter;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.Date;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public final class TimeHelp {
    private TimeHelp() {
        throw new IllegalStateException();
    }

    public static final DateTimeFormatter MONTH_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM");
    public static final DateTimeFormatter DATE_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd");
    public static final DateTimeFormatter TIME_PATTERN = DateTimeFormatter.ofPattern("HH:mm:ss");
    public static final DateTimeFormatter DATE_TIME_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
    public static final DateTimeFormatter DATE_TIME_MILL_PATTERN = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS");
    public static final ZoneId ZONE_ID = ZoneId.of("Asia/Shanghai");
    public static final ZoneOffset ZONE_OFFSET = ZoneOffset.of("+8");
    public static final TimeZone TIME_ZONE = TimeZone.getTimeZone("Asia/Shanghai");

    public static LocalDateTime toLdt(Instant instant) {
        if (instant == null) return null;
        return LocalDateTime.ofInstant(instant, ZONE_ID);
    }

    public static LocalDateTime toLdt(ZonedDateTime zdt) {
        if (zdt == null) return null;
        return zdt.toLocalDateTime();
    }

    public static LocalDateTime toLdt(Date date) {
        if (date == null) return null;
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(date.getTime()), ZONE_ID);
    }

    public static LocalDateTime toLdt(Number timestamp) {
        if (timestamp == null) return null;
        return LocalDateTime.ofInstant(Instant.ofEpochMilli(timestamp.longValue()), ZONE_ID);
    }

    public static LocalDateTime toLdt(CharSequence cs) {
        if (cs == null) return null;
        String s = cs.toString();
        if ((s = formatDateTimeString(s)) == null) return null;
        return LocalDateTime.parse(s, DATE_TIME_PATTERN);
    }

    private static String formatDateTimeString(String s) {
        String yyyy = find(s);
        if (yyyy == null || yyyy.length() < 4) return null;
        boolean flag = yyyy.length() == s.length();
        if (flag) s = s.substring(4);
        StringBuilder sb = new StringBuilder(19).append(flag ? yyyy.substring(0, 4) : yyyy);
        for (int i = 0; i < 5; ++i) {
            boolean appendTime;
            if (flag) {
                int length = s.length();
                String value = null;
                if (length > 2) {
                    value = s.substring(0, 2);
                    s = s.substring(2);
                } else if (length == 2 || length == 1) {
                    value = s;
                    s = "";
                }
                appendTime = appendTime(sb, value, i);
            } else {
                s = s.replaceFirst("\\d+", "");
                appendTime = appendTime(sb, find(s), i);
            }
            if (!appendTime) break;
        }
        return sb.toString();
    }

    private static String find(String s) {
        if (s == null) return null;
        Matcher matcher = Pattern.compile("\\d+", 0).matcher(s);
        return matcher.find() ? matcher.group(0) : null;
    }

    private static boolean appendTime(StringBuilder sb, String value, int nest) {
        if (value == null) {
            switch (nest) {
                case 0:
                    sb.append("-01-01 00:00:00");
                    break;
                case 1:
                    sb.append("-01 00:00:00");
                    break;
                case 2:
                    sb.append(" 00:00:00");
                    break;
                case 3:
                    sb.append(":00:00");
                    break;
                case 4:
                    sb.append(":00");
                    break;
            }
            return false;
        }
        switch (nest) {
            case 0:
            case 1:
                sb.append("-");
                break;
            case 2:
                sb.append(" ");
                break;
            case 3:
            case 4:
                sb.append(":");
                break;
        }
        if (value.length() == 1) sb.append("0");
        sb.append(value);
        return true;
    }

    public static LocalDateTime toLdt(LocalDate ld) {
        if (ld == null) return null;
        return LocalDateTime.of(ld, LocalTime.MIDNIGHT);
    }

    public static LocalDateTime toLdt(Object obj) {
        if (obj == null) return null;
        if (obj instanceof LocalDateTime) return (LocalDateTime) obj;
        if (obj instanceof Instant) return toLdt((Instant) obj);
        if (obj instanceof ZonedDateTime) return toLdt((ZonedDateTime) obj);
        if (obj instanceof Date) return toLdt((Date) obj);
        if (obj instanceof Number) return toLdt((Number) obj);
        if (obj instanceof CharSequence) return toLdt((CharSequence) obj);
        if (obj instanceof LocalDate) return toLdt((LocalDate) obj);
        return null;
    }

    public static LocalTime toLt(CharSequence cs) {
        if (cs == null) return null;
        String s = cs.toString();
        if ((s = formatTimeString(s)) == null) return null;
        if ("00:00:00".equals(s)) return LocalTime.MIDNIGHT;
        return LocalTime.parse(s, TIME_PATTERN);
    }

    private static String formatTimeString(String s) {
        String hh = find(s);
        if (hh == null || hh.length() < 1) return null;
        boolean flag = hh.length() == s.length();
        if (flag) s = s.substring(2);
        StringBuilder sb = new StringBuilder(8);
        if (flag) sb.append(hh, 0, 2);
        else if (hh.length() == 1) sb.append('0').append(hh);
        else sb.append(hh);

        for (int i = 3; i < 5; ++i) {
            boolean appendTime;
            if (flag) {
                int length = s.length();
                String value = null;
                if (length > 2) {
                    value = s.substring(0, 2);
                    s = s.substring(2);
                } else if (length == 2 || length == 1) {
                    value = s;
                    s = "";
                }
                appendTime = appendTime(sb, value, i);
            } else {
                s = s.replaceFirst("\\d+", "");
                appendTime = appendTime(sb, find(s), i);
            }
            if (!appendTime) break;
        }
        return sb.toString();
    }

    public static LocalTime toLt(Object obj) {
        if (obj == null) return null;
        if (obj instanceof LocalTime) return (LocalTime) obj;
        if (obj instanceof CharSequence) return toLt((CharSequence) obj);
        if (obj instanceof LocalDate) return LocalTime.MIDNIGHT;
        LocalDateTime ldt = toLdt(obj);
        if (ldt == null) return null;
        return ldt.toLocalTime();
    }

    public static Instant toInstant(LocalDateTime ldt) {
        if (ldt == null) return null;
        return ldt.toInstant(ZONE_OFFSET);
    }

    public static Instant toInstant(ZonedDateTime zdt) {
        if (zdt == null) return null;
        return zdt.toInstant();
    }

    public static Instant toInstant(Date date) {
        if (date == null) return null;
        return Instant.ofEpochMilli(date.getTime());
    }

    public static Instant toInstant(Number timestamp) {
        if (timestamp == null) return null;
        return Instant.ofEpochMilli(timestamp.longValue());
    }

    public static Instant toInstant(CharSequence cs) {
        return toInstant(toLdt(cs));
    }

    public static Instant toInstant(LocalDate ld) {
        return toInstant(toLdt(ld));
    }

    public static Instant toInstant(Object obj) {
        if (obj instanceof Instant) return (Instant) obj;
        return toInstant(toLdt(obj));
    }

    public static ZonedDateTime toZdt(LocalDateTime ldt) {
        if (ldt == null) return null;
        return ZonedDateTime.ofInstant(ldt.toInstant(ZONE_OFFSET), ZONE_ID);
    }

    public static ZonedDateTime toZdt(Instant instant) {
        if (instant == null) return null;
        return ZonedDateTime.ofInstant(instant, ZONE_ID);
    }

    public static ZonedDateTime toZdt(Date date) {
        if (date == null) return null;
        return ZonedDateTime.ofInstant(Instant.ofEpochMilli(date.getTime()), ZONE_ID);
    }

    public static ZonedDateTime toZdt(Number timestamp) {
        if (timestamp == null) return null;
        return ZonedDateTime.ofInstant(Instant.ofEpochMilli(timestamp.longValue()), ZONE_ID);
    }

    public static ZonedDateTime toZdt(CharSequence cs) {
        return toZdt(toLdt(cs));
    }

    public static ZonedDateTime toZdt(LocalDate ld) {
        return toZdt(toLdt(ld));
    }

    public static ZonedDateTime toZdt(Object obj) {
        if (obj instanceof ZonedDateTime) return (ZonedDateTime) obj;
        return toZdt(toLdt(obj));
    }

    public static Date toDate(LocalDateTime ldt) {
        if (ldt == null) return null;
        return Timestamp.valueOf(ldt);
    }

    public static Date toDate(Instant instant) {
        if (instant == null) return null;
        return Timestamp.from(instant);
    }

    public static Date toDate(ZonedDateTime zdt) {
        if (zdt == null) return null;
        return Timestamp.from(zdt.toInstant());
    }

    public static Date toDate(Number timestamp) {
        if (timestamp == null) return null;
        return new Date(timestamp.longValue());
    }

    public static Date toDate(CharSequence cs) {
        return toDate(toLdt(cs));
    }

    public static Date toDate(LocalDate ld) {
        return toDate(toLdt(ld));
    }

    public static Date toDate(Object obj) {
        if (obj instanceof Date) return (Date) obj;
        return toDate(toLdt(obj));
    }

    public static Long toTimestamp(LocalDateTime ldt) {
        if (ldt == null) return null;
        return ldt.toInstant(ZONE_OFFSET).toEpochMilli();
    }

    public static Long toTimestamp(Instant instant) {
        if (instant == null) return null;
        return instant.toEpochMilli();
    }

    public static Long toTimestamp(ZonedDateTime zdt) {
        if (zdt == null) return null;
        return zdt.toInstant().toEpochMilli();
    }

    public static Long toTimestamp(Date date) {
        if (date == null) return null;
        return date.getTime();
    }

    public static Long toTimestamp(CharSequence cs) {
        return toTimestamp(toLdt(cs));
    }

    public static Long toTimestamp(LocalDate ld) {
        return toTimestamp(toLdt(ld));
    }

    public static Long toTimestamp(Object obj) {
        if (obj instanceof Number) return ((Number) obj).longValue();
        return toTimestamp(toLdt(obj));
    }

    public static String format(LocalDateTime ldt, String pattern) {
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(LocalDateTime ldt) {
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(Instant instant, String pattern) {
        LocalDateTime ldt = toLdt(instant);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(Instant instant) {
        LocalDateTime ldt = toLdt(instant);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(ZonedDateTime zdt, String pattern) {
        LocalDateTime ldt = toLdt(zdt);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(ZonedDateTime zdt) {
        LocalDateTime ldt = toLdt(zdt);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(Date date, String pattern) {
        LocalDateTime ldt = toLdt(date);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(Date date) {
        LocalDateTime ldt = toLdt(date);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(Number timestamp, String pattern) {
        LocalDateTime ldt = toLdt(timestamp);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(Number timestamp) {
        LocalDateTime ldt = toLdt(timestamp);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(LocalDate ld, String pattern) {
        LocalDateTime ldt = toLdt(ld);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(LocalDate ld) {
        LocalDateTime ldt = toLdt(ld);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(Object obj, String pattern) {
        LocalDateTime ldt = toLdt(obj);
        if (ldt == null) return null;
        return ldt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(Object obj) {
        LocalDateTime ldt = toLdt(obj);
        if (ldt == null) return null;
        return ldt.format(DATE_TIME_PATTERN);
    }

    public static String format(LocalTime lt, String pattern) {
        if (lt == null) return null;
        return lt.format(DateTimeFormatter.ofPattern(pattern));
    }

    public static String format(LocalTime lt) {
        if (lt == null) return null;
        return lt.format(TIME_PATTERN);
    }

    public static LocalDateTime exchange(LocalDateTime ldt, TimeZone src, TimeZone des) {
        return LocalDateTime.ofInstant(exchange(ldt.toInstant(ZONE_OFFSET), src, des), ZONE_ID);
    }

    public static LocalDateTime exchange(LocalDateTime ldt) {
        return exchange(ldt, TimeZone.getDefault(), TIME_ZONE);
    }

    public static Instant exchange(Instant instant, TimeZone src, TimeZone des) {
        return Instant.ofEpochMilli(instant.toEpochMilli() + des.getRawOffset() - src.getRawOffset());
    }

    public static Instant exchange(Instant instant) {
        return exchange(instant, TimeZone.getDefault(), TIME_ZONE);
    }

    public static long between(ChronoUnit unit, Temporal t1, Temporal t2) {
        return unit.between(t1, t2);
    }

    public static Duration between(Temporal t1, Temporal t2) {
        return Duration.between(t1, t2);
    }

    public static Period between(LocalDate ld1, LocalDate ld2) {
        return Period.between(ld1, ld2);
    }

    /**
     * 获取指定时间凌晨时间
     */
    public static LocalDateTime morning(LocalDate ld) {
        return LocalDateTime.of(ld, LocalTime.MIDNIGHT);
    }

    /**
     * 获取指定时间凌晨时间
     */
    public static LocalDateTime morning(LocalDateTime ldt) {
        return LocalDateTime.of(ldt.toLocalDate(), LocalTime.MIDNIGHT);
    }

    /**
     * 获取当日凌晨时间
     */
    public static LocalDateTime morning() {
        return LocalDateTime.of(LocalDate.now(), LocalTime.MIDNIGHT);
    }

    /**
     * 获取指定时间凌晨时间less
     */
    public static LocalDateTime morningLess(LocalDate ld) {
        return LocalDateTime.of(ld.minusDays(1), LocalTime.MAX);
    }

    /**
     * 获取指定时间凌晨时间less
     */
    public static LocalDateTime morningLess(LocalDateTime ldt) {
        return LocalDateTime.of(ldt.toLocalDate().minusDays(1), LocalTime.MAX);
    }

    /**
     * 获取当日凌晨时间less
     */
    public static LocalDateTime morningLess() {
        return LocalDateTime.of(LocalDate.now().minusDays(1), LocalTime.MAX);
    }

    /**
     * 获取指定时间凌晨时间more
     */
    public static LocalDateTime morningMore(LocalDate ld) {
        return LocalDateTime.of(ld, LocalTime.ofNanoOfDay(1));
    }

    /**
     * 获取指定时间凌晨时间more
     */
    public static LocalDateTime morningMore(LocalDateTime ldt) {
        return LocalDateTime.of(ldt.toLocalDate(), LocalTime.ofNanoOfDay(1));
    }

    /**
     * 获取当日凌晨时间more
     */
    public static LocalDateTime morningMore() {
        return LocalDateTime.of(LocalDate.now(), LocalTime.ofNanoOfDay(1));
    }

    /**
     * 获取指定时间午夜时间
     */
    public static LocalDateTime midnight(LocalDate ld) {
        return LocalDateTime.of(ld, LocalTime.MAX);
    }

    /**
     * 获取指定时间午夜时间
     */
    public static LocalDateTime midnight(LocalDateTime ldt) {
        return LocalDateTime.of(ldt.toLocalDate(), LocalTime.MAX);
    }

    /**
     * 获取当日午夜时间
     */
    public static LocalDateTime midnight() {
        return LocalDateTime.of(LocalDate.now(), LocalTime.MAX);
    }

    public static boolean isLeap(int year) {
        return year % 400 == 0 || (year % 4 == 0 && year % 100 != 0);
    }

    public static int monthDays(boolean leapYear, int month) {
        switch (month) {
            case 4:
            case 6:
            case 9:
            case 11:
                return 30;
            case 2:
                return leapYear ? 29 : 28;
            default:
                return 31;
        }
    }

    public static LocalDate monday(LocalDate ld) {
        DayOfWeek dayOfWeek = ld.getDayOfWeek();
        int value = dayOfWeek.getValue();
        return value == 1 ? ld : ld.minusDays(value - 1);
    }

    public static LocalDate monday() {
        return monday(LocalDate.now());
    }
}
