package shz;

import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collector;
import java.util.stream.Stream;

@SuppressWarnings("unchecked")
public final class ToList<E> {
    private final List<E> list;

    private ToList(int initialCapacity, int idx) {
        list = new ArrayList<>(reduce(initialCapacity, idx));
    }

    private int reduce(int cap, int idx) {
        if (idx <= 0 || cap < 3) return cap;
        int newCap = cap;
        for (int i = idx; i > 0; --i) newCap = (newCap << 1) / 3;
        while (calc(newCap, idx) < cap) ++newCap;
        return newCap;
    }

    private int calc(int newCap, int idx) {
        while (idx > 0) {
            newCap = newCap + (newCap >> 1);
            --idx;
        }
        return newCap;
    }

    public static <E> ToList<E> get(int initialCapacity, int idx) {
        return new ToList<>(initialCapacity, idx);
    }

    public static <E> ToList<E> get(int initialCapacity) {
        return new ToList<>(initialCapacity, 0);
    }

    public <T> T build() {
        return (T) list;
    }

    public ToList<E> add(Collection<? extends E> c) {
        list.addAll(c);
        return this;
    }

    public final <EE extends E> ToList<E> add(EE e) {
        list.add(e);
        return this;
    }

    public static <E, R extends List<E>> Collector<E, ?, R> collector(Supplier<R> supplier) {
        return Collector.of(
                supplier,
                List::add,
                (left, right) -> {
                    left.addAll(right);
                    return left;
                }
        );
    }

    /**
     * 大致确定元素个数时的收集器
     *
     * @param initialCapacity 大致确定的初始容量
     * @param idx             初始容量衰减次数
     */
    public static <E> Collector<E, ?, List<E>> collector(int initialCapacity, int idx) {
        //提供初始化容量的ArraysList
        return collector(() -> get(initialCapacity, idx).build());
    }

    public static <E> Collector<E, ?, List<E>> collector(int initialCapacity) {
        return collector(initialCapacity, 0);
    }

    /**
     * 大致确定元素个数时的流收集
     */
    public static <E> List<E> explicitCollect(Stream<E> stream, int initialCapacity, int idx) {
        List<E> list = stream.collect(collector(initialCapacity, idx));
        return list.isEmpty() ? Collections.emptyList() : list;
    }

    public static <E> List<E> explicitCollect(Stream<E> stream, int initialCapacity) {
        return explicitCollect(stream, initialCapacity, 0);
    }

    /**
     * 无法确定元素个数时的收集器
     */
    public static <E> Collector<E, ?, List<E>> collector() {
        return collector(LinkedList::new);
    }

    /**
     * 无法确定元素个数时的流收集
     *
     * @param extra 是否需要添加额外元素
     */
    public static <E> List<E> collect(Stream<E> stream, boolean extra) {
        List<E> list = stream.collect(collector());
        return list.isEmpty()
                ? extra ? new LinkedList<>() : Collections.emptyList()
                : list;
    }

    public static <E> List<E> collect(Stream<E> stream) {
        return collect(stream, false);
    }

    /**
     * 无法确定元素个数时的流收集（需要频繁的进行索引查询时选择此方法收集流）
     *
     * @param extra 申请额外空间
     */
    public static <E> List<E> collectArray(Stream<E> stream, int extra) {
        int size = Math.max(extra, 0);
        List<E> list = stream.collect(collector());
        if (list.isEmpty()) return size == 0 ? Collections.emptyList() : new ArrayList<>(size);
        if (size == 0) return new ArrayList<>(list);
        //转ArrayList 提供更快的索引查询
        List<E> result = new ArrayList<>(size + list.size());
        result.addAll(list);
        return result;
    }

    public static <E> List<E> collectArray(Stream<E> stream) {
        return collectArray(stream, 0);
    }
}